(ns telsos.lib.validation)

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

(defmacro invalid
  ([message data]
   (let [file (str *file*)
         line (long (or (-> &form meta :line) -1))]
     `(invalid {:message ~message
                :data    ~data
                :file    ~file
                :line    ~line})))

  ([{:keys [message data cause file line]}]
   (let [file (or file (str *file*))
         line (long (or line (-> &form meta :line) -1))]

     `(telsos.lib.ValidationException.
        (str ~message)
        ~cause
        false
        ~file
        ~line
        ~data))))

(defmacro validating
  [& body]
  `(try (do ~@body)
        (catch telsos.lib.ValidationException e# (throw e#))
        (catch java.lang.Exception e#
          ;; In rare cases catching Exception and wrapping it into ValidationException
          ;; may be hiding problems in the system unrelated to invalid data, so use with
          ;; care and don't broaden the scope of the macro beyond what's necessary.
          (throw (invalid {:message           (Exception/.getMessage e#)
                           :data              {}
                           :cause             e#
                           :with-stack-trace? false})))))

(defn validation-exception?
  [x]
  (instance? telsos.lib.ValidationException x))

(defn ex-validation-data
  [e]
  (when (validation-exception? e)
    (telsos.lib.ValidationException/.getData e)))

(defn ex-file
  [e]
  (when (validation-exception? e)
    (telsos.lib.ValidationException/.getFile e)))

(defn ex-line
  [e]
  (when (validation-exception? e)
    (telsos.lib.ValidationException/.getLine e)))
