(ns telsos.lib.edn-json
  #?(:cljs
     (:require-macros
      [telsos.lib.assertions :refer [the]]))
  (:require
   [malli.core :as m]
   [malli.transform :as mt]
   [telsos.lib.malli :as malli]
   [tick.core :as t]
   #?@(:clj
       [[jsonista.core :as json]
        [telsos.lib.assertions :refer [the]]]

       :cljs
       [[cljs.reader]])))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; JsonEdn Schema
;; ==============
;; Schema for JSON-compatible EDN values.

(def JsonEdn
  [:schema
   {:registry
    {::json-edn
     [:or
      :nil
      :boolean
      :int
      :double
      :string
      [:vector [:ref ::json-edn]]
      [:map-of [:ref ::json-edn] [:ref ::json-edn]]]}}

   [:ref ::json-edn]])

(def the-JsonEdn
  (malli/create-the JsonEdn {:on-invalid-fn malli/on-invalid-throw-ex-info}))

(defn json-edn->string
  [json-edn]
  (-> json-edn
      the-JsonEdn
      #?(:clj  json/write-value-as-string
         :cljs (-> clj->js js/JSON.stringify))))

(defn string->json-edn [s]
  (-> s
      #?(:clj  json/read-value
         :cljs (-> js/JSON.parse (js->clj :keywordize-keys false)))
      the-JsonEdn))

;; JsonEdn Transformation
;; =======================
;; Bidirectional transformation between Clojure EDN and JsonEdn compatible EDN.
;;
;; Handles:
;; - Keyword keys in [:map ...] <-> string keys
;; - :keyword values <-> strings
;; - :uuid <-> string
;; - :inst <-> ISO-8601 string (via tick)
;; - :local-date <-> yyyy-MM-dd string (via tick)
;; - [:enum :kw1 :kw2] <-> strings
;;
;; Does NOT transform keys in:
;; - [:map-of :string ...] (keys stay as strings)
;; - [:map-of :int    ...] (keys stay as ints)

(defn- trans [from-what? to-what? how]
  (fn [x]
    (cond (to-what?   x)      x
          (from-what? x) (how x)

          ;; Possibly incompatible value gets returned with no changes and will
          ;; (presumably) violate the target schema.
          :else x)))

(defn- malli-schema?
  "Returns true if x is a valid malli schema object."
  [x]
  (try
    (m/schema? x)
    (catch #?(:clj Exception :cljs :default) _
      false)))

;; Custom schema registry with tick-based time types
(def ^:private json-edn-registry
  "Registry with custom :inst and :local-date schemas using tick."
  (merge
    (m/default-schemas)
    {:inst
     (m/-simple-schema {:type :inst :pred t/instant?})

     :local-date
     (m/-simple-schema {:type :local-date :pred t/date?})}))

(defn- keyword-enum?
  "Returns true if schema is [:enum :kw1 :kw2 ...] with all keyword values."
  [schema]
  (and (malli-schema? schema)
       (= :enum (m/type schema))
       (every? keyword? (m/children schema))))

(def ^:private json-edn-transformer
  "Creates a transformer for converting between Clojure EDN and JSON-compatible EDN.

   Handles:
   - Keyword keys in [:map ...] schemas <-> string keys
   - :keyword values <-> strings
   - :uuid <-> string (UUID string format)
   - :inst <-> string (ISO-8601 format via tick)
   - :local-date <-> string (yyyy-MM-dd format via tick)
   - [:enum :kw1 :kw2] <-> strings

   Does NOT transform:
   - [:map-of :string ...] keys (they stay as strings)
   - [:map-of <non-keyword> ...] keys (they stay as their type)

   Decoders pass through values already in the target type, convert strings,
   and returns the argument intact for other types."
  (mt/transformer
    ;; Key transformer for [:map ...] keyword keys only
    (mt/key-transformer {:decode keyword :encode name})

    {:name :json-edn
     :decoders
     {:inst       {:compile (fn [_ _] (trans string? t/instant? t/instant))}
      :local-date {:compile (fn [_ _] (trans string? t/date?       t/date))}
      :uuid       {:compile (fn [_ _] (trans string? uuid?     parse-uuid))}
      :keyword    {:compile (fn [_ _] (trans string? keyword?     keyword))}
      :symbol     {:compile (fn [_ _] (trans string? symbol?       symbol))}
      :enum       {:compile (fn [schema _]
                              (when (keyword-enum? schema)
                                (trans string? keyword? keyword)))}}
     :encoders
     {:inst       {:compile (fn [_ _] (trans t/instant? string?  str))}
      :local-date {:compile (fn [_ _] (trans t/date?    string?  str))}
      :uuid       {:compile (fn [_ _] (trans uuid?      string?  str))}
      :keyword    {:compile (fn [_ _] (trans keyword?   string? name))}
      :symbol     {:compile (fn [_ _] (trans symbol?    string?  str))}
      :enum       {:compile (fn [schema _]
                              (when (keyword-enum? schema)
                                (trans keyword? string? name)))}}}))

(defn create-json-edn-encoder
  "Creates an encoder that transforms domain EDN (valid per schema) to JsonEdn-compatible EDN.

   The input is validated against the schema before encoding.
   Throws ex-info on validation failure.

   Uses json-edn-registry which includes :inst and :local-date schemas.

   Example:
     (def encode (create-json-edn-encoder [:map [:id :uuid] [:name :string]]))
     (encode {:id #uuid \"...\" :name \"John\"})
     ;; => {\"id\" \"uuid-string\" \"name\" \"John\"}"
  ([schema]
   (create-json-edn-encoder schema nil))

  ([schema {:keys [on-invalid-fn]
            :or   {on-invalid-fn malli/on-invalid-throw-ex-info}}]
   (the ifn? on-invalid-fn)
   (let [opts        {:registry json-edn-registry}
         transformer json-edn-transformer
         encoder     (m/encoder   schema opts transformer)
         explainer   (m/explainer schema opts)
         validator   (m/validator schema opts)

         the-JsonEdn
         (malli/create-the JsonEdn {:on-invalid-fn on-invalid-fn})]

     (fn [data]
       (when-not (validator data)
         (on-invalid-fn data explainer))

       (the-JsonEdn (encoder data))))))

(defn create-json-edn-decoder
  "Creates a decoder that transforms JsonEdn-compatible EDN to domain EDN (valid per schema).

   The output is validated against the schema after decoding.
   Throws validation exception on failure.

   Uses json-edn-registry which includes :inst and :local-date schemas.

   Decoders accept values already in target type (pass through) or strings (convert).
   Other types will throw an error.

   Example:
     (def decode (create-json-edn-decoder [:map [:id :uuid] [:name :string]]))
     (decode {\"id\" \"550e8400-e29b-41d4-a716-446655440000\" \"name\" \"John\"})
     ;; => {:id #uuid \"550e8400-e29b-41d4-a716-446655440000\" :name \"John\"}"
  ([schema]
   (create-json-edn-decoder schema nil))

  ([schema {:keys [on-invalid-fn]
            :or   {on-invalid-fn malli/on-invalid-throw-ex-info}}]
   (the ifn? on-invalid-fn)
   (let [opts        {:registry json-edn-registry}
         transformer json-edn-transformer
         decoder     (m/decoder   schema opts transformer)
         explainer   (m/explainer schema opts)
         validator   (m/validator schema opts)]
     (fn [data]
       ;; We could validate data to be the-JsonEdn here. We don't do this allowing data
       ;; that is some where in the middle of transformation from JsonEdn to schema.
       (let [data-decoded (decoder data)]
         (if (validator data-decoded)
           data-decoded
           (on-invalid-fn data data-decoded explainer)))))))
