(ns telsos.lib.algorithms.edn
  (:require
   [clojure.walk :refer [postwalk]]
   [telsos.lib.algorithms.maps :refer [submap?]]
   [telsos.lib.algorithms.vecs :refer [vassoc vcount vec-count-found vinsert vnth]]
   [telsos.lib.clojure :refer [named?]]))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; DEEP MERGING EDN
(defn- edn-type
  [x]
  (cond (map?     x) :map
        (vector?  x) :vector
        (set?     x) :set
        (string?  x) :string
        (boolean? x) :boolean
        (number?  x) :number
        (nil?     x) :nil
        :else        :other))

(defn- edn-primitive? [x]
  (or (named? x) (boolean? x) (number? x)))

(defn- parse-unary-edn-control [y y-control]
  (let [args-count (dec (vcount y))]
    (when-not (= 1 args-count)
      (throw (ex-info "Malformed unary-edn-control form" {:mailformed y})))

    (let [y' (vnth y 1)]
      {:y y' :y-type (edn-type y') :y-control y-control})))

(defn- parse-insert-edn-control [y y-control] ;; [:!insert <y> <index> <default>]
  (let [args-count (dec (vcount y))]
    (when-not (#{2 3} args-count)
      (throw (ex-info "Malformed [:!insert ...] form" {:malformed y :args-count args-count})))

    (let [y'      (vnth y 1)
          index   (vnth y 2)
          default (nth  y 3 nil)]

      (when-not (some? y')
        (throw (ex-info "Malformed [:!insert <y> <index> <default>?] with nil y"
                        {:malformed y})))

      (when-not (sequential? y')
        (throw (ex-info "Malformed [:!insert <y> <index> <default>?] with non-sequential y"
                        {:malformed y})))

      (when-not (nat-int? index)
        (throw (ex-info "Malformed [:!insert <y> <index> <default>?] misses nat-int? index"
                        {:malformed y :index index})))

      {:y y' :y-type (edn-type y') :y-control y-control :index index :default default})))

(def ^:private edn-control-parsers
  {:!left   parse-unary-edn-control
   :!right  parse-unary-edn-control
   :!merge  parse-unary-edn-control
   :!conj   parse-unary-edn-control
   :!cons   parse-unary-edn-control
   :!insert parse-insert-edn-control})

(defn- std-edn-info [y _] {:y y :y-type (edn-type y)})

(defn- parse-right-edn-info [y]
  (if (or (not (vector? y)) (= [] y))
    (std-edn-info y nil)

    (let [y-control (vnth y 0)
          parser    (get edn-control-parsers y-control std-edn-info)]

      (parser y y-control))))

(defn- remove-right-edn-controls [y]
  (postwalk (fn [form]
              (if-not (vector? form)
                form

                (let [args-count (dec (vcount form))]
                  (if-not (pos-int? args-count)
                    form

                    (let [y-control (vnth form 0)]
                      (if (edn-control-parsers y-control)
                        (vnth form 1)

                        form))))))
            y))

(defrecord ^:private DeepMergePred [f v])

(defn- parse-deep-merge-sub-pred ;; [:!sub m v']
  [form]
  (when (vector? form)
    (let [n (vcount form)]
      (when-not (zero? n)
        (let [pred (vnth form 0)]
          (when (= :!sub pred)
            (when-not (= n 3)
              (throw (ex-info "Malformed [:!sub m v']" {:form form})))

            (let [m (vnth form 1)
                  _ (when-not (map? m)
                      (throw
                        (ex-info "In [:!sub m v'], m has to be a map" {:form form :m m})))

                  v' (vnth form 2)]

              (->DeepMergePred
                (fn [e] (and (or (map? e) (vector? e)) (submap? e m))) v'))))))))

(defn- parse-deep-merge-kv-pred ;; [:!kv k(s) v(s) v']
  [form]
  (when (vector? form)
    (let [n (vcount form)]
      (when-not (zero? n)
        (let [pred (vnth form 0)]
          (when (= :!kv pred)
            (when-not (= n 4)
              (throw (ex-info "Malformed [:!kv k(s) v(s) v']" {:form form})))

            (let [ks (vnth form 1)
                  ks (cond (sequential?    ks) (vec ks)
                           (edn-primitive? ks) [ks]
                           :else
                           (throw
                             (ex-info "In [:!kv k(s) v(s) v'], k(s) must be a sequential or primitive"
                                      {:form form :ks ks})))
                  vs (vnth form 2)
                  vs (cond (coll?          vs) (set vs)
                           (edn-primitive? vs) #{vs}
                           :else
                           (throw
                             (ex-info "In [:!kv k(s) v(s) v'], v(s) must be a coll or primitive"
                                      {:form form :vs vs})))
                  v' (vnth form 3)]

              (->DeepMergePred
                (fn [e] (and (map? e) (vs (get-in e ks ::n0t-f0und)))) v'))))))))

(defn- parse-deep-merge-pred [form]
  (or (parse-deep-merge-sub-pred form)
      (parse-deep-merge-kv-pred  form)

      ;; Placeholder for other pred forms, if needed in future
      nil))

(declare deep-merge-edns)

(defn- -merge-with
  "Works like clojure.core/merge-with but treats lack of assignment for a given key in a
  map like an assignment of nil value. We need it to achieve consistent behavior of :!left
  and :!right control structures."
  [f & maps]
  (when (some identity maps)
    (let [merge-entry
          (fn [m e] (update m (key e) f (val e)))

          merge2
          (fn [m1 m2] (reduce merge-entry (or m1 {}) (seq m2)))]

      (reduce merge2 maps))))

(defn- deep-merge-edn-maps [vector-mode m1 m2]
  (-merge-with (partial deep-merge-edns vector-mode) m1 m2))

(defn- deep-merge-edn-vectors-with-pred
  [vector-mode v1 {:keys [f v] :as _pred}]
  (mapv (fn [e]
          (if (f e)
            (deep-merge-edns vector-mode e v)
            e))
        v1))

(defn- deep-merge-edn-vectors-with-iter [vector-mode v1 v2]
  (let [n1 (vcount v1)
        n2 (vcount v2)
        n  (max n1 n2)]

    (loop [i 0, result (transient [])]
      (if (= i n)
        (persistent! result)

        (let [e (cond (and (< i n1) (< i n2))
                      (deep-merge-edns vector-mode (vnth v1 i) (vnth v2 i))

                      (< i n1)
                      (vnth v1 i)

                      :else
                      (vnth v2 i))]

          (recur (inc i) (conj! result e)))))))

(defn- analyze-deep-merge-preds [preds]
  [(count preds) (vec-count-found some? preds)])

(defn- deep-merge-edn-vectors
  [vector-mode v1 v2]
  (let [preds
        (mapv parse-deep-merge-pred v2)

        [total-count non-nils-count]
        (analyze-deep-merge-preds preds)]

    (cond (zero? (long non-nils-count))
          (deep-merge-edn-vectors-with-iter vector-mode v1 v2)

          (= (long total-count) (long non-nils-count))
          ;; All elements in v2 are !pred specs
          (reduce (fn [v1' pred]
                    (deep-merge-edn-vectors-with-pred vector-mode v1' pred))
                  #_we-start-with v1
                  preds)
          :else
          ;; There is a !pred in v2 but there are other forms too
          (throw (ex-info "Don't mix preds and other forms when deep-merging vectors"
                          {:v2 v2})))))

(defn deep-merge-edns
  ([x y]
   (deep-merge-edns :conj x y))

  ([vector-mode x y]
   (let [y-original                                      y
         {:keys [y y-type y-control] :as right-edn-info} (parse-right-edn-info y)]

     (cond
       (and (map? x) (or (= :map y-type) (= :nil y-type)))
       (case y-control
         :!left
         x

         :!right
         (remove-right-edn-controls y)

         :!merge
         (deep-merge-edn-maps :merge x (or y {}))

         :!conj
         (deep-merge-edn-maps :conj x (or y {}))

         :!cons
         (deep-merge-edn-maps :cons x (or y {}))

         ;; else
         (deep-merge-edn-maps vector-mode x (or y {})))

       (and (vector? x) (or (= :vector y-type) (= :nil y-type)))
       (cond
         (= :!left y-control)
         x

         (= :!right y-control)
         (remove-right-edn-controls y)

         (= :!merge y-control)
         (deep-merge-edn-vectors :merge x (or y []))

         (= :!conj y-control)
         (into x (remove-right-edn-controls y))

         (= :!cons y-control)
         (into (vec (remove-right-edn-controls y)) x)

         (= :!insert y-control)
         (vinsert x (:index right-edn-info) y (:default right-edn-info))

         (= :merge vector-mode)
         (deep-merge-edn-vectors :merge x (or y []))

         (= :conj vector-mode)
         (into x (remove-right-edn-controls y))

         (= :cons vector-mode)
         (into (vec (remove-right-edn-controls y)) x)

         :else
         (throw (ex-info "Unrecognized vector-mode" {:vector-mode vector-mode})))

       (and (set? x) (or (= :set y-type) (= :nil y-type)))
       (case y-control
         :!left
         x

         :!right
         (remove-right-edn-controls y)

         ;; In clojure         (conj #{1 2 3} #{4}) => #{1 2 3 #{4}} merge likewise,
         ;; but our :!merge and :!conj, :!cons both => #{1 2 3   4}
         (into x (remove-right-edn-controls y)))

       (#{:!insert} y-control)
       (throw (ex-info "[<y-control> ...] used in illegal context"
                {:y-control      y-control
                 :x              x
                 :y              y-original
                 :vector-mode    vector-mode
                 :right-edn-info right-edn-info}))
       :else
       (if (and (= :!left y-control) (some? x))
         x
         (remove-right-edn-controls y))))))

;; READING BACK EDN PATHS AS PRINTED WITH telsos.lib.pprint/print-edn-paths
(defn- edn-path->edn
  [path]
  (cond
    (vector? path)
    (let [n (vcount path)]
      (cond (zero? n) []
            (= 2   n) (vassoc [] (vnth path 0) (edn-path->edn (vnth path 1)) [:!left nil])
            :else     (throw (ex-info "Malformed edn-path (vector)" {:vector path}))))

    (map? path)
    (let [n (count path)]
      (cond (zero? n) {}
            (= 1   n) (let [[k v] (first path)] {k (edn-path->edn v)})
            :else     (throw (ex-info "Malformed edn-path (map)" {:map path}))))

    (set? path)
    (let [n (count path)]
      (cond (zero? n) #{}
            (= 1   n) #{(edn-path->edn (first path))}
            :else     (throw (ex-info "Malformed edn-path (set)" {:set path}))))

    (sequential? path)
    (let [n (count path)]
      (cond (zero? n) []
            (= 2   n) (vassoc [] (nth path 0) (edn-path->edn (nth path 1)) [:!left nil])
            :else     (throw (ex-info "Malformed edn-path (sequential)" {:sequential path}))))

    :else path))

(defn edn-paths->edn
  [paths]
  (when (seq paths)
    (reduce (partial deep-merge-edns :merge)
            (edn-path->edn     (first paths))
            (map edn-path->edn (rest  paths)))))

;; DEEP SELECTION OF EDN STRUCTURES
(def ^:private star? #{* '* :* "*"})
(def ^:private plus? #{+ '+ :+ "+"})

(declare deep-select-edn)

(defn- deep-select-edn-map
  [x selector]
  (let [x (if (sequential? x) (vec x) x)] ;; we always cast sequences to vectors ...
    (when (or (map? x)
              ;; ... and allow (get ...) in vectors too (makes sense for integral keys in selector)
              (vector? x))

      (reduce-kv (fn [result k v]
                   (cond (star? v)
                         (assoc result k (get x k))

                         (plus? v)
                         (if-let [v' (get x k)]
                           (assoc result k v')
                           result)

                         :else
                         (if-let [v' (deep-select-edn (get x k) v)]
                           (assoc result k v')
                           result)))
                 nil
                 selector))))

(defn- apply-deep-select-pred
  [x v pred]
  (loop [i 0, n (vcount x), v v]
    (if (= i n)
      v

      (let [e  (vnth x i)
            e' (when ((:f pred) e) (deep-select-edn e (:v pred)))]
        (recur (inc i) n (if (some? e') (conj! v e') v))))))

(defn- deep-select-edn-vector
  [x selector]
  (let [x (if (sequential? x) (vec x) x)] ;; we always cast sequences to vectors
    (when (vector? x)
      (if (zero? (vcount selector))
        (throw (ex-info "Empty vector selectors are not allowed, use */+ instead" {}))

        (let [preds
              (mapv parse-deep-merge-pred selector)

              [total-count non-nils-count]
              (analyze-deep-merge-preds preds)]

          (if (not= (long total-count) (long non-nils-count))
            (throw (ex-info "Vector selector can only contain preds"
                            {:selector selector}))
            (let [result
                  (->> preds
                       (reduce (partial apply-deep-select-pred x) (transient []))
                       persistent!)]

              (when-not (-> result count zero?)
                result))))))))

(defn deep-select-edn
  [x selector]
  (cond
    (map? selector)
    (deep-select-edn-map x selector)

    (sequential? selector)
    (deep-select-edn-vector x (vec selector))

    (or (star? selector) (plus? selector))
    x

    :else
    (throw (ex-info "Unknow selector" {:selector selector}))))
