(ns telsos.lib.malli
  #?(:cljs
     (:require-macros
      [telsos.lib.assertions :refer [asrt the]]
      [telsos.lib.validation.cljs.macros :refer [invalid]]))
  (:require
   [malli.core :as malli]
   [malli.error :as malli.error]
   [malli.transform :as malli.transform]
   #?@(:clj
       [[telsos.lib.assertions :refer [asrt the]]
        [telsos.lib.validation :refer [invalid]]])))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

(defn on-invalid-throw-invalid
  ([data explainer]
   (throw (invalid "Data validation failed"
                   {:data data

                    :explanation
                    ;; Force before logging. Invalid exceptions are lighweight (in
                    ;; Clojure, not ClojureScript), so in order to avoid excessive
                    ;; explanations, we deliberately delay them.
                    (delay (malli.error/humanize (explainer data)))})))

  ([data data-decoded explainer]
   (throw (invalid "Data validation failed"
                   {:data         data
                    :data-decoded data-decoded

                    :explanation
                    (delay (malli.error/humanize (explainer data)))}))))

(defn on-invalid-throw-ex-info
  ([data explainer]
   (throw (ex-info "Data validation failed"
                   {:data        data
                    :explanation (malli.error/humanize (explainer data))})))

  ([data data-decoded explainer]
   (throw (ex-info "Data validation failed"
                   {:data         data
                    :data-decoded data-decoded
                    :explanation  (malli.error/humanize (explainer data))}))))

(defn create-validating-decoder
  ([schema]
   (create-validating-decoder schema nil))

  ([schema {:keys [transformer on-invalid-fn]
            :or   {transformer   (malli.transform/json-transformer)
                   on-invalid-fn on-invalid-throw-invalid}}]

   (asrt transformer)
   (the ifn? on-invalid-fn)

   (let [decoder   (malli/decoder   schema transformer)
         explainer (malli/explainer schema)
         validator (malli/validator schema)]

     (fn [data]
       (let [data-decoded (decoder data)]
         (if (validator data-decoded)
           data-decoded

           (on-invalid-fn data data-decoded explainer)))))))

(defn create-validating-encoder
  ([schema]
   (create-validating-encoder schema nil))

  ([schema {:keys [transformer on-invalid-fn]
            :or   {transformer   (malli.transform/json-transformer)
                   on-invalid-fn on-invalid-throw-ex-info}}]

   (asrt transformer)
   (the ifn? on-invalid-fn)

   (let [encoder   (malli/encoder schema transformer)
         explainer (malli/explainer schema)
         validator (malli/validator schema)]

     (fn [data]
       (when-not (validator data)
         (on-invalid-fn data explainer))

       (encoder data)))))
