(ns telsos.lib.edn-json
  (:require
   [jsonista.core :as json])
  (:import
   (com.fasterxml.jackson.core JsonGenerator)
   (java.time Instant LocalDate ZoneId ZonedDateTime)
   (java.time.format DateTimeFormatter)
   (java.util UUID)))

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

;; UTC ZONE
;; Preferably, this should be set in Postgres (for interoperability):
;; <psql-prompt>=# show time zone;
;;  TimeZone
;; ----------
;;  Etc/UTC
;; (1 row)

(def ^:private utc-zone (ZoneId/of "UTC"))

;; INSTANT <-> POSTGRESQL TIMESTAMPTZ (STRING) CONVERSION
(def ^:private timestamptz-formatter
  (DateTimeFormatter/ofPattern "yyyy-MM-dd HH:mm:ss.SSSSSSX"))

(defn instant->timestamptz
  ^String [^Instant inst]
  (.format ^DateTimeFormatter timestamptz-formatter (.atZone inst utc-zone)))

(defn timestamptz->instant
  ^Instant [^String timestamptz]
  (-> timestamptz (ZonedDateTime/parse timestamptz-formatter) Instant/from))

(defn ->instant
  ^Instant [x]
  (cond
    (instance? Instant x)
    x

    (string? x)
    ;; When going from strings in json/edn we assume they are timestamptz strings
    ;; (most probably originating from Postgres)
    (timestamptz->instant x)

    :else
    (throw (ex-info "Can't convert to Instant" {:x x :class (class x)}))))

;; DATE <-> POSTGRESQL/JDBC DATE (STRING) CONVERSION
(def ^:private date-formatter
  (DateTimeFormatter/ofPattern "yyyy-MM-dd"))

(defn local-date->string
  ^String [^LocalDate local-date]
  (.format ^DateTimeFormatter date-formatter local-date))

(defn string->local-date
  ^LocalDate [^String date-string]
  (LocalDate/parse date-string date-formatter))

(defn instant->local-date ;; in UTC
  ^LocalDate [^Instant instant]
  (.toLocalDate (.atZone instant utc-zone)))

(defn ->local-date
  ^LocalDate [x]
  (cond
    (instance? LocalDate x)
    x

    (instance? Instant x)
    (instant->local-date x)

    (string? x)
    (string->local-date x)

    :else
    (throw (ex-info "Can't convert to LocalDate" {:x x :x-class (class x)}))))

;; UUIDS
(defn string->uuid
  ^UUID [s] (UUID/fromString s))

(defn ->uuid
  ^UUID [x]
  (cond
    (instance? UUID x)
    x

    (string? x)
    (string->uuid x)

    :else
    (throw (ex-info "Can't convert to UUID" {:x x :x-class (class x)}))))

;; MAPPER
(def ^:private edn->json-mapper
  ;; We don't use any special formatting for instants, local dates, and uuids. They will
  ;; be written as parseable strings and read back into edn as strings. It's the duty of
  ;; the edn user to parse the values on demand.
  (json/object-mapper
    {:encode-key-fn name
     :decode-key-fn keyword

     :encoders
     {java.time.Instant
      (fn [instant ^JsonGenerator gen]
        (.writeString gen (instant->timestamptz instant)))

      java.time.LocalDate
      (fn [local-date ^JsonGenerator gen]
        (.writeString gen (local-date->string local-date)))

      java.util.UUID
      (fn [uuid ^JsonGenerator gen]
        (.writeString gen (str uuid)))

      clojure.lang.Keyword
      (fn [k ^JsonGenerator gen]
        (.writeString gen (name k)))}}))

;; FACADE
(defn edn->json-string
  ^String [x]
  (json/write-value-as-string x edn->json-mapper))

(defn json-string->edn [s]
  (json/read-value s edn->json-mapper))
