(ns telsos.lib.time
  (:require
   [telsos.lib.human-readable]))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

#?(:clj
   (defn stopwatch {:inline (fn [] `(System/nanoTime))}
     ^long []
     (System/nanoTime)))

#?(:clj
   (defn elapsed-nanosecs
     ^long [^long swatch]
     (- (System/nanoTime) swatch)))

#?(:clj
   (defn elapsed-msecs
     ^double [^long swatch]
     (/ (double (elapsed-nanosecs swatch)) 1e6)))

#?(:clj
   (defn elapstr
     ^String [^long swatch]
     (telsos.lib.human-readable/human-readable-nanosecs
       (elapsed-nanosecs swatch) #_digits 2)))

#?(:clj
   (defn catimela
     [body {:keys [msg]}]
     (let [start (stopwatch)
           value (body)
           msecs (elapsed-msecs start)
           msg   (if msg (str msg " elapsed ") "elapsed ")]

       (println (str msg  msecs " msecs"))

       value)))

#?(:cljs
   (defn date->iso-string-with-tz
     "Converts a js/Date to ISO string format with timezone offset (e.g.,
     '2024-07-15T14:30:00.000+02:00')"
     [date]
     (let [year         (.getFullYear     date)
           month        (inc (.getMonth   date)) ; getMonth returns 0-11
           day          (.getDate         date)
           hours        (.getHours        date)
           minutes      (.getMinutes      date)
           seconds      (.getSeconds      date)
           milliseconds (.getMilliseconds date)

           ;; Get timezone offset in minutes (negative for positive timezones)
           tz-offset-minutes (.getTimezoneOffset date)

           ;; Convert to hours and minutes with correct sign
           tz-offset-hours (Math/floor (/ (Math/abs tz-offset-minutes) 60))
           tz-offset-mins  (mod (Math/abs tz-offset-minutes) 60)
           tz-sign         (if (neg? tz-offset-minutes) "+" "-")

           pad-zero
           (fn [n]
             (if (< n 10) (str "0" n) (str n)))

           pad-three
           (fn [n]
             (cond
               (< n 10)  (str "00" n)
               (< n 100) (str "0"  n)
               :else     (str      n)))]

       (str year "-"
            (pad-zero month) "-"
            (pad-zero day) "T"
            (pad-zero hours) ":"
            (pad-zero minutes) ":"
            (pad-zero seconds) "."
            (pad-three milliseconds)
            tz-sign
            (pad-zero tz-offset-hours) ":"
            (pad-zero tz-offset-mins)))))

#?(:cljs
   (defn postgres-timestamp-string->date
     "Parse PostgreSQL timestamp string to js/Date, assuming UTC if no timezone"
     [timestamp-str]
     (cond
       ;; Already has timezone info (Z or +00)
       (or (.includes timestamp-str "Z")
           (.includes timestamp-str "+")
           (.includes timestamp-str "-" 10)) ;; timezone offset position
       (js/Date. timestamp-str)

       ;; No timezone info - assume UTC and add Z
       :else (js/Date. (str (.replace timestamp-str " " "T") "Z")))))
