(ns telsos.lib.algorithms.vecs
  #?(:clj
     (:import
      (clojure.lang IPersistentVector))))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

(defn vcount ^long
  [v]
  (or (when (nil? v) 0)
      #?(:clj  (.count ^IPersistentVector v)
         :cljs (count v))))

(defn vempty?
  [v]
  (zero? (vcount v)))

(defn vnth
  [v ^long index]
  #?(:clj  (.nth ^IPersistentVector v index)
     :cljs (nth v index)))

(defn vfirst
  [v]
  (let [n (vcount v)]
    (when-not (zero? n)
      (vnth v 0))))

(defn vlast
  [v]
  (let [n (vcount v)]
    (when-not (zero? n)
      (vnth v (unchecked-dec n)))))

(defn v=
  ([v]
   (assert (vector? v)) true)

  ([v1 v2]
   #?(:clj  (.equals ^IPersistentVector v1 ^IPersistentVector v2)
      :cljs (= v1 v2)))

  ([v1 v2 & more]
   (if (v= v1 v2)
     (if (next more)
       (recur v2 (first more) (next more))
       (v= v2 (first more)))

     false)))

(defn vec-every?
  [pred v]
  (loop [i 0 n (vcount v)]
    (cond (= i n)                true
          (not (pred (vnth v i))) false

          :else
          (recur (unchecked-inc-int i) n))))

(defn dovec
  [v body]
  (let [n      (vcount        v)
        last-i (unchecked-dec n)]

    (loop [i 0, first? true]
      (when (< i n)
        (let [e     (vnth v i)
              last? (= i last-i)]

          (body e first? last?)
          (recur (unchecked-inc i) false))))))

(defn vassoc
  "Like assoc but extends vector with default values if index is beyond current size.

   Usage:
   (vassoc v index value)           ; extends with nil
   (vassoc v index value default)   ; extends with default value

   Examples:
   (vassoc [] 0 4)      ; => [4]
   (vassoc [] 1 4)      ; => [nil 4]
   (vassoc [] 3 4 :x)   ; => [:x :x :x 4]"

  ([v index value]
   (vassoc v index value nil))

  ([v index value default]
   (let [v            (vec   v)
         current-size (count v)
         needed-size  (inc (long index))]
     (if (>= current-size needed-size)
       (assoc v index value)
       (assoc (into v (repeat (- needed-size current-size) default))
              index
              value)))))
