(ns telsos.lib.algorithms.edn
  (:require
   [telsos.lib.algorithms.maps :refer [submap?]]
   [telsos.lib.algorithms.vecs :refer [vassoc]]
   [telsos.lib.clojure :refer [maybe-name named?]]))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; DEEP MERGING EDN
(defn- edn-type-keyword
  [x]
  (cond (map?     x) :map
        (vector?  x) :vector
        (set?     x) :set
        (string?  x) :string
        (boolean? x) :boolean
        (number?  x) :number
        (nil?     x) :nil
        :else        :other))

(defn- edn-primitive? [x]
  (or (named? x) (boolean? x) (number? x)))

(defn- right-edn-info [y]
  (if (and (vector? y) (= 2 (count y)))
    (let [spec  (maybe-name (nth y 0))
          value (nth y 1)]
      (condp = spec
        "!left"       {:y value :y-type (edn-type-keyword value) :!left? true}
        "!right"      {:y value :y-type (edn-type-keyword value) :!right? true}
        "!merge-vecs" {:y value :y-type (edn-type-keyword value) :!merge-vecs? true}
        "!conj-vecs"  {:y value :y-type (edn-type-keyword value) :!conj-vecs? true}

        ;; otherwise
        {:y y, :y-type (edn-type-keyword y)}))

    {:y y :y-type (edn-type-keyword y)}))

(defrecord ^:private DeepMergePred [f v])

(defn- deep-merge-sub-pred ;; [:!sub m v']
  [form]
  (when (vector? form)
    (let [len (count form)]
      (when-not (zero? len)
        (let [pred (nth form 0)]
          (when (= "!sub" (maybe-name pred))
            (when-not (= len 3)
              (throw (ex-info "Malformed [:!sub m v']" {:form form})))

            (let [m (nth form 1)
                  _ (when-not (map? m)
                      (throw
                        (ex-info "In [:!sub m v'], m has to be a map" {:form form :m m})))

                  v' (nth form 2)]

              (->DeepMergePred
                (fn [e] (and (or (map? e) (vector? e)) (submap? e m))) v'))))))))

(defn- deep-merge-kv-pred ;; [:!kv k(s) v(s) v']
  [form]
  (when (vector? form)
    (let [len (count form)]
      (when-not (zero? len)
        (let [pred (nth form 0)]
          (when (= "!kv" (maybe-name pred))
            (when-not (= len 4)
              (throw (ex-info "Malformed [:!kv k(s) v(s) v']" {:form form})))

            (let [ks (nth form 1)
                  ks (cond (sequential?    ks) (vec ks)
                           (edn-primitive? ks) [ks]
                           :else
                           (throw
                             (ex-info "In [:!kv k(s) v(s) v'], k(s) must be a sequential or primitive"
                                      {:form form :ks ks})))
                  vs (nth form 2)
                  vs (cond (coll?          vs) (set vs)
                           (edn-primitive? vs) #{vs}
                           :else
                           (throw
                             (ex-info "In [:!kv k(s) v(s) v'], v(s) must be a coll or primitive"
                                      {:form form :vs vs})))
                  v' (nth form 3)]

              (->DeepMergePred
                (fn [e] (and (map? e) (vs (get-in e ks ::n0t-f0und)))) v'))))))))

(defn- deep-merge-pred [form]
  (or (deep-merge-sub-pred form)
      (deep-merge-kv-pred  form)

      ;; Placeholder for other pred forms, if needed in future
      nil))

(declare deep-merge-edns)

(defn- deep-merge-edn-maps [merge-vecs? m1 m2]
  (merge-with (partial deep-merge-edns merge-vecs?) m1 m2))

(defn- deep-merge-edn-vectors-with-pred
  [merge-vecs? v1 pred]
  (mapv (fn [e]
          (if ((:f pred) e)
            (deep-merge-edns merge-vecs? e (:v pred))
            e))
        v1))

(defn- deep-merge-edn-vectors-with-iter [merge-vecs? v1 v2]
  (let [n1 (count v1)
        n2 (count v2)]
    (vec (for [i (range (max n1 n2))]
           (cond
             (and (< (long i) n1) (< (long i) n2))
             (deep-merge-edns merge-vecs? (nth v1 i) (nth v2 i))

             (< (long i) n1)
             (nth v1 i)

             :else
             (nth v2 i))))))

(defn- analyze-preds [preds]
  (loop [preds          (seq preds)
         count          0
         non-nils-count 0]
    (if-not preds
      [count non-nils-count]

      (let [e (first preds)]
        (if (nil? e)
          (recur (next preds) (inc count)      non-nils-count)
          (recur (next preds) (inc count) (inc non-nils-count)))))))

(defn- deep-merge-edn-vectors
  [merge-vecs? v1 v2]
  (let [preds              (map deep-merge-pred v2)
        [cnt non-nils-cnt] (analyze-preds preds)]

    (cond (zero? (long non-nils-cnt))
          (deep-merge-edn-vectors-with-iter merge-vecs? v1 v2)

          (= (long cnt) (long non-nils-cnt))
          ;; All elements in v2 are !pred specs
          (reduce (fn [v1' pred]
                    (deep-merge-edn-vectors-with-pred merge-vecs? v1' pred))
                  v1 preds)

          :else
          ;; There is a !pred in v2 but there are other forms too
          (throw (ex-info "Don't mix preds and other forms when deep-merging vectors"
                          {:v2 v2})))))

(defn deep-merge-edns
  ([x y]
   (deep-merge-edns
     false ;; By default we conj-vecs, not merge them
     x y))

  ([merge-vecs? x y]
   (let [{:keys [y y-type !left? !right? !merge-vecs? !conj-vecs?] :as _info}
         (right-edn-info y)]

     (cond
       (and (map? x) (or (= :map y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             !merge-vecs?
             ;; we enter vectors-merging mode for deeper levels [1]
             (deep-merge-edn-maps true x (or y {}))

             !conj-vecs?
             (deep-merge-edn-maps false x (or y {}))

             :else
             (deep-merge-edn-maps merge-vecs? x (or y {})))

       (and (vector? x) (or (= :vector y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             !merge-vecs?
             ;; we enter vectors-merging mode for deeper levels ...
             (deep-merge-edn-vectors true x (or y []))

             ;; ... and we can only quit by forcing with !conj-vecs
             !conj-vecs?
             (into x y)

             merge-vecs?
             (deep-merge-edn-vectors true x (or y []))

             :else (into x y))

       (and (set? x) (or (= :set y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             ;; In clojure  (conj #{1 2 3} #{4}) => #{1 2 3 #{4}} merge likewise,
             ;; but our !merge and !conj both    => #{1 2 3   4}
             :else (into x y))

       :else
       (if (and !left? (some? x))
         x
         y)))))

;; READING BACK EDN PATHS AS PRINTED WITH telsos.lib.pprint/print-edn-paths
(defn- edn-path->edn
  [path]
  (cond
    (vector? path)
    (let [n (count path)]
      (cond (zero? n) []
            (= 2   n) (vassoc [] (nth path 0) (edn-path->edn (nth path 1)) [:!left nil])
            :else     (throw (ex-info "Malformed edn-path (vector)" {:vector path}))))

    (map? path)
    (let [n (count path)]
      (cond (zero? n) {}
            (= 1   n) (let [[k v] (first path)] {k (edn-path->edn v)})
            :else     (throw (ex-info "Malformed edn-path (map)" {:map path}))))

    (set? path)
    (let [n (count path)]
      (cond (zero? n) #{}
            (= 1   n) #{(edn-path->edn (first path))}
            :else     (throw (ex-info "Malformed edn-path (set)" {:set path}))))

    (sequential? path)
    (let [n (count path)]
      (cond (zero? n) []
            (= 2   n) (vassoc [] (nth path 0) (edn-path->edn (nth path 1)) [:!left nil])
            :else     (throw (ex-info "Malformed edn-path (sequential)" {:sequential path}))))

    :else path))

(defn edn-paths->edn
  [paths]
  (when (seq paths)
    (reduce (partial deep-merge-edns
                     #_merge-vecs? true)
            (edn-path->edn     (first paths))
            (map edn-path->edn (rest  paths)))))

;; DEEP SELECTION OF EDN STRUCTURES
;; {:address *}
;; {:address +}

;; * - oznacza coś dowolnego, również nil
;;     w mapach może oznaczać również wartości kluczy, których nie ma w mapie, a o które zapytujemy

;; + - oznacza coś, co nie jest nil
;;     w mapach oznacza istniejącą i nie-nil wartość dla podanego klucza

;; {:address
;;  {:facilities *}}

;; {:address
;;  {:facilities [*]}}

;; {:address
;;  {:facilities [[:!sub {:id 156} *]]}}

(def ^:private star? #{* '* :* "*"})
(def ^:private plus? #{+ '+ :+ "+"})

(declare deep-select-edn)

(defn- deep-select-edn-map
  [x selector]
  (let [x (if (sequential? x) (vec x) x)] ;; we always cast sequences to vectors ...
    (when (or (map? x)
              ;; ... and allow (get ...) in vectors too (makes sense for integral keys in selector)
              (vector? x))

      (reduce-kv (fn [result k v]
                   (cond (star? v)
                         (assoc result k (get x k))

                         (plus? v)
                         (if-let [v' (get x k)]
                           (assoc result k v')
                           result)

                         :else
                         (if-let [v' (deep-select-edn (get x k) v)]
                           (assoc result k v')
                           result)))
                 nil
                 selector))))

(defn- apply-deep-select-pred
  [x v pred]
  (loop [i 0, n (count x), v v]
    (if (= i n)
      v

      (let [e  (nth x i)
            e' (when ((:f pred) e) (deep-select-edn e (:v pred)))]
        (recur (inc i) n (if (some? e') (conj! v e') v))))))

(defn- deep-select-edn-vector
  [x selector]
  (let [x (if (sequential? x) (vec x) x)] ;; we always cast sequences to vectors
    (when (vector? x)
      (if (zero? (count selector))
        (throw (ex-info "Empty vector selectors are not allowed, use */+ instead" {}))

        (let [preds                  (map deep-merge-pred selector)
              [cnt non-nils-cnt] (analyze-preds preds)]

          (if (not= (long cnt) (long non-nils-cnt))
            (throw (ex-info "Vector selector can only contain preds"
                            {:selector selector}))

            (let [result (->> preds
                              (reduce (partial apply-deep-select-pred x) (transient []))
                              persistent!)]

              (when-not (zero? (long (count result)))
                result))))))))

(defn deep-select-edn
  [x selector]
  (cond
    (map? selector)
    (deep-select-edn-map x selector)

    (sequential? selector)
    (deep-select-edn-vector x (vec selector))

    (or (star? selector) (plus? selector))
    x

    :else
    (throw (ex-info "Unknow selector" {:selector selector}))))
