(ns telsos.lib.pprint
  (:require
   [clojure.string :as str]
   [lambdaisland.deep-diff2 :as ddiff]
   [telsos.lib.algorithms.seqs :refer [doseq-enhanced]]
   [telsos.lib.algorithms.trees :refer [->TreeEntry tree->associative-paths]])
  (:import
   (telsos.lib.algorithms.trees TreeEntry)))

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

;; PRETTY-PRINTING DIFFS
(defn pprint-ddiff
  [x y]
  (ddiff/pretty-print (ddiff/diff x y)))

;; PRETTY-PRINTING TREES
(def ^:private PRINT-TREE-INDENT       "│   ")
(def ^:private PRINT-TREE-EMPTYINDENT  "    ")
(def ^:private PRINT-TREE-FORCHILD     "├── ")
(def ^:private PRINT-TREE-FORLASTCHILD "└── ")
(def ^:private PRINT-TREE-EOL          "\n")
(def ^:private PRINT-TREE-EMPTY        "")

(defn- print-tree-genindent
  [[last? & last-child-infos]]
  (let [suffix (if last?
                 PRINT-TREE-FORLASTCHILD
                 PRINT-TREE-FORCHILD)
        prefix (->> last-child-infos
                    (butlast)
                    (reverse)
                    (map #(if % ;; empty?
                            PRINT-TREE-EMPTYINDENT
                            PRINT-TREE-INDENT))
                    (apply str))]

    (str prefix suffix)))

(defn- print-tree-impl
  [node adjs show depth level last-child-infos first?]
  (let [s     (show node)
        pfx   (if first? PRINT-TREE-EMPTY PRINT-TREE-EOL)
        level (long level)
        repr  (if (zero? level)
                (str pfx s)
                (str pfx (print-tree-genindent last-child-infos) s))]

    (print repr)

    (when-not (= level (long depth))
      (let [next-level (inc level)
            children   (adjs node)]

        (doseq-enhanced
          children

          (fn [child _first? last?]
            (print-tree-impl child adjs show depth next-level
              #_last-child-infos (cons last? last-child-infos)

              ;; 1st level only when called on top (in print-tree)
              #_first? false)))))))

(defn print-tree
  "Prints a tree using a textual representation like in UNIX tree command.
   adjs : node -> [node]
   show : node -> String"
  ([node adjs]
   (print-tree node adjs str))

  ([node adjs show]
   (print-tree node adjs show 9223372036854775807))

  ([node adjs show depth]
   (print-tree-impl node adjs show depth 0
     #_last-child-infos '(true)
     #_first? true)))

;; PRETTY-PRINTING EDN (COMMONS)
(defn- edn-adjs
  [e]
  (cond
    (instance? TreeEntry e) [(:node e)]
    (set?                e) (seq e)
    (map?                e) (map (fn [[k v]] (->TreeEntry e k v))                 e)
    (sequential?         e) (map (fn  [i v]  (->TreeEntry e i v)) (iterate inc 0) e)
    :else                   nil))

(defn- edn-struct-repr
  [e]
  (cond (vector?     e) (symbol "[")
        (list?       e) (symbol "(")
        (set?        e) (symbol "#{")
        (sequential? e) (symbol "(")
        (record?     e) (symbol (str (type e)))
        (map?        e) (symbol "{")
        :else           nil))

(def ^:private edn-complementary-struct-repr
  {(symbol "[")  (symbol "]")
   (symbol "(")  (symbol ")")
   (symbol "#{") (symbol "}")
   (symbol "{")  (symbol "}")})

(defn- double-quote-when-string [x]
  (if (string? x) (pr-str x) x))

(defn- edn-show
  [e]
  (or (when (instance? TreeEntry e) (double-quote-when-string (:index-or-key e)))
      (edn-struct-repr e)
      (double-quote-when-string e)))

;; PRETTY-PRINTING EDN (PATHS)
(defn- sort-edn-paths [paths]
  (let [max-len (apply max (map count paths))]
    ;; Because of the way Clojure sorts vectors using their length improperly from our
    ;; needs perspective, we need to extend the vector paths (only when sorting) with
    ;; repreated max integer.
    (sort-by #(into (vec %) (repeat (- (long max-len) (count %)) Long/MAX_VALUE))
             paths)))

(defn- add-edn-path-postfix
  [path]
  (let [path (vec path)]
    (loop [elems path, postfix nil]
      (if-not (seq elems)
        (into path postfix)
        (if-let [s (edn-complementary-struct-repr (first elems))]
          (recur (rest elems) (cons s postfix))
          (recur (rest elems) postfix))))))

(defn print-edn-paths
  [edn]
  (let [paths (tree->associative-paths edn edn-adjs edn-show)
        paths (sort-edn-paths paths)
        paths (map add-edn-path-postfix paths)]

    (doseq [p paths] (println (str/join " " p)))))

;; PRETTY-PRINTING EDN (TREES)
(defn print-edn-tree
  ([e]
   (print-tree e edn-adjs edn-show))

  ([e depth]
   (print-tree e edn-adjs edn-show depth)))
