(ns telsos.lib.fast
  (:import
   (clojure.lang IPersistentMap IPersistentVector)))

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

;; PERSISTENT VECTOR
(defn vcount ^long
  [^IPersistentVector v]
  (or (when (nil? v) 0) (.count v)))

(defn vempty?
  [v]
  (zero? (vcount v)))

(defn vnth
  [^IPersistentVector v ^long index]
  (.nth v index))

(defn vfirst
  [v]
  (let [n (vcount v)]
    (when-not (zero? n)
      (vnth v 0))))

(defn vlast
  [v]
  (let [n (vcount v)]
    (when-not (zero? n)
      (vnth v (unchecked-dec n)))))

(defn v=
  ([v]
   (assert (vector? v)) true)

  ([v1 v2]
   (.equals ^IPersistentVector v1
            ^IPersistentVector v2))

  ([v1 v2 & more]
   (if (v= v1 v2)
     (if (next more)
       (recur v2 (first more) (next more))
       (v= v2 (first more)))

     false)))

(defn vec-every?
  [pred v]
  (loop [i 0 n (vcount v)]
    (cond (= i n)                true
          (not (pred (vnth v i))) false

          :else
          (recur (unchecked-inc-int i) n))))

(defn dovec
  [v body]
  (let [n      (vcount        v)
        last-i (unchecked-dec n)]

    (loop [i 0, first? true]
      (when (< i n)
        (let [e     (vnth v i)
              last? (= i last-i)]

          (body e first? last?)
          (recur (unchecked-inc i) false))))))

;; PERSISTENT MAP
(defn mcount ^long
  [^IPersistentMap m]
  (or (when-not m 0) (.count m)))

(defn mempty?
  [m]
  (zero? (mcount m)))

(defn mmerge-2
  [m1 m2]
  (.cons ^IPersistentMap m1 m2))

(defn mmerge
  ([] nil)
  ([m]  m)
  ([m1 m2]
   (cond (mempty? m1) m2
         (mempty? m2) m1
         :else        (mmerge-2 m1 m2)))

  ([m1 m2 & maps]
   (if-not maps
     (mmerge m1 m2)
     (reduce mmerge (mmerge m1 m2) maps))))

(defn m=
  ([m]
   (assert (map? m)) true)

  ([m1 m2]
   (.equals ^IPersistentMap m1
            ^IPersistentMap m2))

  ([m1 m2 & more]
   (if (m= m1 m2)
     (if (next more)
       (recur m2 (first more) (next more))
       (m= m2 (first more)))

     false)))
