(ns telsos.lib.algorithms.deep-merge
  (:require
   [telsos.lib.algorithms.maps :refer [submap?]]
   [telsos.lib.clojure :refer [false->nil maybe-name named?]]))

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

;; DEEP MERGING EDN
(defn- edn-type-keyword
  [x]
  (cond (map?     x) :map
        (vector?  x) :vector
        (set?     x) :set
        (string?  x) :string
        (boolean? x) :boolean
        (number?  x) :number
        (nil?     x) :nil
        :else        :other))

(defn- edn-primitive? [x]
  (or (named? x) (boolean? x) (number? x)))

(defn- right-edn-info [y]
  (if (and (vector? y) (= 2 (count y)))
    (let [spec  (maybe-name (nth y 0))
          value (nth y 1)]
      (condp = spec
        "!left"       {:y value :y-type (edn-type-keyword value) :!left?       true}
        "!right"      {:y value :y-type (edn-type-keyword value) :!right?      true}
        "!merge-vecs" {:y value :y-type (edn-type-keyword value) :!merge-vecs? true}
        "!conj-vecs"  {:y value :y-type (edn-type-keyword value) :!conj-vecs?  true}

        ;; otherwise
        {:y y, :y-type (edn-type-keyword y)}))

    {:y y :y-type (edn-type-keyword y)}))

(defrecord ^:private DeepMergePred [f v])

(defn- deep-merge-sub-pred ;; [:!sub m v']
  [form]
  (false->nil
    (and (vector? form)
         (= 3 (count form))
         (let [pred (nth form 0)] (= "!sub" (maybe-name pred)))
         (let [m  (nth form 1)
               _  (when-not (map? m)
                    (throw (ex-info "!sub m must be a map" {:form form :m m})))
               v' (nth form 2)]
           (->DeepMergePred (fn [e] (and (or (map? e) (vector? e)) (submap? e m))) v')))))

(defn- deep-merge-kv-pred ;; [:!kv k(s) v(s) v']
  [form]
  (false->nil
    (and (vector? form)
         (= 4 (count form))
         (let [pred (nth form 0)] (= "!kv" (maybe-name pred)))
         (let [ks (nth form 1)
               ks (cond (coll?          ks) (vec ks)
                        (edn-primitive? ks) [ks]
                        :else
                        (throw (ex-info "!kv k(s) must be a coll or primitive"
                                        {:form form :ks ks})))
               vs (nth form 2)
               vs (cond (coll?          vs) (set vs)
                        (edn-primitive? vs) #{vs}
                        :else
                        (throw (ex-info "!kv v(s) must be a coll or primitive"
                                        {:form form :vs vs})))
               v' (nth form 3)]
           (->DeepMergePred (fn [e] (and (map? e) (vs (get-in e ks :n0t-f0und)))) v')))))

(defn- deep-merge-pred [form]
  (or (deep-merge-sub-pred form)
      (deep-merge-kv-pred  form)

      ;; Placeholder for other pred forms, if needed in future
      nil))

(declare deep-merge-edns)

(defn- deep-merge-edn-maps [merge-vecs? m1 m2]
  (merge-with (partial deep-merge-edns merge-vecs?) m1 m2))

(defn- deep-merge-edn-vectors-with-pred
  [merge-vecs? v1 pred]
  (mapv (fn [e]
          (if ((:f pred) e)
            (deep-merge-edns merge-vecs? e (:v pred))
            e))
        v1))

(defn- deep-merge-edn-vectors-with-iter [merge-vecs? v1 v2]
  (let [n1 (count v1)
        n2 (count v2)]
    (vec (for [i (range (max n1 n2))]
           (cond
             (and (< (long i) n1) (< (long i) n2))
             (deep-merge-edns merge-vecs? (nth v1 i) (nth v2 i))

             (< (long i) n1)
             (nth v1 i)

             :else
             (nth v2 i))))))

(defn- deep-merge-edn-vectors
  [merge-vecs? v1 v2]
  (let [preds (map deep-merge-pred v2)]
    (cond
      (every? some? preds)
      ;; All elements in v2 are !pred specs
      (reduce (fn [v1' pred]
                (deep-merge-edn-vectors-with-pred merge-vecs? v1' pred))
              v1 preds)

      (some some? preds)
      ;; There is a !pred in v2 but there are other forms too
      (throw (ex-info "Don't mix preds and other forms when deep-merging vectors"
                      {:v2 v2}))
      :else
      (deep-merge-edn-vectors-with-iter merge-vecs? v1 v2))))

(defn deep-merge-edns
  ([x y]
   ;; By default we conj-vecs
   (deep-merge-edns false x y))

  ([merge-vecs? x y]
   (let [{:keys [y y-type !left? !right? !merge-vecs? !conj-vecs?]} (right-edn-info y)]
     (cond
       (and (map? x) (or (= :map y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             !merge-vecs?
             ;; we enter vectors-merging mode for deeper levels [1]
             (deep-merge-edn-maps true x (or y {}))

             !conj-vecs?
             (deep-merge-edn-maps false x (or y {}))

             :else
             (deep-merge-edn-maps merge-vecs? x (or y {})))

       (and (vector? x) (or (= :vector y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             !merge-vecs?
             ;; we enter vectors-merging mode for deeper levels [1]
             (deep-merge-edn-vectors true x (or y []))

             ;; [1] and we can only quit by forcing !conj-vecs and ending the story
             !conj-vecs?
             (into x y)

             merge-vecs?
             (deep-merge-edn-vectors true x (or y []))

             :else (into x y))

       (and (set? x) (or (= :set y-type) (= :nil y-type)))
       (cond !left?  x
             !right? y

             ;; In clojure  (conj #{1 2 3} #{4}) => #{1 2 3 #{4}} merge likewise,
             ;; but our !merge and !conj both    => #{1 2 3   4}
             :else (into x y))

       :else
       (cond !left?  (if (some? x) x y)
             !right? y
             :else   y)))))
