(ns telsos.lib.pprint
  (:require
   [lambdaisland.deep-diff2 :as ddiff]
   [telsos.lib.algorithms.seqs :as seqs]
   [telsos.lib.algorithms.trees :refer [->TreeEntry]])
  (:import
   (telsos.lib.algorithms.trees TreeEntry)))

(set! *warn-on-reflection*       true)
(set! *unchecked-math* :warn-on-boxed)

;; (PRETTY) PRINTING TREES
(def ^:private PPRINT-TREE-INDENT       "│   ")
(def ^:private PPRINT-TREE-EMPTYINDENT  "    ")
(def ^:private PPRINT-TREE-FORCHILD     "├── ")
(def ^:private PPRINT-TREE-FORLASTCHILD "└── ")
(def ^:private PPRINT-TREE-EOL          "\n")
(def ^:private PPRINT-TREE-EMPTY        "")

(defn- pprint-tree-genindent
  [[last? & last-child-infos]]
  (let [suffix (if last?
                 PPRINT-TREE-FORLASTCHILD
                 PPRINT-TREE-FORCHILD)
        prefix (->> last-child-infos
                    (butlast)
                    (reverse)
                    (map #(if % ;; empty?
                            PPRINT-TREE-EMPTYINDENT
                            PPRINT-TREE-INDENT))
                    (apply str))]

    (str prefix suffix)))

(defn- pprint-tree-impl
  [node adjs show depth level last-child-infos first?]
  (let [s     (show node)
        pfx   (if first? PPRINT-TREE-EMPTY PPRINT-TREE-EOL)
        level (long level)
        repr  (if (zero? level)
                (str pfx s)
                (str pfx (pprint-tree-genindent last-child-infos) s))]

    (print repr)

    (when-not (= level (long depth))
      (let [next-level (inc level)
            children   (adjs node)]
        (doseq [[child last?] (map vector children (seqs/mark-last children))]
          (pprint-tree-impl child adjs show depth next-level
                            #_last-child-infos (cons last? last-child-infos)

                            ;; 1st level only when called on top (in pprint-tree)
                            #_first? false))))))

(defn pprint-tree
  "Prints a tree using a textual representation like in UNIX tree command.
   adjs : node -> [node]
   show : node -> String"
  ([node adjs]
   (pprint-tree node adjs str))

  ([node adjs show]
   (pprint-tree node adjs show 9223372036854775807))

  ([node adjs show depth]
   (pprint-tree-impl node adjs show depth 0
                     #_last-child-infos '(true)
                     #_first? true)))

;; (PRETTY) PRINTING EDN TREES
(defn- edn-adjs
  [e]
  (cond
    (instance? TreeEntry e) [(:node e)]
    (set?                e) (seq e)
    (map?                e) (map (fn [[k v]] (->TreeEntry e k v))                 e)
    (sequential?         e) (map (fn  [i v]  (->TreeEntry e i v)) (iterate inc 0) e)

    :else  nil))

(defn- edn-repr
  [e]
  (cond (vector?     e) "[]"
        (list?       e) "(list)"
        (set?        e) "#{}"
        (sequential? e) "()"
        (record?     e) (str (type e))
        (map?        e) "{}"

        :else nil))

(defn- edn-show
  [e]
  (or (when (instance? TreeEntry e) (str (:index-or-key e)))
      (edn-repr e)
      (str e)))

(defn pprint-edn-tree
  ([e]
   (pprint-tree e edn-adjs edn-show))

  ([e depth]
   (pprint-tree e edn-adjs edn-show depth)))

;; (PRETTY) PRINTING DIFFS
(defn pprint-ddiff
  [x y]
  (ddiff/pretty-print (ddiff/diff x y)))
