(ns clograms.ui.fastfx
  (:require [clojure.reflect :as reflect]
            [clograms.ui.utils :as ui-utils]
            [clojure.string :as str]
            [camel-snake-kebab.core :as csk]
            [clograms.ui.compute-graph :as cg]))

(defonce force-toolkit-init (javafx.embed.swing.JFXPanel.))

(defn construct [klass & args]
  (clojure.lang.Reflector/invokeConstructor klass (into-array Object args)))

;; returns [comp-instance #{[node-id tap prop]}]
(defn build-component [comp-desc cg state]
  (when comp-desc
    (let [val-type (fn [x]
                     (cond
                       (and (map? x) (:fx/type x))             :comp
                       (and (vector? x) (-> x first :fx/type)) :comp-childs
                       (and (vector? x) (= (first x) :<<))     :tap
                       :else                                   :val))
          resolve-val (fn [x]
                        (-> (case (val-type x)
                              :comp        (build-component x cg state)
                              :comp-childs (->> x
                                                (mapv #(build-component % cg state))
                                                (reduce (fn [[cs ts] [nc nts]]
                                                          [(conj cs nc) (into ts nts)])
                                                        [[] #{}]))
                              :tap         (let [tap (into [] (rest x))]
                                             [(cg/compute cg state tap) #{tap}])
                              :val         [x #{}])
                            (into [(val-type x)])))
          class-name (-> comp-desc :fx/type name)
          ctor-args (:fx/ctor-args comp-desc)
          [ctor-vals ctor-taps] (map resolve-val ctor-args)
          [attr->val attrs-taps] (->> (dissoc comp-desc :fx/type :fx/ctor-args)
                                      (reduce (fn [[a->v ats] [k v]]
                                                (let [[v ts] (resolve-val v)]
                                                  [(assoc a->v k v) (into ats ts)]))
                                              [{} #{}]))
          node (apply construct (cond-> [(Class/forName class-name)]
                                  ctor-vals (into  ctor-vals)))

          taps (into ctor-taps attrs-taps)]

      ;; Set all object attributes including childrens
      (doseq [[k v] attr->val]
        (if (:childs? (meta v))

          (.addAll (.getChildren node) (into-array Object v))

          (let [setter-name (str "set" (csk/->PascalCaseString k))]
            (clojure.lang.Reflector/invokeInstanceMethod node setter-name (object-array [v])))))

      (let [node-id (.setId node (str (java.util.UUID/randomUUID)))
            taps-props (map #(into [node-id] %) taps)]
        [node taps-props]))))

(def stage (atom nil))

(defn re-render [root cg state]
  (if @stage
    (ui-utils/run-now
     (.close @stage)
     (.setScene @stage (build-component (:scene root)
                                        cg
                                        state))
     (.show @stage))
    (ui-utils/run-now
     (reset! stage (build-component root
                                    cg
                                    state))
     (.show @stage))))

(comment

  {:comp/id "id"
   ::cg/tap [::cg/root :btn-text]
   :comp/property :text}


  (build-component {:fx/type :javafx.scene.layout.StackPane
                           :children [{:fx/type :javafx.scene.text.Text
                                       :text "Some text"}
                                      {:fx/type :javafx.scene.control.Button
                                       :text "Say Hello"}]})
  (ui-utils/run-now
   (construct (Class/forName (name :javafx.scene.text.Text))))

  (def t (construct (Class/forName (name :javafx.scene.text.Text))))

  )
