(ns clograms.ui.compute-graph
  (:require [loom.io :as loom-io]
            [loom.graph :as loom-graph]
            [loom.label :as loom-label]
            [loom.attr :as loom-attr]
            [clojure.string :as str]
            [clojure.set :as set]))

(defprotocol IComputeNode
  (node-id [_])
  (make-input-signals [_ inputs])
  (compute-node [_ inputs]))


(defprotocol IComputeGraph
  (add-node [_ parents-ids node])
  (remove-node [_ node-id] "Removing a node removes everything that depends on it")
  (compute [_ input-data tap]))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Naive implementation ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrecord NaiveComputeNode [id make-input-signals-fn compute-fn])
(defrecord NaiveComputeGraph [nodes parent->childs child->parents])


(defn make-naive-node [node-id make-input-signals-fn compute-fn]
  (->NaiveComputeNode node-id make-input-signals-fn compute-fn))

(defn make-naive-compute-graph []
  (-> (->NaiveComputeGraph {} {} {})
      (add-node nil (make-naive-node ::root
                                     (fn [_])
                                     (fn [input [k]] (get input k))))))


(defn compute-nodes [g tap]
  (loop [[[cur-nid & cur-params :as cur-tap] & to-visit] #{tap}
         computes nil]
    (if-not cur-nid
      computes
      (let [new-taps (apply (get-in g [:nodes cur-nid :make-input-signals-fn]) (or cur-params [nil]))]
        (recur (into to-visit new-taps)
               (conj computes {:tap cur-tap
                               :inputs (or new-taps [])
                               :compute-fn (get-in g [:nodes (first cur-tap) :compute-fn])}))))))

(defn execute-path [input-data comp-nodes]
  (let [is-root-tap #(= (get-in % [:tap 0]) ::root)
        root-results (->> (filter is-root-tap comp-nodes)
                          (map (fn [{:keys [tap compute-fn]}]
                                 (let [[node-id & params] tap]
                                   [tap (apply compute-fn [input-data (into [] params)])])))
                          (into {}))]
    (loop [results root-results
           [cur-comp & remaining-comp] (remove is-root-tap comp-nodes)]
      (if-not cur-comp
        results
        (recur (let [inputs-vals (map results (:inputs cur-comp))]
                 (assoc results
                        (:tap cur-comp) (apply (:compute-fn cur-comp) [inputs-vals
                                                                       (if-let [p (seq (rest (:tap cur-comp)))]
                                                                         p
                                                                         [nil])])))
         remaining-comp)))))

(defn- add-node-naive [g parents-ids node]
  (let [nid (node-id node)]
    (cond-> g
      true        (update :nodes assoc nid node)
      parents-ids (update :parent->childs (fn [pc]
                                            (reduce (fn [r p]
                                                      (update r p (fnil conj #{}) nid))
                                                    pc
                                                    parents-ids)))
      parents-ids (assoc-in [:child->parents nid] parents-ids))))

(defn- remove-node-naive [g node-id]
  g
  ;; TODO: define behavior
  #_(let []
      (-> g
          ;; remove the node-id from nodes
          (update :nodes dissoc node-id)

          (update :parent->childs dissoc node-id)

          ;; remove node-id from all that have it as a child
          (update :child->parents dissoc node-id)

          ;; remove this node as child of other nodes
          (update :parent->childs (fn [p->c]
                                    (->> (dissoc p->c node-id)
                                         (reduce-kv (fn [r p chs]
                                                      (assoc r p (dissoc chs node-id)))))))

          ;; remove this node
          (update :child->parents (fn [c->p]
                                    (->> (dissoc c->p node-id)
                                         (reduce-kv (fn [r c ps]
                                                      (assoc r c (dissoc ps node-id))))))))))

(extend-type NaiveComputeNode

  IComputeNode

  (node-id [node] (:id node))

  (make-input-signals [node inputs] ((:make-input-signals-fn node) inputs))

  (compute-node [node inputs] ((:compute-fn node) inputs)))

(extend-type NaiveComputeGraph

  IComputeGraph

  (add-node [g parents-ids node] (add-node-naive g parents-ids node))

  (remove-node [g node-id] (remove-node-naive g node-id))

  (compute [g input-data tap]
    (-> (execute-path input-data (compute-nodes g tap))
        (get tap)))

  ;; Quick hacks to get loom draw our graph
  ;; Loom protocols required for loom-io/view
  loom-graph/Graph
  (nodes [g] (vals (:nodes g)))
  (edges [g]
    (reduce-kv (fn [r p childs]
                 (into r (map #(vector (get (:nodes g) p)
                                       (get (:nodes g) %)) childs)))
               []
               (:parent->childs g)))

  ;; this is so the graph drowing shows arrows, not
  ;; need to implement any method, just mark the graph as Digraph
  loom-graph/Digraph

  )

(comment

  (def s {:todos [{:id 1 :text "First todo" :done? false}
                  {:id 2 :text "Second todo" :done? true}
                  {:id 3 :text "Third todo" :done? false}]
          :filter {:text "Sec"}})

  (def cg (-> (make-naive-compute-graph)
              (add-node [::root] (make-naive-node :all-todos
                                                 (fn  [_] [[::root :todos]])
                                                 (fn  [[todos] _] todos)))
              (add-node [:all-todos] (make-naive-node :done-todos
                                                      (fn [_] [[:all-todos]])
                                                      (fn [[all-todos] _]
                                                        (filter :done? all-todos))))
              (add-node [::root :all-todos] (make-naive-node :filtered-todos
                                                            (fn [_] [[::root :filter]
                                                                     [:all-todos]])
                                                            (fn [[filt all-todos] _]
                                                              (filter #(str/includes? % (:text filt)) all-todos))))
              (add-node [:filtered-todos] (make-naive-node :filtered-count
                                                           (fn [_] [[:filtered-todos]])
                                                           (fn [[filtered-todos] _] (count filtered-todos))))
              (add-node [:all-todos] (make-naive-node :todo
                                                      (fn [_] [[:all-todos]])
                                                      (fn [[all-todos] [tid]] (first (filter #(= (:id %) tid) all-todos)))))))

  (loom-io/view cg :node-label :id)

  (compute cg s [:filtered-count])

  )
