;; copyright (c) 2019-2021 sean corfield

(ns ^:no-doc hf.depstar.task
  "Utilities to allow parts of depstar to run as -X tasks."
  (:require [clojure.tools.deps.alpha :as t]))

(set! *warn-on-reflection* true)

(comment
  (t/create-basis {})
  (t/create-basis {:aliases [:test-issue-5]}))


(defn- preprocess-options
  "Given an options hash map, if any of the values are keywords, look them
  up as alias values from the full basis (including user `deps.edn`).

  :jar-type is the only option that is expected to have a keyword value
  and it is generally set automatically so we skip the lookup for that."
  [options]
  (let [kw-opts #{:compile-ns :jar-type} ; :compile-ns can be :all
        edn-fn  (juxt :root-edn :user-edn :project-edn)
        aliases (-> (t/find-edn-maps)
                    (edn-fn)
                    (t/merge-edns)
                    :aliases)]
    (reduce-kv (fn [opts k v]
                 (if (and (not (contains? kw-opts k)) (keyword? v))
                   (if (contains? aliases v)
                     (assoc opts k (get aliases v))
                     (do
                       (println k "has value" v "which is an unknown alias")
                       opts))
                   opts))
               options
               options)))

(defn options-and-basis
  "Given a raw options hash map, apply our defaults and preprocess the
  options (above), then use the options to calculate the project basis,
  if needed, and return the updated options with the :basis added.

  If :basis is provided in the initial options, that is returned as-is
  instead of calculating a new project basis."
  [options]
  (let [{:keys [aliases aot basis group-id jar-type jvm-opts mvn/local-repo paths-only repro]
         :as options}
        (-> (preprocess-options
             (merge {:aliases  []
                     :jar-type :uber
                     :jvm-opts []
                     :pom-file "pom.xml"
                     :repro    true}
                    options))
            ;; these can be symbols or strings:
            (update :jar        #(some-> % str))
            (update :main-class #(some-> % str))
            (update :target-dir #(some-> % str)))]

    (when (and aot (= :thin jar-type))
      (println ":aot is not recommended for a 'thin' JAR!"))
    (when (and group-id (not (re-find #"\." (str group-id))))
      (println ":group-id should probably be a reverse domain name, not just" group-id))
    (when (and jvm-opts (not (sequential? jvm-opts)))
      (println ":jvm-opts should be a vector -- ignoring" jvm-opts))
    (when (and (not= :thin jar-type) paths-only)
      (println ":paths-only is ignored when building an uberjar"))

    ;; so that we can rely on :deps in the basis for the
    ;; sync pom operation, we add in any :extra-deps here:
    (let [{:keys [resolve-args] :as basis}
          (or basis
              (t/create-basis (cond-> {:aliases aliases}
                                local-repo (assoc-in [:extra :mvn/local-repo] local-repo)
                                repro (assoc :user nil))))]
      (assoc options :basis (update basis :deps merge (:extra-deps resolve-args))))))
