;; Copyright (c) George Lipov. All rights reserved.
;; The use and distribution terms for this software are covered by the Eclipse
;; Public License 2.0 (https://choosealicense.com/licenses/epl-2.0/) which can
;; be found in the file LICENSE at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.


;; In my brief experience writing and debugging nREPL middleware the process was
;; obscure and fragile in the extreme, including sporadic and seemingly
;; non-deterministic failures on superficially innocuous code changes like
;; trying to set the correct middleware description. Not sure if related to
;; Shadow CLJS. Thread with caution.


(ns playback.nrepl-middleware
  (:require [clojure.string :as string]
            [clojure.pprint :as pprint]))


(def ^:private ^:dynamic *refresh-fn-syms* [])
(def ^:private ^:dynamic *refreshable-ns-prefixes* [])
;; Call this from your user.clj, e.g.:
;; (middleware/init-refresh-on-eval! 'gnl.clojure-playground.main/mount-root
;;                                   ["gnl.clojure-playground"])

(defn init-refresh-on-eval!
  [refresh-fn-syms refreshable-ns-prefixes]
  (alter-var-root #'*refresh-fn-syms* (constantly refresh-fn-syms))
  (alter-var-root #'*refreshable-ns-prefixes* (constantly refreshable-ns-prefixes)))


(def ^:private ^:dynamic *refresh-on-trace-eval?* false)
(defn set-refresh-on-trace-eval!
  [refresh-on-trace-eval?]
  (alter-var-root #'*refresh-on-trace-eval?* (constantly refresh-on-trace-eval?)))


(def ^:private ^:dynamic *debug?* false)
(defn set-debug!
  [debug?]
  (alter-var-root #'*debug?* (constantly debug?)))


(defn refresh-on-eval [handler]
  (fn [{:keys [op code ns] :as msg}]
    (if (and (= op "eval")
             (some #(string/starts-with? (str ns) %) *refreshable-ns-prefixes*)
             (or (not (string/starts-with? code "#>"))
                 *refresh-on-trace-eval?*))
      (do
        (when *debug?*
          (println (str "\nEval and refresh in " ns ":\n"))
          (pprint/pprint msg))
        (handler (assoc msg :code (str "(let [result (do " code ")]"
                                       "  (doseq [refresh-fn " *refresh-fn-syms* "]"
                                       "    (refresh-fn))"
                                       "  result)"))))
      (handler msg))))
