(ns com.ginger.cljs.views.search
  (:require 
    [com.ginger.cljs.views.common :as common])
  (:use [com.ginger.cljs.search :only [search-request]]
        [com.ginger.cljs.domain :only (to-point domain-keys *domains*)]
        [clojure.data.json :only  (read-json json-str)]
        [noir.core :only [defpage]] 
        [hiccup.page-helpers :only [include-css html5]]
        [hiccup.form-helpers :only 
           [form-to submit-button text-field label select-options]]
        [hiccup.core :only [html]]))

(defpage "/search" []
  (common/layout
    [:header {"class" "jumbotron subhead" "id" "overview"}
     [:h1 "Search" ]
     [:p {"class" "lead"} 
      "Using the following search bar you can query a domain for documents, note that if the search domain isn't configured properly then search will be disabled"]] 
    (form-to {"class" "form-horizontal"} [:post "/query"]
             [:input {"type" "hidden" "value" 0 "name" "start"}]
             [:fieldset
              [:div {"class" "control-group"}
               [:label {"class" "control-label" "for" "search"}  "Search query"] 
               [:div {"class" "controls"} 
                (text-field {"id" "search"} "search") 
                ]] 
              [:div {"class" "control-group"} 
               [:label {"class" "control-label" "for" "domain"}  "Select Domain"]
               [:div {"class" "controls"}
                [:select {"name" "domain" "id" "domain"} 
                 (select-options (map (comp str name) (keys *domains*)))]]]
              [:div {"class" "form-actions"} 
               (submit-button {"class" "btn btn-primary" "type" "submit"} "Run")] 
              ] 
             )))

(defn headers [hs]
  [:thead [:tr (for [h hs] [:th h])]])

(defn rows [rs]
  ^{:test (rows [["x" "1"] ["y" "2"]])}
  (let [trs (reduce 
                   (fn [r v] 
                     (conj r (into  [:tr] (map #(vector :td % ) v)))) [] rs)]
    (when-not (empty? trs)
      (apply conj [:tbody] trs))))


(defn mark-active [links]
  (update-in links [0 1 "class"] (fn [c] "active"))) 

(def res-per-page 25)

(defn link [pos disp]
  [:li {} [:a {"onclick" (str "pageInto(" pos ");")} disp]])

(defn pgn-lnk [at-page delta]
  "Get a total of n links from starting point of start + delta"
  (into [] 
        (mapcat 
          #(vector (link % %)) 
          (range at-page (+ at-page delta)))))

(defn 
  ^{:test #(do 
             (assert (= (link 0 "<<") ((append-back [:ul] 1) 1)))
             (assert (= [:ul] (append-back [:ul] 0))))}
  append-back [target at-page]
  (if-not (= 0 at-page)
    (conj target (link (- at-page 1)  "<<"))
    target 
    ))

(defn 
  ^{:test #(do 
             (assert (= 0 (left-over 50 )))
             (assert (= 1 (left-over 47 ))))}
  left-over [total]
   (if (and (> (int (/ total res-per-page)) 0) (> (rem total res-per-page) 0)) 1 0))


(defn pagination [total at-page]
  "Couple of cases here: first batch, middle batch, end batch"
  (let [links-per-page 5 
        pages (int (/ total res-per-page)) 
        remaining (- (+ pages (left-over total)) at-page)
        displayed (pgn-lnk at-page (min links-per-page (+ remaining 1)))]
    [:div {"class" "pagination"} (into (append-back [:ul] at-page) (mark-active displayed))]))

(defn hits [d-k search start]
  (let [headers (merge {:q search :start start :size res-per-page} (domain-keys d-k))
        res (search-request (to-point d-k) headers)]
    (res :hits)))

(defn table-from [{:keys [domain search start] :as params}]
  (let [d-k (keyword domain)
        s-pos (Integer/valueOf start)
        hs (hits d-k search  (* s-pos res-per-page))
        header (get-in hs [:hit 0 :data])
        rs (map (fn [{:keys [data]}] (->> data  vals (map first))) (-> hs :hit))] 
    {:table 
     [:table {"class" "table table-striped"}
      (headers (->> header keys (map name))) 
      (rows (reduce (fn [r d] (conj r (into [] d))) [] rs))
      ]
     :paginate (pagination (hs :found) s-pos)
     }))

(defpage [:post "/paginate"] {:as query-params}
  (let [{:keys [domain search start]} query-params
        {:keys [table paginate]} (table-from query-params)]
    (html5 [:div {"class" "results"} table paginate])))

(defpage  [:post "/query"]  {:as query-params}
  (let [{:keys [domain search start]} query-params
        {:keys [table paginate]} (table-from query-params)]
    (common/layout
      [:script (str "var params = " (json-str query-params) ";")]
      [:script "var pageInto = function(i){
               params ['start'] = i;
               $.post('/paginate',params, function(data) {
               $('.results').html(data);
               }); 
               }"]
      [:p {"class" "lead"} "Search results:"]
      [:div {"class" "results"}
       table     
       paginate
       ])))

