(ns com.ginger.cljs.domain
  (:use  [clojure.data.json :only  (read-json json-str)]
         [clojure.string :only (capitalize)])
  (:import com.amazonaws.services.cloudsearch.AmazonCloudSearchClient
           com.amazonaws.auth.BasicAWSCredentials
           (com.amazonaws.services.cloudsearch.model IndexDocumentsRequest
             IndexField DefineIndexFieldRequest CreateDomainRequest DeleteDomainRequest
             TextOptions SourceAttribute SourceData UpdateServiceAccessPoliciesRequest
             DescribeServiceAccessPoliciesRequest)))

(def ^:dynamic *domains* (read-json (slurp "domains.json")))

(defn to-point [d-k]
  (str (-> d-k name ) "-" (-> *domains* d-k :point))) 

(defn domain-keys [d-k]
   (dissoc (*domains* d-k) :point))

(defn client  []
  (let [{:keys  [access_id private_key]}  (-> "credentials.json" slurp read-json)]
    (AmazonCloudSearchClient.  (BasicAWSCredentials. access_id private_key))))

(defn create-src-attr [{:keys [src-fn src-name]}]
  {:pre [(#{"Copy" "TrimTitle" "Map"} src-fn)]} 
  (-> (SourceAttribute.)
    (.withSourceDataFunction src-fn) 
    (.withSourceDataCopy 
      (-> (SourceData.) (.withSourceName src-name)))))

(defn create-txt-opts [{:keys [result] :as opts}] 
  {:pre [(every? opts [:result])]} 
  (-> (TextOptions.) (.withResultEnabled result)))

(defn create-field [{:keys [^String type ^String name txt-opts src-opts] :as opts} ]
  {:pre [(every? opts [:type :name :txt-opts :src-opts]) (#{"uint" "literal" "text"} type)]} 
  (-> (IndexField.)
    (.withIndexFieldType type)
    (.withIndexFieldName name)
    (.withTextOptions (create-txt-opts txt-opts))
    (.withSourceAttributes [(create-src-attr src-opts)])))

(defn 
  ^{:test #(assert (dash-to-camel "have-fun") "haveFun") }
  dash-to-camel  [s] 
  (let [[r & rs] (.split s "\\-")]
    (reduce str r (map capitalize rs))))

(defn name-to-method [method]
  (->> method (str ".") dash-to-camel symbol))

(defmacro def-req [name args body]
  (let [new-body (cons (name-to-method name) (list body))]
    `(defn ~name ~args (-> (client) ~new-body))))

(defn allow-all [ip]
  "Creates an IAM rule that allows access to all resources for a given ip address "
  (json-str {:Statement [{:Effect "Allow" :Action "*" :Resource "*" 
                          :Condition {:IpAddress {"aws:SourceIp" [(str ip "/32")]}}}]}))

(def-req update-service-access-policies [domain policy]
  (-> (UpdateServiceAccessPoliciesRequest.) (.withAccessPolicies policy) (.withDomainName domain)))

(def-req index-documents [domain]
  (.withDomainName (IndexDocumentsRequest.) domain))

(def-req delete-domain [domain]
  (.withDomainName (DeleteDomainRequest.) domain))

(def-req define-index-field [domain f-opts]
  (-> (DefineIndexFieldRequest.) (.withDomainName domain) (.withIndexField (create-field f-opts)))) 

(def-req create-domain [name]
  (.withDomainName (CreateDomainRequest.) name))

