(ns com.gfredericks.chess.rules
  "Which moves are legal and such.

  A move is just a [from-square to-square]."
  (:require [com.gfredericks.chess.board :as board]
            [com.gfredericks.chess.pieces :as pieces]
            [com.gfredericks.chess.position] ;; need this for data readers
            [com.gfredericks.chess.squares :as sq]))

(def other-color {:white :black, :black :white})

;; movements

(def rectilinear-movements
  [[0 1] [1 0] [0 -1] [-1 0]])

(def diagonal-movements
  [[1 1] [1 -1] [-1 1] [-1 -1]])

(def all-standard-movements
  (concat rectilinear-movements diagonal-movements))

(def knight-moves
  [[2 1] [1 2] [-2 1] [-1 2] [2 -1] [1 -2] [-2 -1] [-1 -2]])

(defn ^:private move?
  "Returns true if arg is a valid move data structure."
  [x]
  (and (vector? x)
       (number? (first x))
       (number? (second x))
       (sq/square? (first x))
       (sq/square? (second x))
       (or (= 2 (count x))
           (and (= 3 (count x))
                (keyword? (last x))))))

(defn ^:private sqs-in-dir
  "Takes a square and a [dcol drow]"
  [sq [dcol drow]]
  (->> (iterate #(sq/translate % drow dcol) sq)
       (rest)
       (take-while identity)))

(defn ray-moves
  [directions board sq its-color]
  (for [dir directions
        :let [sqs (sqs-in-dir sq dir)
              [blanks more] (split-with #(= :_ (board/get board %)) sqs)
              move-tos (cond-> blanks
                               (if-let [sq (first more)]
                                 (not (pieces/color? its-color (board/get board sq))))
                               (conj (first more)))]
        move-to move-tos]
    [sq move-to]))

(defn king-and-knight-squares
  [dirs sq]
  (->> dirs
       (map (fn [[dcol drow]]
              (sq/translate sq drow dcol)))
       (filter identity)))

(defn king-and-knight-moves
  [dirs board sq color]
  (->> (king-and-knight-squares dirs sq)
       (remove #(pieces/color? color (board/get board %)))
       (map #(vector sq %))))

(def normal-king-moves
  (partial king-and-knight-moves all-standard-movements))
(def normal-queen-moves
  (partial ray-moves all-standard-movements))
(def normal-rook-moves
  (partial ray-moves rectilinear-movements))
(def normal-bishop-moves
  (partial ray-moves diagonal-movements))
(def normal-knight-moves
  (partial king-and-knight-moves knight-moves))

(def pawn-start-row {:white 1, :black 6})
(def pawn-direction {:white 1, :black -1})
(def pawn-penultimate-row {:white 6, :black 1})
(defn normal-pawn-moves
  [board sq color]
  (let [dir (pawn-direction color)
        forward (sq/translate-row sq dir)
        jump (sq/translate-row forward dir)
        attack-left (sq/translate sq dir -1)
        attack-right (sq/translate sq dir 1)
        opponent (other-color color)

        applicable-moves
        (remove nil?
                [(if (= :_ (board/get board forward))
                   [sq forward])
                 (if (and jump
                          (= (sq/row sq) (pawn-start-row color))
                          (= :_ (board/get board forward))
                          (= :_ (board/get board jump)))
                   [sq jump])
                 (if (and attack-left
                          (pieces/color? opponent (board/get board attack-left)))
                   [sq attack-left])
                 (if (and attack-right
                          (pieces/color? opponent (board/get board attack-right)))
                   [sq attack-right])])]
    (if (= (pawn-penultimate-row color) (sq/row sq))
      (for [[from to] applicable-moves
            promote-piece (case color :white [:Q :R :B :N] :black [:q :r :b :n])]
        [from to promote-piece])
      applicable-moves)))

(defn normal-moves-for-piece
  [board piece color sq]
  ((case (pieces/piece-type piece)
     :king normal-king-moves
     :queen normal-queen-moves
     :rook normal-rook-moves
     :bishop normal-bishop-moves
     :knight normal-knight-moves
     :pawn normal-pawn-moves)
   board
   sq
   color))

(defn normal-moves
  [board color-to-move]
  (for [[sq p] (board/piece-placements board)
        :when (pieces/color? color-to-move p)
        mv (normal-moves-for-piece board p color-to-move sq)]
    mv))

(defn attacks?
  "Returns true if the given color is attacking the given square."
  [board attacking-color square]
  (->> (normal-moves board attacking-color)
       (some (fn [[from-square to-square]]
               (and (= to-square square)
                    ;; not a forward pawn move
                    (not (and (pieces/pawn? (board/get board from-square))
                              (= (sq/col from-square) (sq/col to-square)))))))))

(defn castling-moves
  [board turn {:keys [king queen]}]
  (let [castling-row (case turn :white 0 :black 7)
        king-square (sq/square 4 castling-row)
        queen-hop-square (sq/square 3 castling-row)
        king-hop-square (sq/square 5 castling-row)
        attack-free? #(not (attacks? board (other-color turn) %))]
    (if (attack-free? king-square)
      (filter identity
              [(and queen
                    (= :_ (board/get board queen-hop-square))
                    (attack-free? queen-hop-square)
                    [king-square (sq/square 2 castling-row)])
               (and king
                    (= :_ (board/get board king-hop-square))
                    (attack-free? king-hop-square)
                    [king-square (sq/square 6 castling-row)])]))))

(defn en-passant-moves
  [board turn en-passant-square]
  (if en-passant-square
    (let [row (sq/row en-passant-square)
          col (sq/col en-passant-square)
          anti-pawn-dir (case turn :white -1 :black 1)
          left-sq (sq/translate en-passant-square anti-pawn-dir -1)

          [left-type left-color]
          (if left-sq (pieces/piece-info (board/get board left-sq)))

          right-sq (sq/translate en-passant-square anti-pawn-dir 1)

          [right-type right-color]
          (if right-sq (pieces/piece-info (board/get board right-sq)))]
      (filter identity
              [(and left-sq
                    (pieces/pawn? (board/get board left-sq))
                    (= turn (pieces/piece-color (board/get board left-sq)))
                    [left-sq en-passant-square])
               (and right-sq
                    (pieces/pawn? (board/get board right-sq))
                    (= turn (pieces/piece-color (board/get board right-sq)))
                    [right-sq en-passant-square])]))))

(defn progressive-move?
  "Returns true if the move is a capture or a pawn move"
  [{:keys [board]} [from-square to-square]]
  (or (pieces/pawn? (board/get board from-square))
      (not (pieces/blank? (board/get board to-square)))))

(defn make-move-board
  [board [from-square to-square promotion]]
  (-> board
      (board/set from-square :_)
      (board/set to-square (or promotion (board/get board from-square)))))

(defn make-move
  "Woah man is this gonna be a workhorse."
  [{:keys [board turn en-passant castling half-move] :as pos}
   [from-square to-square promotion :as move]]
  (let [board' (make-move-board board move)
        piece-moved (pieces/piece-type (board/get board from-square))
        frow (sq/row from-square)
        fcol (sq/col from-square)
        trow (sq/row to-square)
        tcol (sq/col to-square)]
    (-> pos
        (assoc :board board'
               ;; set en-passant on pawn jump
               :en-passant (if (and (= :pawn piece-moved)
                                    (#{2 -2} (- frow trow)))
                             (sq/square fcol (/ (+ frow trow) 2)))
               :turn (other-color turn)
               :half-move (if (progressive-move? pos move)
                            0
                            (inc half-move)))
        (cond-> (= turn :black)
                (update-in [:full-move] inc)

                ;; move rook on castling moves
                (and (= :king piece-moved)
                     (#{2 -2} (- fcol tcol)))
                (update-in [:board]
                           (fn [board']
                             (let [left? (= 2 tcol)

                                   old-rook-square
                                   (sq/square (if left? 0 7) frow)

                                   new-rook-square
                                   (sq/square (if left? 3 5) frow)]
                               (-> board'
                                   (board/set old-rook-square :_)
                                   (board/set new-rook-square
                                              (board/get board' old-rook-square))))))

                ;; capture on en-passant moves
                (and (= en-passant to-square)
                     (= :pawn piece-moved))
                (update-in [:board]
                           (fn [board']
                             (let [pawn-dir (case turn :white 1 :black -1)
                                   capture-square (sq/square tcol (- trow pawn-dir))]
                               (board/set board' capture-square :_))))

                ;; set castling on king/rook moves
                (= :king piece-moved)
                (assoc-in [:castling turn] {:king false :queen false})

                (= :rook piece-moved)
                (cond->
                 (= from-square (sq/square 0 (case turn :white 0 :black 7)))
                 (assoc-in [:castling turn :queen] false)
                 (= from-square (sq/square 7 (case turn :white 0 :black 7)))
                 (assoc-in [:castling turn :king] false))))))

(defn ^:private king-checker
  [sq]
  (let [sqs (set (king-and-knight-squares all-standard-movements sq))]
    (fn [board sq] (sqs sq))))

(defn ^:private knight-checker
  [sq]
  (let [sqs (set (king-and-knight-squares knight-moves sq))]
    (fn [board sq] (sqs sq))))

(def rowcol (juxt sq/row sq/col))
(defn dirtype
  "Returns [dcol drow] if the relationship between the squares is rectilinear
  or diagonal. Assumes they are not equal."
  [sq1 sq2]
  (let [row1 (sq/row sq1)
        row2 (sq/row sq2)
        col1 (sq/col sq1)
        col2 (sq/col sq2)
        d-row (- row2 row1)
        d-col (- col2 col1)
        abs #(* % (Long/signum %))]
    (if (or (= (abs d-row) (abs d-col))
            (zero? d-row)
            (zero? d-col))
      [(Long/signum d-col) (Long/signum d-row)])))

(defn ^:private check-dir
  [board sq1 sq2 dir]
  (loop [[sq & more] (sqs-in-dir sq1 dir)]
    (cond (= sq sq2) true
          (pieces/blank? (board/get board sq)) (recur more)
          :else false)))

(defn ^:private queen-checker
  [sq]
  (fn [board sq']
    (if-let [dir (dirtype sq sq')]
      (check-dir board sq sq' dir))))

(defn ^:private rook-checker
  [sq]
  (fn [board sq']
    (if-let [[dcol drow :as dir] (dirtype sq sq')]
      (if (or (zero? dcol) (zero? drow))
        (check-dir board sq sq' dir)))))

(defn ^:private bishop-checker
  [sq]
  (fn [board sq']
    (if-let [[dcol drow :as dir] (dirtype sq sq')]
      (if (not (or (zero? dcol) (zero? drow)))
        (check-dir board sq sq' dir)))))

(defn ^:private pawn-checker
  [sq piece-color]
  (let [drow (case piece-color :white 1 :black -1)
        sqs (->> [1 -1]
                 (map (fn [dcol] (sq/translate sq drow dcol)))
                 (filter identity)
                 (set))]
    (fn [board sq'] (sqs sq'))))

(defn ^:private check-checker-piece
  [board sq piece]
  (let [piece-type (pieces/piece-type piece)
        checker (if (= :pawn piece-type)
                  (pawn-checker sq (pieces/piece-color piece))
                  ((case piece-type
                       :king king-checker
                       :queen queen-checker
                       :rook rook-checker
                       :bishop bishop-checker
                       :knight knight-checker)
                   sq))]
    (fn [board' sq']
      ;; make sure the piece is still there; if not it has
      ;; presumably been captured
      (if (= piece (board/get board' sq))
        (checker board' sq')))))

(defn some-fn'
  "Like some-fn but does something sane for multiple-arg functions."
  [& fs]
  (fn [& args]
    (some #(apply % args) fs)))

(defn ^:private check-checker
  "Returns a function
    (fn [board' sq]) => bool
  that expects a board that is similar to the given board, such that
  none of the attacking pieces have moved (though they may have been
  captured), and returns a boolean indicating if the given square is
  under attack."
  [board attacking-player]
  (->> (board/piece-placements board)
       (filter #(pieces/color? attacking-player (second %)))
       (map (fn [[sq piece]] (check-checker-piece board sq piece)))
       (apply some-fn')))

(defn king-square
  "Returns the square on which is the given color's king."
  [board color]
  (->> (board/piece-placements board)
       (filter (fn [[sq p]]
                 (= [:king color] (pieces/piece-info p))))
       (ffirst)))

(defn moves
  "Returns a list of all the legal moves from this position, ignoring
  the positions' half-move attribute."
  [{:keys [board turn castling en-passant] :as pos}]
  ;; the checker is a pre-optimized (for this position) function for
  ;; determining if a move puts the moving player in check.
  (let [checker (check-checker board (other-color turn))
        king-sq (king-square board turn)]
    (->> (concat (normal-moves board turn)
                 (castling-moves board turn (castling turn))
                 (en-passant-moves board turn en-passant))
         (remove (fn [[from to :as move]]
                   (let [board' (make-move-board board move)
                         king-sq' (if (= from king-sq) to king-sq)]
                     (checker board' king-sq')))))))

(defn legal-move?
  [pos move]
  (boolean (some #{move} (moves pos))))

(defn player-to-move-in-check?
  [{:keys [turn board]}]
  (attacks? board (other-color turn) (king-square board turn)))

(defn position-status
  "Returns one of #{:checkmate :stalemate :ongoing}."
  [pos]
  (if (empty? (moves pos))
    (if (player-to-move-in-check? pos)
      :checkmate
      :stalemate)
    :ongoing))
