(ns srs.algorithm
  "All algorithms necessary to score reviews. Based somewhat on SuperMemo2.
  All functions are pure EXCEPT for set-ahead."
  (:require [dambei.db.reviews :as reviews]
            [dambei.srs.config :as config]
            [clj-time.core :as time]
            [clj-time.coerce :as time-coerce]))

(defn easy-factor [ef q]
  "calculates EF' based on EF-initial and quality score"
  (let [res (+ (- ef 0.8) (- (* 0.28 q) (* 0.02 q q)))]
    (if (>= res config/ef-floor) res 1.3)))

(defn- set-ahead
  "sets ahead from the current time, takes a number of MINUTES
  returns an #inst (Date)"
  [amount-minutes]
  (time-coerce/to-date (time/plus (time/now) (time/minutes amount-minutes))))

(defn- update-interval
  "returns a review map with interval and easy factor set. interval is previous interval"
  [review score]
  (let [new-easy-factor (easy-factor (:review/easyfactor review) score)
        new-interval (* new-easy-factor (:review/interval review))]
    (merge review {:review/due (set-ahead new-interval)
                   :review/interval new-interval
                   :review/easyfactor new-easy-factor})))

(defn- graduate
  "graduates a new card to a learned card"
  [review]
  (let [graduation-interval (or (:review/graduation-interval review)
                                config/graduation-interval)]
    (dissoc (merge review {:review/new false
                           :review/interval graduation-interval
                           :review/due (set-ahead graduation-interval)
                           :review/easyfactor config/ef-start})
            :review/newstage)))

(defn- fail
  "Fails a previously learned card"
  [review score]
  (let [failinterval (or (:review/failinterval review)
                          config/failinterval)]
      (merge review {:review/due (set-ahead failinterval)
                     :review/easyfactor (easy-factor (:review/easyfactor review) score)
                     :review/failed true
                     :review/interval failinterval})))

(defn- fail-then-pass
  [review score]
  (let [failthenpassinterval (or (:review/failthenpassinterval review)
                                 config/failthenpassinterval)]
    (merge review {:review/due (set-ahead failthenpassinterval)
                   :review/failed false
                   :review/easyfactor (easy-factor (:review/easyfactor review) score)
                   :review/interval failthenpassinterval})))

(defmulti score-new :review/newstage)
(defmethod score-new 0 [review score]
  (let [newstageinterval0 (or (:review/newstageinterval0 review)
                                 config/newstageinterval0)
        newstageinterval1 (or (:review/newstageinterval1 review)
                                 config/newstageinterval1)]
    (if (> score config/wrong-response)
      (merge review {:review/newstage 1
                     :review/interval newstageinterval1
                     :review/due (set-ahead newstageinterval1)})
      (merge review {:review/due (set-ahead newstageinterval0)}))))
(defmethod score-new 1 [review score]
  (let [newstageinterval0 (or (:review/newstageinterval0 review)
                                 config/newstageinterval0)
        newstageinterval1 (or (:review/newstageinterval1 review)
                                 config/newstageinterval1)]
    (if (> score config/wrong-response)
      (graduate review)
      (merge review {:review/newstage 0
                     :review/interval newstageinterval0
                     :review/due (set-ahead newstageinterval0)}))))

(defn- score-learned
  [review score]
  (if (> score config/wrong-response)
    (if (:review/failed review)
      (fail-then-pass review score)
      (update-interval review score))
    (fail review score)))

(defn score
  "Returns the a new review that is scored according to logic in the review card
  score should be 0-5.
  review should be a hash-map that has all review-fields"
  [review score]
  (cond (:review/new review)
        (score-new review score)
        (not (:review/new review))
        (score-learned review score)))
