(ns rules.core
  (:require [datomic.api :refer [q]]
            [clojure.core.match :refer [match]]))

(defn rule-match?
  "rule is a vector of vectors. First vector must have a ?variable at the end
  user-data is user stats from the db
  unifier is a symbol like ?id

  Keyword->Rule->[UserStat]->Boolean"
  [unifier rule user-data]
  (when (q (concat [:find [unifier]
                    :where]
                   rule)
           user-data)
    true))

(defn rules-result
  "CampaignID->[Rule]->[UserStat]->CampaignID
  Returns original campaign id if nothing matches
  [Rule] should be the list found under keys like :history or :network in rules.edn

  NOTE: UserStat MUST be in a vector of vectors form"
  [campaign-id rules user-data]
  (if-let [dest-id (first (filter identity (for [{:keys [unifier clauses result]} rules]
                                             (if (rule-match? unifier clauses user-data)
                                               result nil))))]
    dest-id
    campaign-id))

;;RULE VALIDATION
(defn valid-clauses?
  [clauses]
  (every? identity
          (for [c clauses]
            (match [c]
                   [[_ (_ :guard keyword?) _ (_ :guard symbol?)]] true
                   [[_]] true
                   :else     nil))))

(defn valid-rule?
  [rule]
  (match [rule]
         [{:name (_ :guard string?) :result _ :unifier (_ :guard symbol?)
           :clauses (cs :guard vector?)}] (valid-clauses? cs)
           :else nil))

(defn valid-rules-campaign?
  "must have :history and/or :network as keys"
  [rc]
  (match [rc]
         [{:history rules}]
         (every? identity (map valid-rule? rules))
         [{:network rules}]
         (every? identity (map valid-rule? rules))
         :else nil))

(defn valid-rules-map?
  [rules-map]
  (let [checked-rules (for [[camp-id rules-campaign] rules-map]
                        [camp-id
                         (and
                          (or (string? camp-id) (integer? camp-id) (= :default camp-id))
                          (valid-rules-campaign? rules-campaign))])]
    (if (every? second checked-rules)
      true
      (do (println {:bad-rules (vec (->> (filter (comp not second) checked-rules)
                                         (map first)))})
          nil))))
