(ns winst.presentation
  "The presentation layer of the winst server. It renders HTML reports
   for the trading activity, holdings, and realized gains/losses of an
   account over a specified period of time."
  {:author "Filippo Tampieri <fxt@fxtlabs.com>"}
  (:use [clojure.contrib.def :only (defvar-)]
        [clojure.contrib.string :only (upper-case)]
        hiccup.core
        hiccup.page-helpers
        [clj-time.core :only (now minus millis start end)]
        [winst.currency :only (currency-name)]
        [winst.securities :only (security-qualified-symbol lookup-security)]
        [winst.utils :only
         (normalize-activity
          activity-type-name
          format-date format-quantity format-currency format-percentage)]))

(defn- page-head
  "Returns the <head> element used by all pages."
  [{:keys [head] :as slots}]
  [:head
   [:title "Winst"]
   ;[:link {:rel "shortcut icon" :type "image/x-icon" :href "/images/favicon.ico"}]
   (include-css "/style.css")
   ;(include-js "http://ajax.googleapis.com/ajax/libs/jquery/1.4.4/jquery.min.js")
   
   head])

(defn- account-info
  "Returns the <div> element containing the info for the given account."
  [account]
  [:div {:class "account-info"}
   [:span (str (:name account) " - "
               (name (:type account)) " - "
               (currency-name (:currency account)))]
   [:br]
   [:span {:class "account-holder"} (:holder account)]])

(defn- page-header
  "Returns the <header> element used by all pages."
  [{:keys [title] :as slots}]
  [:header
   [:h1 title]])

(defn- page-footer
  "Returns the <footer> element used by all pages."
  [slots]
  [:footer])

(defn- page-body
  "Returns a <body> element with contents generated by the content
   function corresponding to the :content key in the given slots map."
  [{:keys [content] :as slots}]
  [:body
   [:div {:id "main" :class "container"}
    (page-header slots)
    (content slots)]
   (page-footer slots)])

(defn- render-page-template
  "Renders a page for the given slots map."
  [slots]
  (html5
   (page-head slots)
   (page-body slots)))

(defn- render-simple-page
  "Renders a simple page with a given tag, title, and text blurb (one
   paragraph)."
  [tag title blurb]
  (letfn [(content [{:keys [blurb]}]
                   [:div {:id "content"}
                    [:p blurb]])]
    (render-page-template {:tag tag
                           :content content
                           :title title
                           :blurb blurb})))


(defn render-not-found
  "Renders a 404 error (not found) page."
  []
  (render-simple-page :error
                      "404 Not Found"
                      "Sorry, the page you trying to view cannot be found."))

(defn render-error
  "Renders an error page for the given account and error message."
  [_ msg]
  (render-simple-page :error "Error" msg))

(defn- report-info
  "Returns the <div> element containing the info for the required report
   (e.g. date printed and reporting currency)."
  [currency]
  [:div {:class "report-info"}
   [:span (str "Reporting currency: " (currency-name currency))]
   [:br]
   [:span (str "Printed " (format-date (now)))]])

(defn- render-report
  "Renders a report page with the given pieces."
  [tag title account report-currency caption table]
  (letfn [(render-content [_]
                          [:div {:class "content"}
                           (account-info account)
                           [:br]
                           (report-info report-currency)
                           [:h2 caption]
                           table])]
    (render-page-template {:tag tag
                           :content render-content
                           :title title})))

(defn render-holdings
  "Renders a report showing the holdings for the given account at the given
   time and in the given reporting currency."
  [account report-currency report-time holdings]
  (let [tag :holdings
        title "Account Holdings"
        caption (str "Account Holdings as of "
                     (format-date (minus report-time (millis 1))))
        table [:table
               [:tr
                [:th {:class "number"} "Quantity"]
                [:th "Symbol"]
                [:th "Description"]
                [:th {:class "number"} "Book Value"]]
               (for [[security-uid holding] holdings]
                 (let [{:keys [quantity cost]} holding
                       security (lookup-security security-uid)]
                   [:tr
                    [:td {:class "number"} (format-quantity quantity)]
                    [:td (security-qualified-symbol security)]
                    [:td (:name security)]
                    [:td {:class "number"} (format-currency cost)]]))
               [:tr
                [:th ""]
                [:th ""]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:cost %2)) 0 (vals holdings)))]]]]
    (render-report tag title account report-currency caption table)))

(defn render-gains
  "Renders a report showing the realized gain/loss for the given account over
   the given time interval and in the given reporting currency."
  [account report-currency report-interval gains]
  (let [tag :gains
        title "Realized Gain Loss"
        caption (str "Realized Gain Loss from "
                     (format-date (start report-interval))
                     " to "
                     (format-date (minus (end report-interval)
                                         (millis 1))))
        table [:table
               [:tr
                [:th {:class "date"} "Transaction Date"]
                [:th {:class "activity"} "Transaction Type"]
                [:th "Symbol"]
                [:th "Description"]
                [:th {:class "number"} "Quantity"]
                [:th {:class "number"} "Average Cost"]
                [:th {:class "number"} "Book Value"]
                [:th {:class "number"} "Transaction Price"]
                [:th {:class "number"} "Net Amount"]
                [:th {:class "number"} "Realized Gain/Loss"]
                [:th {:class "number"} "Percentage Gain/Loss"]]
               (for [gain gains]
                 (let [{:keys [date security-uid quantity cost proceeds]} gain
                       security (lookup-security security-uid)]
                   [:tr
                    [:td {:class "date"} (format-date date)]
                    [:td {:class "activity"} "SELL"]
                    [:td (security-qualified-symbol security)]
                    [:td (:name security)]
                    [:td {:class "number"} (format-quantity (- quantity))]
                    [:td {:class "number"} (format-currency (/ cost quantity))]
                    [:td {:class "number"} (format-currency cost)]
                    [:td {:class "number"} (format-currency (/ proceeds quantity))]
                    [:td {:class "number"} (format-currency proceeds)]
                    [:td {:class "number"} (format-currency (- proceeds cost))]
                    [:td {:class "number"} (format-percentage (/ (- proceeds cost) cost))]]))
               [:tr
                [:th ""]
                [:th ""]
                [:th ""]
                [:th "Total"]
                [:th ""]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:cost %2)) 0 gains))]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:proceeds %2)) 0 gains))]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (- (:proceeds %2) (:cost %2))) 0 gains))]
                [:th ""]]]]
    (render-report tag title account report-currency caption table)))

(defn render-activities
  "Renders a report showing the trading activity for the given account over
   the given time interval and in the given reporting currency."
  [account report-currency report-interval activities]
  (let [tag :activities
        title "Trading Activity"
        caption (str "Trading Activity from "
                     (format-date (start report-interval))
                     " to "
                     (format-date (minus (end report-interval)
                                         (millis 1))))
        table [:table
               [:tr
                [:th {:class "date"} "Date"]
                [:th {:class "activity"} "Activity"]
                [:th {:class "number"} "Quantity"]
                [:th "Symbol"]
                [:th "Security Description"]
                [:th {:class "number"} "Price"]
                [:th {:class "number"} "Credit/(Debit)"]
                ]
               (for [item (mapcat normalize-activity activities)]
                 (let [{:keys [date type quantity security-uid
                               description price credit]} item
                       security (lookup-security security-uid)]
                   [:tr
                    [:td {:class "date"} (format-date date)]
                    [:td {:class "activity"} (activity-type-name type)]
                    [:td {:class "number"} (format-quantity quantity)]
                    [:td (security-qualified-symbol security)]
                    [:td description]
                    [:td {:class "number"} (if price (format-currency price) "")]
                    [:td {:class "number"} (if credit (format-currency credit) "")]]))
               ]]
    (render-report tag title account report-currency caption table)))

