(ns winst.securities
  (:use [clojure.java.io :only (resource)]
        [clojure.contrib.string :only (lower-case upper-case)]
        [clojure.contrib.def :only (defvar defvar-)]
        [clj-time.core :only (date-time)]
        [clojure-csv.core :only (parse-csv)]))

(defvar- exchanges [:nasdaq :nyse :amex])

(defn security-qualified-symbol [security]
  (let [{sym :symbol, exchange :exchange} security]
    (str (upper-case (name exchange)) ":" (upper-case sym))))

(defn- valid-record? [r]
  (and
   (vector? r)
   (= 9 (count r))
   (< 0 (count (first r)))))

(defn- convert-record [exchange r]
  {:exchange exchange
   :symbol (upper-case (nth r 0))
   :name (nth r 1)
   :sector (nth r 5)
   :industry (nth r 6)
   })

(defn- parse-company-list [exchange]
  (let [res (resource (str (lower-case (name exchange)) "_companies.csv"))]
    (->> res slurp parse-csv (drop 1) (filter valid-record?)
         (map (partial convert-record exchange)))))

(defn- create-map-by-exchange [f]
  (apply hash-map (mapcat (fn [ex] [ex (f ex)]) exchanges)))

(defvar- securities-by-exchange
  (create-map-by-exchange parse-company-list))

(defn- create-securities-map-by [key securities]
  (apply hash-map
         (mapcat (fn [security] [(key security) security]) securities)))

(defvar- securities-by-exchange-by-symbol
  (create-map-by-exchange
   (fn [ex] (create-securities-map-by :symbol (ex securities-by-exchange)))))

(defn lookup-security-symbol
  ([exchange sym]
     (get-in securities-by-exchange-by-symbol [exchange (upper-case sym)]))
  ([sym]
     (let [results (->> exchanges
                        (map #(lookup-security-symbol % sym))
                        (remove #(nil? %)))]
       (case (count results)
             0 nil
             1 (first results)
             (vec results)))))

(defvar security-uid security-qualified-symbol)

(defvar securities-by-uid
  (apply hash-map
         (mapcat (fn [sec] [(security-uid sec) sec])
                 (apply concat (vals securities-by-exchange)))))

(defn lookup-security [uid]
  (get securities-by-uid uid))

