(ns winst.presentation
  (:use [clojure.contrib.def :only (defvar-)]
        [clojure.contrib.string :only (upper-case)]
        hiccup.core
        hiccup.page-helpers
        [clj-time.format :only (formatters unparse)]
        [winst.currency :only (currency-name)]
        [winst.securities :only (security-qualified-symbol lookup-security)])
  (:require [clj-time.core :as ct]))

(defn- page-head [{:keys [head] :as slots}]
  [:head
   [:title "Winst"]
   ;[:link {:rel "shortcut icon" :type "image/x-icon" :href "/images/favicon.ico"}]
   (include-css "/style.css")
   ;(include-js "http://ajax.googleapis.com/ajax/libs/jquery/1.4.4/jquery.min.js")
   
   head])

(defn- account-info [account]
  [:div {:class "account-info"}
   [:span (str (:name account) " - "
               (name (:type account)) " - "
               (currency-name (:currency account)))]
   [:br]
   [:span {:class "account-holder"} (:holder account)]])

(defn- page-header [{:keys [title] :as slots}]
  [:header
   [:h1 title]])

(defn- page-footer [slots]
  [:footer])

(defn- page-body [{:keys [content] :as slots}]
  [:body
   [:div {:id "main" :class "container"}
    (page-header slots)
    (content slots)]
   (page-footer slots)])

(defn- render-page-template [slots]
  (html5
   (page-head slots)
   (page-body slots)))

(defn- render-simple-page [tag title blurb]
  (letfn [(content [{:keys [blurb]}]
                   [:div {:id "content"}
                    [:p blurb]])]
    (render-page-template {:tag tag
                           :content content
                           :title title
                           :blurb blurb})))


(defn render-not-found []
  (render-simple-page :error
                      "404 Not Found"
                      "Sorry, the page you trying to view cannot be found."))

(defn render-error [_ msg]
  (render-simple-page :error "Error" msg))


(defvar- date-formatter (formatters :year-month-day))

(defn format-date [dt]
  (unparse date-formatter dt))

(defn- format-quantity [f]
  (if (integer? f) (format "%(,d" f) (format "%(,.4f" (float f))))

(defn- format-currency [f]
  (format "%(,.2f" (float f)))

(defn- format-percentage [f]
  (format "%(,.2f" (* 100 f)))

(defn- type-name [t]
  (upper-case (name t)))


(defn- report-info [currency]
  [:div {:class "report-info"}
   [:span (str "Reporting currency: " (currency-name currency))]
   [:br]
   [:span (str "Printed " (format-date (ct/now)))]])

(defn- render-report [tag title account report-currency caption table]
  (letfn [(render-content [_]
                          [:div {:class "content"}
                           (account-info account)
                           [:br]
                           (report-info report-currency)
                           [:h2 caption]
                           table])]
    (render-page-template {:tag tag
                           :content render-content
                           :title title})))

(defn render-holdings [account report-currency report-time holdings]
  (let [tag :holdings
        title "Account Holdings"
        caption (str "Account Holdings as of "
                     (format-date (ct/minus report-time (ct/millis 1))))
        table [:table
               [:tr
                [:th {:class "number"} "Quantity"]
                [:th "Symbol"]
                [:th "Description"]
                [:th {:class "number"} "Book Value"]]
               (for [[security-uid holding] holdings]
                 (let [{:keys [quantity cost]} holding
                       security (lookup-security security-uid)]
                   [:tr
                    [:td {:class "number"} (format-quantity quantity)]
                    [:td (security-qualified-symbol security)]
                    [:td (:name security)]
                    [:td {:class "number"} (format-currency cost)]]))
               [:tr
                [:th ""]
                [:th ""]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:cost %2)) 0 (vals holdings)))]]]]
    (render-report tag title account report-currency caption table)))

(defn render-gains [account report-currency report-interval gains]
  (let [tag :gains
        title "Realized Gain Loss"
        caption (str "Realized Gain Loss from "
                     (format-date (ct/start report-interval))
                     " to "
                     (format-date (ct/minus (ct/end report-interval)
                                            (ct/millis 1))))
        table [:table
               [:tr
                [:th {:class "date"} "Transaction Date"]
                [:th {:class "activity"} "Transaction Type"]
                [:th "Symbol"]
                [:th "Description"]
                [:th {:class "number"} "Quantity"]
                [:th {:class "number"} "Average Cost"]
                [:th {:class "number"} "Book Value"]
                [:th {:class "number"} "Transaction Price"]
                [:th {:class "number"} "Net Amount"]
                [:th {:class "number"} "Realized Gain/Loss"]
                [:th {:class "number"} "Percentage Gain/Loss"]]
               (for [gain gains]
                 (let [{:keys [date security-uid quantity cost proceeds]} gain
                       security (lookup-security security-uid)]
                   [:tr
                    [:td {:class "date"} (format-date date)]
                    [:td {:class "activity"} "SELL"]
                    [:td (security-qualified-symbol security)]
                    [:td (:name security)]
                    [:td {:class "number"} (format-quantity (- quantity))]
                    [:td {:class "number"} (format-currency (/ cost quantity))]
                    [:td {:class "number"} (format-currency cost)]
                    [:td {:class "number"} (format-currency (/ proceeds quantity))]
                    [:td {:class "number"} (format-currency proceeds)]
                    [:td {:class "number"} (format-currency (- proceeds cost))]
                    [:td {:class "number"} (format-percentage (/ (- proceeds cost) cost))]]))
               [:tr
                [:th ""]
                [:th ""]
                [:th ""]
                [:th "Total"]
                [:th ""]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:cost %2)) 0 gains))]
                [:th ""]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (:proceeds %2)) 0 gains))]
                [:th {:class "number"}
                 (format-currency (reduce #(+ %1 (- (:proceeds %2) (:cost %2))) 0 gains))]
                [:th ""]]]]
    (render-report tag title account report-currency caption table)))

(defn- normalize-activity [{:keys [date type quantity security-uid] :as activity}]
  (let [security (lookup-security security-uid)]
    (case type
          :buy [{:date date
                 :type type
                 :quantity quantity
                 :symbol (security-qualified-symbol security)
                 :description (:name security)
                 :price (/ (:cost activity) quantity)
                 :credit (- (:cost activity))}]
          :sell [{:date date
                  :type type
                  :quantity (- quantity)
                  :symbol (security-qualified-symbol security)
                  :description (:name security)
                  :price (/ (:proceeds activity) quantity)
                  :credit (:proceeds activity)
                  }]
          :split (let [r (:split-ratio activity)
                       [to from] (if (ratio? r)
                                   [(numerator r) (denominator r)] [r 1])]
                   [{:date date
                     :type type
                     :quantity (* (- r 1) quantity)
                     :symbol (security-qualified-symbol security)
                     :description (str "Split "
                                       (format-quantity quantity) " "
                                       (:name security) " "
                                       to " for " from)}])
          :exchange (let [{:keys [new-quantity new-security-uid]} activity
                          new-security (lookup-security new-security-uid)]
                      [{:date date
                        :type type
                        :quantity (- quantity)
                        :symbol (security-qualified-symbol security)
                        :description (str (:name security)
                                          "; exchanged for "
                                          (format-quantity new-quantity)
                                          " " (:name new-security))}
                       {:date date
                        :type type
                        :quantity new-quantity
                        :symbol (security-qualified-symbol new-security)
                        :description (str (:name new-security)
                                          "; exchanged for "
                                          (format-quantity quantity)
                                          " " (:name security))}])
          (throw (RuntimeException. "Unexpected activity type!")))))

(defn render-activities [account report-currency report-interval activities]
  (let [tag :activities
        title "Trading Activity"
        caption (str "Trading Activity from "
                     (format-date (ct/start report-interval))
                     " to "
                     (format-date (ct/minus (ct/end report-interval)
                                            (ct/millis 1))))
        table [:table
               [:tr
                [:th {:class "date"} "Date"]
                [:th {:class "activity"} "Activity"]
                [:th {:class "number"} "Quantity"]
                [:th "Symbol"]
                [:th "Security Description"]
                [:th {:class "number"} "Price"]
                [:th {:class "number"} "Credit/(Debit)"]
                ]
               (for [item (mapcat normalize-activity activities)]
                 (let [{:keys [date type quantity symbol description price credit]} item]
                   [:tr
                    [:td {:class "date"} (format-date date)]
                    [:td {:class "activity"} (type-name type)]
                    [:td {:class "number"} (format-quantity quantity)]
                    [:td symbol]
                    [:td description]
                    [:td {:class "number"} (if price (format-currency price) "")]
                    [:td {:class "number"} (if credit (format-currency credit) "")]]))
               ]]
    (render-report tag title account report-currency caption table)))

