(ns winst.currency
  (:use [clojure.java.io :only (reader resource)]
        [clojure.contrib.def :only (defvar-)]
        [clojure.contrib.string :only (lower-case upper-case)]
        [clj-time.core :only (date-time year month day)]
        clojure.test))

(defvar- currency-pairs [[:usd :cad]])

(defvar- re-line
  #"([12][0-9]{3})/(0?[1-9]|1[012])/([0-9]{1,2}),([0-9]+\.[0-9]*).*")

(defn- valid-record? [r]
  (and r
       (= 5 (count r))))

(defn- convert-record [r]
  (let [year (Integer/parseInt (nth r 1) 10)
        month (Integer/parseInt (nth r 2) 10)
        day (Integer/parseInt (nth r 3) 10)
        rate (Float/parseFloat (nth r 4))]
    [(date-time year month day) rate]))

(defn- parse-exchange-rates [res]
  (with-open [rdr (reader res)]
    (apply hash-map (->> (line-seq rdr)
                         (map (partial re-matches re-line))
                         (filter valid-record?)
                         (mapcat convert-record)))))

(defn- resource-for [[from to]]
  (resource (str (name from) "_" (name to) "_rates.csv")))

(defvar- currencies-rates-map
  (apply hash-map
         (mapcat (fn [pair]
                   [pair (parse-exchange-rates (resource-for pair))])
                 currency-pairs)))

(defn currency-name
  "Turn a currency keyword into an upper-case string."
  [k]
  (-> k name str upper-case))

(defn currency-keyword
  "Turn a currency string into a keyword."
  [#^String s]
  (-> s lower-case keyword))

(defn- build-exchange-rate-lookup [from to m]
  (fn [dt]
    (let [dt0 (date-time (year dt) (month dt) (day dt))]
      (if-let [r (m dt0)]
        r
        (throw (RuntimeException. (str "No " (currency-name from) " to "
                                       (currency-name to)
                                       " exchange rate for " dt)))))))

(defn- reciprocal-lookup [f]
  (fn [dt] (/ 1.0 (f dt))))

(defn get-exchange-rate-lookup
  "Return a function that returns the exchange rate at the closing
   of the given time (a org.joda.time.DateTime object) from the first
   currency (keyword) to the second currency (keyword)."
  [from to]
  (if (= from to)
    (constantly 1.0)
    (condp #(get %2 %1) currencies-rates-map
      [from to] :>> #(build-exchange-rate-lookup from to %)
      [to from] :>> #(-> (build-exchange-rate-lookup to from %) reciprocal-lookup)
      nil)))

;; C-c , to run the tests
;; C-c ' to see the error details (you must be on the line of the test)
;; C-c k to clear the errors

(deftest test-currency-name
  (is (= "USD" (currency-name :usd)))
  (is (= "USD" (currency-name :USD))))

(deftest test-build-rate-lookup
  (let [usd-cad-rates (get currencies-rates-map [:usd :cad])]
    (is (< 1.4 ((build-exchange-rate-lookup :usd :cad usd-cad-rates)
                (date-time 2000 1 4 8))))))

(deftest test-reciprocal-lookup
  (let [usd-cad-rates (get currencies-rates-map [:usd :cad])
        lookup-rate (build-exchange-rate-lookup :usd :cad usd-cad-rates)
        dt (date-time 2000 1 4)]
    (is (< 1.4 (lookup-rate dt)))
    (is (> (/ 1.0 1.4) ((reciprocal-lookup lookup-rate) dt)))))

(deftest test-get-exchange-rate-lookup
  (let [dt (date-time 2000 1 4)]
    (let [lookup-rate (get-exchange-rate-lookup :usd :cad)]
      (is (< 1.4 (lookup-rate dt))))
    (let [lookup-rate (get-exchange-rate-lookup :cad :usd)]
      (is (> (/ 1.0 1.4) (lookup-rate dt))))))
