(ns com.fulcrologic.statecharts.simple
  "Functions that set up the simplest version of a state chart that uses the v20150901
   implementation (version), a working memory data model,
   CLJC execution support, and an event queue that requires manual polling."
  (:require
    [clojure.string :as str]
    [com.fulcrologic.statecharts :as sc]
    [com.fulcrologic.statecharts.algorithms.v20150901 :as alg]
    [com.fulcrologic.statecharts.algorithms.v20150901-validation :as v]
    [com.fulcrologic.statecharts.data-model.working-memory-data-model :as wmdm]
    [com.fulcrologic.statecharts.event-queue.manually-polled-queue :as mpq]
    [com.fulcrologic.statecharts.execution-model.lambda :as lambda]
    [com.fulcrologic.statecharts.registry.local-memory-registry :as lmr]
    [com.fulcrologic.statecharts.working-memory-store.local-memory-store :as lms]
    [com.fulcrologic.statecharts.protocols :as sp]
    [com.fulcrologic.statecharts.util :refer [new-uuid]]
    [com.fulcrologic.statecharts.invocation.statechart :as i.statechart]
    #?(:clj [com.fulcrologic.statecharts.invocation.future :as i.future])
    [taoensso.timbre :as log]))

(defn simple-env
  "Creates an env that has a local and simple implementation of all required components.

   It defaults to standard processing with a flat working memory data model, a manual event queue,
   local memory registry and memory storage, a statechart invocation processor,
   the lambda executor, and a v20150901 processor.

   `extra-env` can contain anything extra you want in `env`, and can override any of the above by
   key (e.g. ::sc/data-model).

   Returns an `env` ready to be used with the processor (which is ::sc/processor in `env`).

   Remember to register your charts via `(env/register! env k chart)` or by using the
   ::sc/statechart-registry returned in `env` directly.

   ```
   (def env (simple-env))
   (register! env ::chart some-chart)
   (def processor (::sc/processor env))
   (def s0 (sp/start! processor env ::chart {::sc/session-id 42}))
   (def s1 (sp/process-event! processor env s0 event))
   ...
   ```
   "
  ([] (simple-env {}))
  ([{::sc/keys [data-model execution-model event-queue
                working-memory-store statechart-registry] :as extra-env}]
   (let [dm       (or data-model (wmdm/new-flat-model))
         q        (or event-queue (mpq/new-queue))
         ex       (or execution-model (lambda/new-execution-model dm q))
         registry (or statechart-registry (lmr/new-registry))
         wmstore  (or working-memory-store (lms/new-store))
         env      (merge {::sc/statechart-registry   registry
                          ::sc/data-model            dm
                          ::sc/event-queue           q
                          ::sc/working-memory-store  wmstore
                          ::sc/processor             (alg/new-processor)
                          ::sc/invocation-processors #?(:clj
                                                        [(i.statechart/new-invocation-processor)
                                                         (i.future/new-future-processor)]
                                                        :cljs
                                                        [(i.statechart/new-invocation-processor)])
                          ::sc/execution-model       ex}
                    extra-env)]
     env)))

(defn register!
  "Register a statechart `chart` at `chart-key` in the registry known by `env`."
  [{::sc/keys [statechart-registry]} chart-key chart]
  (let [problems  (v/problems chart)
        errors?   (boolean (some #(= :error (:level %)) problems))
        warnings? (boolean (some #(= :warn (:level %)) problems))]
    (cond
      errors? (throw (ex-info "Cannot register invalid chart" {:chart-key chart-key
                                                               :problems  (vec problems)}))
      warnings? (log/warn "Chart" chart-key "has problems: " (str/join ","
                                                               (map (fn [{:keys [element message]}]
                                                                      (str element ": " message)) problems))))
    (sp/register-statechart! statechart-registry chart-key chart))
  true)

(defn start!
  "Start a statechart that has been previously registered with `env` as `chart-src`. The options map
  can contain ::sc/session-id or one will be autogenerated. Note: This assumes you're going to send events
  via the event queue, and that something is running an event loop on that event queue.

  Returns nothing useful, but throws if there is a problem."
  ([env chart-src]
   (start! env chart-src (new-uuid)))
  ([{::sc/keys [processor working-memory-store statechart-registry] :as env} chart-src session-id]
   (assert statechart-registry "There is a statechart registry in env")
   (assert working-memory-store "There is a working memory store in env")
   (assert (sp/get-statechart statechart-registry chart-src) (str "A chart is registered under " chart-src))
   (let [session-id session-id
         s0         (sp/start! processor env chart-src {::sc/session-id session-id})]
     (sp/save-working-memory! working-memory-store env session-id s0))
   true))

(defn send!
  "Proxy to sp/send! on the event-queue in `env`."
  [{::sc/keys [event-queue] :as env} event]
  (sp/send! event-queue env event))
