(ns com.fulcrologic.guardrails.malli.core
  #?(:cljs (:require-macros com.fulcrologic.guardrails.malli.core))
  (:require
    #?@(:clj [[clojure.spec.alpha :as s]
              [clojure.string :as string]
              [com.fulcrologic.guardrails.config :as gr.cfg]
              [com.fulcrologic.guardrails.impl.pro :as gr.pro]
              [com.fulcrologic.guardrails.utils :refer [cljs-env? clj->cljs]]])
    [com.fulcrologic.guardrails.core :as gr.core]
    [malli.core :as m]
    [malli.error :as me]
    [malli.registry :as mr]))


(def => :ret)
(def | :st)
(def <- :gen)

#?(:clj
   (defmacro ? [& forms]
     (cond-> `[:maybe ~@forms]
       (cljs-env? &env) clj->cljs)))

(comment
  #?(:clj
      (def spec-elem-malli-only
        (s/or
          :pred-sym (s/and symbol?
                      (complement #{'| '=>})
                      ;; REVIEW: should the `?` be a requirement?
                      #(string/ends-with? (str %) "?"))
          :gspec (s/or :nilable-gspec ::gr.core/nilable-gspec :gspec ::gr.core/gspec)
          :spec-key qualified-keyword?
          :malli-key (s/and simple-keyword?
                       (complement #{:st :ret :gen}))
          :malli-sym (s/and symbol?
                       (complement #{'| '=> '<-}))
          :malli-vec (s/and vector?
                       #(simple-keyword? (first %)))
          :fun ::gr.core/pred-fn))))


#?(:clj
   ;; REVIEW: There doesn't appear to be a straightforward way to properly split
   ;; the leaf specs between spec and malli without duplicating all the root ones,
   ;; so for now we accept that some valid spec elements which would be invalid in
   ;; Malli will pass compile-time macro-syntax checking
   (s/def ::gr.core/spec-elem
     (s/or
       :set set?
       :pred-sym (s/and symbol?
                   (complement #{'| '=>})
                   ;; REVIEW: should the `?` be a requirement?
                   #(string/ends-with? (str %) "?"))
       :gspec (s/or :nilable-gspec ::gr.core/nilable-gspec :gspec ::gr.core/gspec)
       :spec-key qualified-keyword?
       :malli-key (s/and simple-keyword?
                    (complement #{:st :gen :ret}))
       :malli-sym (s/and symbol?
                    (complement #{'| '=> '<-}))
       :malli-vec (s/and vector?
                    #(simple-keyword? (first %)))
       :fun ::gr.core/pred-fn
       :list seq?)))


(defonce ^:private registry* (atom (m/default-schemas)))
(defn register! [type ?schema] (swap! registry* assoc type ?schema))
(mr/set-default-registry! (mr/mutable-registry registry*))


#?(:clj
   (defmacro >def [k v] `(register! ~k ~v)))


#?(:clj
   (do
     (defmacro >defn
       "Like defn, but requires a (nilable) gspec definition and generates
       additional Malli function schema metadata and validation code."
       {:arglists '([name doc-string? attr-map? [params*] gspec prepost-map? body?]
                    [name doc-string? attr-map? ([params*] gspec prepost-map? body?) + attr-map?])}
       [& forms]
       (gr.core/>defn* &env &form forms {:private? false :malli? true}))

     (s/fdef >defn :args ::gr.core/>defn-args)))


#?(:clj
   (do
     (defmacro >defn-
       "Like defn-, but requires a (nilable) gspec definition and generates
       additional Malli function schema metadata and validation code."
       {:arglists '([name doc-string? attr-map? [params*] gspec prepost-map? body?]
                    [name doc-string? attr-map? ([params*] gspec prepost-map? body?) + attr-map?])}
       [& forms]
       (gr.core/>defn* &env &form forms {:private? true :malli? true}))

     (s/fdef >defn- :args ::gr.core/>defn-args)))


#?(:clj
   (do
     (defmacro >fdef
       "Defines a Malli function schema using gspec syntax – pretty much a
       `>defn` without the body. Desugars to `(malli.core/=> ...)."
       {:arglists '([name [params*] gspec]
                    [name ([params*] gspec) +])}
       [& forms]
       (when-let [cfg (gr.cfg/get-env-config)]
         (let [env (assoc &env :malli? true)]
           `(do
              ~(when (#{:pro :copilot :all} (gr.cfg/mode cfg))
                 (gr.pro/>fdef-impl env forms))
              ~(cond-> (remove nil? (gr.core/generate-fdef env forms))
                 (cljs-env? &env) clj->cljs)))))

     (s/fdef >fdef :args ::gr.core/>fdef-args)))

