;; Copyright (c) George Lipov. All rights reserved.
;; Additional code Copyright Fulcrologic, LLC.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 2.0 (https://choosealicense.com/licenses/epl-2.0/)
;; which can be found in the file LICENSE at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns com.fulcrologic.guardrails.core
  #?(:cljs (:require-macros com.fulcrologic.guardrails.core)))

;; It doesn't actually matter what these are bound to, they are stripped by
;; the macros they're used in and never end up in the final code. This is just
;; so they can be used without '=> cannot be resolved' errors in the IDE.
(def => :ret)
(def | :st)
(def <- :gen)

(defonce ^:private !global-context nil)

(defn enter-global-context!
  "Push a global context, accessible from all threads, onto a stack. Used to add
  information to what guardrails will report when a function failed a check."
  [ctx]
  )

(defn leave-global-context!
  "Pops a global context (see `enter-global-context!`). Should be passed the
  same context that was pushed, although is not enforced, as it's only to be
  easily compatible with fulcro-spec's hooks API."
  [ctx]
  )

#?(:clj
   (defmacro with-global-context
     "Wraps the body with an enter and leave global context.
      Will always call leave as it uses a try finally block.
      See `enter-global-context!`."
     [ctx & body]
     @body
     ))

(defn- get-global-context [])

#?(:clj
   (defn now-ms ^long []
     )
   :cljs
   (defn now-ms []
     ))

(defn humanize-spec [explain-data {:guardrails/keys [compact? args? fqnm] :as expound-options}]
  )

(defn run-check [{:guardrails/keys [malli? compact? use-stderr? validate-fn explain-fn fqnm humanize-fn]
                  :keys            [tap>? args? vararg? humanize-opts throw? fn-name]
                  :as              options}
                 spec
                 value]
  nil)

#?(:clj
   (defn clean-defn
     "This removes the gspec and returns a clean defn for use in production
     builds."
     [op forms]
     ))

#?(:clj
   (defmacro ? [& forms]
     nil))

#?(:clj
   (defn generate-fdef [env forms]))

;; The now-nano was taken from taoensso/encore. I didn't want to include that entire dep just for this one
;; function. The are covered by Eclipse Public License - v 1.0. See https://github.com/taoensso/encore
#?(:cljs (def ^:no-doc js-?window nil))                     ; Present iff in browser

#?(:cljs
   (defn oget "Like `get` for JS objects."
     ([k])
     ([o k])
     ([o k not-found])))

#?(:clj
   (defn now-nano
     "Returns current value of best-resolution time source as nanoseconds."
     ^long [])

   :cljs
   (def now-nano
     "Returns current value of best-resolution time source as nanoseconds."
     ))

#?(:clj
   (defmacro >defn
     "Like defn, but requires a (nilable) gspec definition and generates
     additional `s/fdef` function spec definition and validation code."
     {:arglists '([name doc-string? attr-map? [params*] gspec prepost-map? body?]
                  [name doc-string? attr-map? ([params*] gspec prepost-map? body?) + attr-map?])}
     [& forms]
     `(defn ~@forms)))

#?(:clj
   (defmacro >defn-
     "Like defn-, but requires a (nilable) gspec definition and generates
     additional `s/fdef` function spec definition and validation code."
     {:arglists '([name doc-string? attr-map? [params*] gspec prepost-map? body?]
                  [name doc-string? attr-map? ([params*] gspec prepost-map? body?) + attr-map?])}
     [& forms]
     `(defn- ~@forms)))

#?(:clj
   (defmacro >def
     "Just like Clojure s/def, except there is a stub for this in the `noop`
     namespace, which you can substitute via CLJS build parameters, turning it
     into code that can be dead-code eliminated in a CLJS production build. See
     the docstring for the `com.fulcrologic.guardrails.noop` namespace."
     ([k spec-form])
     ([k _doc spec-form])))

#?(:clj
   (defmacro >fdef
     "Defines an fspec using gspec syntax – pretty much a `>defn` without the body.

     `name` can be a symbol or a qualified keyword, depending on whether the
     fspec is meant to be registered as a top-level fspec (=> s/fdef fn-sym ..)
     or used in other specs (=> s/def ::spec-keyword (s/fspec ...)). "
     {:arglists '([name [params*] gspec]
                  [name ([params*] gspec) +])}
     [& forms]
     ))

#?(:clj
   ;; TODO: clean >fn (no gspec)
   (defmacro >fn [& forms]))

#?(:clj
   (defmacro >fspec [& forms]
     ))
