(ns com.fulcrologic.fulcro.clj-testing
  "CLJ-only testing support for Fulcro applications.

   This namespace provides utilities for synchronous testing of Fulcro applications
   in Clojure (JVM). Key features:

   - Synchronous transaction processing (all operations complete before returning)
   - Render frame capture (inspect before/after state for assertions)
   - Hiccup rendering with preserved event handlers for DOM inspection
   - Controlled execution (step through transaction phases)
   - Integration with Ring handlers and Pathom parsers

   Example:
   ```clojure
   (require '[com.fulcrologic.fulcro.clj-testing :as ct])
   (require '[com.fulcrologic.fulcro.clj-testing.remote :as ctr])

   (def app (ct/build-test-app
              {:root-class Root
               :remotes {:remote (ctr/sync-remote my-handler)}}))

   ;; Transactions are synchronous
   (comp/transact! app [(my-mutation)])

   ;; Inspect render frames (includes hiccup)
   (let [frame (ct/last-frame app)]
     (is (= expected (:tree frame)))
     (is (some? (ct/find-in-hiccup (:hiccup frame) \"my-button\"))))

   ;; Click on elements to trigger their handlers
   (ct/click-on! app \"submit-button\")
   ```"
  (:require
    [clojure.string :as str]
    [com.fulcrologic.fulcro.algorithms.denormalize :as fdn]
    [com.fulcrologic.fulcro.algorithms.lookup :as ah]
    [com.fulcrologic.fulcro.algorithms.tx-processing.synchronous-tx-processing :as stx]
    [com.fulcrologic.fulcro.components :as comp]
    [com.fulcrologic.fulcro.dom-hiccup :as domh]
    [com.fulcrologic.fulcro.raw.application :as rapp]
    [com.fulcrologic.fulcro.raw.components :as rc]
    [taoensso.timbre :as log]))

(def ^:dynamic *render-frame-max*
  "Default maximum number of render frames to keep in history."
  10)

(defn- render-app-hiccup
  "Render the app's root component to hiccup.
   Binds the proper dynamic vars for component rendering."
  [app]
  (let [{:com.fulcrologic.fulcro.application/keys [state-atom runtime-atom]} app
        state-map @state-atom
        {:com.fulcrologic.fulcro.application/keys [root-class root-factory]} @runtime-atom
        ;; Create factory if not present (common when using raw/fulcro-app)
        factory (or root-factory
                    (when root-class
                      (comp/factory root-class)))]
    (when (and root-class factory)
      (let [query (comp/get-query root-class state-map)
            tree (fdn/db->tree query state-map state-map)]
        (binding [comp/*app*    app
                  comp/*parent* nil
                  comp/*shared* (comp/shared app)]
          (domh/render-to-hiccup (factory tree)))))))

(defn- capture-frame
  "Capture a render frame from the current app state."
  [app]
  (let [{:com.fulcrologic.fulcro.application/keys [state-atom runtime-atom]} app
        state-map @state-atom
        {:com.fulcrologic.fulcro.application/keys [root-class]} @runtime-atom
        query (when root-class (comp/get-query root-class state-map))
        tree (when query (fdn/db->tree query state-map state-map))
        hiccup (try
                 (render-app-hiccup app)
                 (catch Exception e
                   (log/trace e "Could not render hiccup (root component may not have render fn)")
                   nil))]
    {:state     state-map
     :tree      tree
     :hiccup    hiccup
     :timestamp (System/currentTimeMillis)}))

(defn- add-frame!
  "Add a frame to the render history, maintaining max size."
  [app frame]
  (let [runtime-atom (:com.fulcrologic.fulcro.application/runtime-atom app)
        max-frames (or (::render-frame-max @runtime-atom) *render-frame-max*)]
    (swap! runtime-atom update ::render-frames
      (fn [frames]
        (let [new-frames (vec (cons frame (or frames [])))]
          (if (> (count new-frames) max-frames)
            (subvec new-frames 0 max-frames)
            new-frames))))))

(defn- frame-capturing-render
  "A render function that captures frames to history.
   This replaces the optimized-render! algorithm."
  [app options]
  (let [frame (capture-frame app)]
    (add-frame! app frame)
    frame))

(defn current-state
  "Get the current state map from the app."
  [app]
  (-> app :com.fulcrologic.fulcro.application/state-atom deref))

(defn render-history
  "Get the render history (newest first).
   Each entry is a map with:
   - :state - The state map at render time
   - :tree - The denormalized props tree
   - :timestamp - System time in milliseconds"
  [app]
  (-> app :com.fulcrologic.fulcro.application/runtime-atom deref ::render-frames))

(defn last-frame
  "Get the most recent render frame, or nil if none."
  [app]
  (first (render-history app)))

(defn clear-render-history!
  "Clear all captured render frames."
  [app]
  (swap! (:com.fulcrologic.fulcro.application/runtime-atom app)
    assoc ::render-frames []))

(defn render-frame!
  "Force a render and capture the frame in history.
   Returns the rendered tree (denormalized props)."
  [app]
  (let [render! (ah/app-algorithm app :render!)]
    (when render!
      (render! app {:force-root? true}))
    (:tree (last-frame app))))

(defn get-props
  "Get denormalized props for a component at the given ident.
   Useful for inspecting component state in tests."
  [app component-class ident]
  (let [state-map (current-state app)
        query (comp/get-query component-class state-map)]
    (fdn/db->tree query (get-in state-map ident) state-map)))

(defn query-tree
  "Query the current tree for a specific ident's data.
   Returns the entity at the given ident, denormalized."
  [app ident]
  (let [state-map (current-state app)]
    (get-in state-map ident)))

(defn build-test-app
  "Create a test application configured for synchronous testing.

   Options:
   - :root-class - Root component class (required for render frame capture)
   - :remotes - Map of remote-name to remote implementation
   - :initial-state - Initial state map (merged with root's initial state)
   - :render-history-size - Number of frames to keep (default 10)
   - :shared - Static shared props
   - :shared-fn - Function to compute shared props from root tree

   Returns a Fulcro app configured with:
   - Synchronous transaction processing
   - Frame-capturing render
   - The specified remotes"
  [{:keys [root-class remotes initial-state render-history-size shared shared-fn]
    :or   {render-history-size *render-frame-max*
           remotes             {}}}]
  (let [;; Create the base app with sync tx processing
        base-app (rapp/fulcro-app
                   {:initial-db        (or initial-state {})
                    :root-class        root-class
                    :remotes           (if (empty? remotes)
                                         {:remote {:transmit! (fn [{:com.fulcrologic.fulcro.algorithms.tx-processing/keys [result-handler]}]
                                                               (log/warn "No remote configured, returning empty response")
                                                               (result-handler {:status-code 200 :body {}}))}}
                                         remotes)
                    :shared            shared
                    :shared-fn         shared-fn
                    :optimized-render! frame-capturing-render
                    :core-render!      (fn [app options]
                                         (let [optimized-render! (ah/app-algorithm app :optimized-render!)]
                                           (binding [comp/*app*    app
                                                     comp/*parent* nil
                                                     comp/*shared* (comp/shared app)]
                                             (optimized-render! app options))))})
        ;; Install sync tx processing
        app (stx/with-synchronous-transactions base-app)]
    ;; Initialize state from root if provided
    (when root-class
      (rapp/initialize-state! app root-class))
    ;; Set render history size
    (swap! (:com.fulcrologic.fulcro.application/runtime-atom app)
      assoc ::render-frame-max render-history-size
            ::render-frames [])
    app))

(defn set-remote!
  "Set or replace a remote on the test app.
   Convenience wrapper around rapp/set-remote!"
  [app remote-name remote]
  (rapp/set-remote! app remote-name remote))

(defn wait-for-idle!
  "Block until all pending work is complete.
   With synchronous tx processing, this is usually instant,
   but useful after operations that might queue follow-on work."
  [app]
  (loop [iterations 0]
    (when (< iterations 100) ; Safety limit
      (let [submission-queue (stx/submission-queue app)
            active-queue (stx/active-queue app)]
        (when (or (seq submission-queue) (seq active-queue))
          (Thread/sleep 1)
          (recur (inc iterations)))))))

(defn pending-sends
  "Get the pending send queue for a remote.
   Useful for verifying what would be sent to the server."
  [app remote-name]
  (stx/send-queue app remote-name))

(defn has-pending-work?
  "Returns true if there is any pending transaction work."
  [app]
  (or (stx/available-work? app)
      (seq (stx/active-queue app))
      (stx/post-processing? app)))

(defmacro with-test-app
  "Convenience macro that creates a test app, binds it to `app-sym`,
   and ensures cleanup after body executes.

   Example:
   ```clojure
   (with-test-app [app {:root-class Root}]
     (comp/transact! app [(my-mutation)])
     (is (= expected (current-state app))))
   ```"
  [[app-sym options] & body]
  `(let [~app-sym (build-test-app ~options)]
     (try
       ~@body
       (finally
         (clear-render-history! ~app-sym)))))

(defn reset-app!
  "Reset the app state to initial state from root component.
   Useful for test isolation."
  [app]
  (let [runtime-atom (:com.fulcrologic.fulcro.application/runtime-atom app)
        {:com.fulcrologic.fulcro.application/keys [root-class]} @runtime-atom]
    (when root-class
      (reset! (:com.fulcrologic.fulcro.application/state-atom app) {})
      (rapp/initialize-state! app root-class)
      (clear-render-history! app))))

(defn state-at-render
  "Get the state map from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (-> (render-history app)
      (nth frame-index nil)
      :state))

(defn tree-at-render
  "Get the denormalized tree from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (-> (render-history app)
      (nth frame-index nil)
      :tree))

(defn frames-since
  "Get all frames captured since the given timestamp."
  [app timestamp]
  (take-while #(> (:timestamp %) timestamp) (render-history app)))

(defn with-render-tracking
  "Execute body and return a map with:
   - :result - The return value of body
   - :frames - All frames captured during execution
   - :frame-count - Number of frames captured

   Example:
   ```clojure
   (let [{:keys [result frames]} (with-render-tracking app
                                   (comp/transact! app [(my-mutation)]))]
     (is (= 1 (count frames))))
   ```"
  [app body-fn]
  (let [start-count (count (render-history app))
        result (body-fn)
        end-count (count (render-history app))
        new-frame-count (- end-count start-count)]
    {:result      result
     :frames      (take new-frame-count (render-history app))
     :frame-count new-frame-count}))

(defn assert-state
  "Assert that the current state matches expected at the given path.
   Throws AssertionError with descriptive message on failure.

   Example:
   ```clojure
   (assert-state app [:user/id 1 :user/name] \"John\")
   ```"
  [app path expected]
  (let [actual (get-in (current-state app) path)]
    (assert (= expected actual)
      (str "State mismatch at " path
           "\n  Expected: " (pr-str expected)
           "\n  Actual: " (pr-str actual)))))

(defn assert-no-pending-work
  "Assert that there is no pending transaction work.
   Useful at the end of tests to ensure all async operations completed."
  [app]
  (assert (not (has-pending-work? app))
    (str "App has pending work:"
         "\n  Submission queue: " (stx/submission-queue app)
         "\n  Active queue: " (stx/active-queue app)
         "\n  Post-processing: " (stx/post-processing? app))))

;; =============================================================================
;; Hiccup Helpers
;; =============================================================================

(defn last-hiccup
  "Get the hiccup from the most recent render frame.
   Returns nil if no frames captured or no hiccup available."
  [app]
  (:hiccup (last-frame app)))

(defn hiccup-at-render
  "Get the hiccup from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (-> (render-history app)
      (nth frame-index nil)
      :hiccup))

(defn find-in-hiccup
  "Find an element in the hiccup tree by its :id attribute.
   If hiccup is not provided, uses the most recent frame's hiccup.

   Returns the first matching element as a hiccup vector, or nil."
  ([app id]
   (find-in-hiccup app id (last-hiccup app)))
  ([app id hiccup]
   (domh/find-element-by-id hiccup id)))

(defn find-all-in-hiccup
  "Find all elements in the hiccup tree matching the predicate.
   If hiccup is not provided, uses the most recent frame's hiccup.

   The predicate receives each hiccup element vector.
   Returns a vector of matching elements."
  ([app pred]
   (find-all-in-hiccup app pred (last-hiccup app)))
  ([app pred hiccup]
   (domh/find-elements hiccup pred)))

(defn find-by-tag
  "Find all elements with the given tag keyword.
   If hiccup is not provided, uses the most recent frame's hiccup."
  ([app tag]
   (find-by-tag app tag (last-hiccup app)))
  ([app tag hiccup]
   (domh/find-elements-by-tag hiccup tag)))

(defn find-by-class
  "Find all elements with the given CSS class.
   If hiccup is not provided, uses the most recent frame's hiccup."
  ([app class-name]
   (find-by-class app class-name (last-hiccup app)))
  ([app class-name hiccup]
   (domh/find-elements-by-class hiccup class-name)))

(defn element-text
  "Get the text content of a hiccup element (recursively extracts all text).
   Returns a string with all text content concatenated."
  [elem]
  (domh/element-text elem))

(defn element-attr
  "Get an attribute value from a hiccup element."
  [elem attr-key]
  (domh/element-attr elem attr-key))

(defn click-on!
  "Simulate a click on an element by finding it by ID and invoking its :onClick handler.

   The element must exist in the current render and have an :onClick handler.
   The handler should be a function (real lambda, not stringified).

   Options:
   - :event - The synthetic event to pass to the handler (default: {})
   - :render? - Whether to force a render after the click (default: true)

   Returns the result of invoking the onClick handler, or nil if no handler.
   Throws if element not found.

   Example:
   ```clojure
   (click-on! app \"submit-button\")
   (click-on! app \"toggle\" :event {:target {:checked true}})
   ```"
  [app element-id & {:keys [event render?]
                     :or   {event {} render? true}}]
  ;; First render to ensure we have current hiccup
  (when render?
    (render-frame! app))
  (let [hiccup (last-hiccup app)]
    (if-let [element (find-in-hiccup app element-id hiccup)]
      (let [attrs (domh/hiccup-attrs element)
            on-click (:onClick attrs)]
        (cond
          (nil? on-click)
          (log/warn "Element" element-id "has no :onClick handler")

          (fn? on-click)
          (on-click event)

          :else
          (throw (ex-info "onClick handler is not a function"
                   {:element-id element-id
                    :onClick    on-click
                    :type       (type on-click)}))))
      (throw (ex-info "Element not found"
               {:element-id element-id})))))

(defn invoke-handler!
  "Generic handler invocation - find element by ID and invoke a specific handler.

   handler-key is the attribute key (e.g., :onClick, :onChange, :onSubmit).

   Returns the result of invoking the handler, or nil if no handler.
   Throws if element not found.

   Example:
   ```clojure
   (invoke-handler! app \"email-input\" :onChange {:target {:value \"test@example.com\"}})
   (invoke-handler! app \"my-form\" :onSubmit {:preventDefault (fn [])})
   ```"
  [app element-id handler-key & {:keys [event render?]
                                  :or   {event {} render? true}}]
  (when render?
    (render-frame! app))
  (let [hiccup (last-hiccup app)]
    (if-let [element (find-in-hiccup app element-id hiccup)]
      (let [attrs (domh/hiccup-attrs element)
            handler (get attrs handler-key)]
        (cond
          (nil? handler)
          (log/warn "Element" element-id "has no" handler-key "handler")

          (fn? handler)
          (handler event)

          :else
          (throw (ex-info (str "Handler " handler-key " is not a function")
                   {:element-id element-id
                    :handler-key handler-key
                    :handler    handler
                    :type       (type handler)}))))
      (throw (ex-info "Element not found"
               {:element-id element-id})))))

(defn type-into!
  "Simulate typing into an input element by invoking its :onChange handler.

   Finds the element by ID and invokes :onChange with a synthetic event
   containing the value in {:target {:value value}}.

   Example:
   ```clojure
   (type-into! app \"username-input\" \"john.doe\")
   ```"
  [app element-id value & {:keys [render?] :or {render? true}}]
  (invoke-handler! app element-id :onChange
    :event {:target {:value value}}
    :render? render?))

(defn submit-form!
  "Simulate form submission by invoking the form's :onSubmit handler.

   Finds the form element by ID and invokes :onSubmit.

   Example:
   ```clojure
   (submit-form! app \"login-form\")
   ```"
  [app form-id & {:keys [event render?]
                  :or   {event {:preventDefault (fn [])} render? true}}]
  (invoke-handler! app form-id :onSubmit
    :event event
    :render? render?))

(defn has-element?
  "Returns true if an element with the given ID exists in the current UI.

   Example:
   ```clojure
   (assertions
     (ct/has-element? app \"submit-btn\") => true)
   ```"
  [app element-id]
  (render-frame! app)
  (some? (find-in-hiccup app element-id)))

(defn text-of
  "Returns the text content of the element with the given ID.
   Returns nil if the element is not found.

   Example:
   ```clojure
   (assertions
     (ct/text-of app \"counter\") => \"42\")
   ```"
  [app element-id]
  (render-frame! app)
  (when-let [element (find-in-hiccup app element-id)]
    (element-text element)))

(defn attr-of
  "Returns the value of an attribute on the element with the given ID.
   Returns nil if the element is not found or the attribute is not present.

   Example:
   ```clojure
   (assertions
     (ct/attr-of app \"email-input\" :type) => \"email\"
     (ct/attr-of app \"checkbox\" :checked) => true)
   ```"
  [app element-id attr-key]
  (render-frame! app)
  (when-let [element (find-in-hiccup app element-id)]
    (element-attr element attr-key)))

(defn classes-of
  "Returns a set of CSS class names on the element with the given ID.
   Returns nil if the element is not found.

   Example:
   ```clojure
   (assertions
     (ct/classes-of app \"my-btn\") => #{\"btn\" \"btn-primary\"})
   ```"
  [app element-id]
  (render-frame! app)
  (when-let [element (find-in-hiccup app element-id)]
    (let [attrs (domh/hiccup-attrs element)
          class-str (or (:className attrs) (:class attrs) "")]
      (if (str/blank? class-str)
        #{}
        (set (str/split class-str #"\s+"))))))

(defn has-class?
  "Returns true if the element has the given CSS class.
   Returns false if the element is not found.

   Example:
   ```clojure
   (assertions
     (ct/has-class? app \"my-btn\" \"active\") => true)
   ```"
  [app element-id class-name]
  (contains? (classes-of app element-id) class-name))
