(ns com.fulcrologic.fulcro.headless.hiccup
  "Hiccup conversion and utilities for headless testing.

   This namespace provides:
   - Conversion from dom-server Element trees to hiccup format
   - Utility functions for inspecting and traversing hiccup trees
   - Functions operate on hiccup trees directly (not apps)

   Hiccup format: [:tag {:attr \"value\"} child1 child2 ...]

   Example workflow:
   ```clojure
   (require '[com.fulcrologic.fulcro.headless :as h])
   (require '[com.fulcrologic.fulcro.headless.hiccup :as hic])

   ;; Get the rendered tree from a frame
   (let [frame (h/last-frame app)
         hiccup (hic/rendered-tree->hiccup (:rendered frame))]
     ;; Inspect and interact with the hiccup tree
     (hic/find-by-id hiccup \"my-button\")
     (hic/click! (hic/find-by-id hiccup \"submit-btn\")))
   ```"
  (:require
    [clojure.string :as str]
    [com.fulcrologic.fulcro.dom-server :as dom]
    [com.fulcrologic.fulcro.raw.components :as rc]))

;; =============================================================================
;; Hiccup Conversion from dom-server Elements
;; =============================================================================

(defprotocol IHiccupConvertible
  "Protocol for converting dom-server elements to hiccup format."
  (to-hiccup* [this] "Convert this element to hiccup."))

(extend-protocol IHiccupConvertible
  com.fulcrologic.fulcro.dom_server.Element
  (to-hiccup* [{:keys [tag attrs children]}]
    (let [converted-children (reduce
                               (fn [acc child]
                                 (let [h (to-hiccup* child)]
                                   (if (nil? h)
                                     acc
                                     (if (and (vector? h) (vector? (first h)))
                                       ;; Fragment - flatten into parent
                                       (into acc h)
                                       (conj acc h)))))
                               []
                               children)]
      (into [(keyword tag) (or attrs {})] converted-children)))

  com.fulcrologic.fulcro.dom_server.Text
  (to-hiccup* [{:keys [s]}] s)

  com.fulcrologic.fulcro.dom_server.ReactText
  (to-hiccup* [{:keys [text]}] text)

  com.fulcrologic.fulcro.dom_server.ReactEmpty
  (to-hiccup* [_] nil)

  com.fulcrologic.fulcro.dom_server.ReactFragment
  (to-hiccup* [{:keys [elements]}]
    (let [converted (keep to-hiccup* elements)]
      (vec converted)))

  nil
  (to-hiccup* [_] nil))

(defn- render-component [c]
  "Render a component instance to an Element tree."
  (if (or (nil? c) (dom/element? c))
    c
    (when-let [render (rc/component-options c :render)]
      (when-let [output (render c)]
        (if (vector? output)
          (mapv render-component output)
          (recur output))))))

(defn rendered-tree->hiccup
  "Convert a dom-server Element tree to hiccup format.
   Preserves all attributes including function handlers (unlike render-to-str
   which strips them for HTML output).

   This is the primary conversion function for headless testing where you want
   to inspect the DOM structure and invoke event handlers.

   Accepts:
   - A dom-server Element record
   - A Fulcro component instance
   - A vector of the above
   - nil

   Returns:
   - A hiccup vector [:tag {...attrs} ...children]
   - Or a vector of hiccup vectors for fragments
   - Or nil for empty elements

   Example:
   ```clojure
   (rendered-tree->hiccup (doms/div {:id \"test\" :onClick my-handler} \"Hello\"))
   ;; => [:div {:id \"test\" :onClick #function} \"Hello\"]
   ```"
  [x]
  (cond
    (nil? x) nil

    (dom/element? x)
    (to-hiccup* x)

    (rc/component-instance? x)
    (let [rendered (render-component x)]
      (cond
        (nil? rendered) nil
        (dom/element? rendered) (to-hiccup* rendered)
        (vector? rendered) (mapv to-hiccup* rendered)
        :else rendered))

    (vector? x)
    (mapv rendered-tree->hiccup x)

    :else
    (throw (IllegalArgumentException.
             (str "Cannot convert to hiccup: " (type x))))))

;; =============================================================================
;; Element Predicates and Accessors
;; =============================================================================

(defn element?
  "Returns true if x is a hiccup element (a vector with a keyword tag).
   Text nodes (strings) and nil are not elements.

   Example:
   ```clojure
   (element? [:div {} \"text\"]) => true
   (element? \"just text\") => false
   ```"
  [x]
  (and (vector? x)
    (keyword? (first x))))

(defn element-tag
  "Get the tag keyword from an element.

   Example:
   ```clojure
   (element-tag [:div {:id \"x\"} \"text\"]) => :div
   ```"
  [elem]
  (when (element? elem)
    (first elem)))

(defn element-attrs
  "Get the attributes map from an element.
   Returns nil if not a valid element or has no attrs map.

   Example:
   ```clojure
   (element-attrs [:div {:id \"x\" :className \"foo\"} \"text\"])
   => {:id \"x\" :className \"foo\"}
   ```"
  [elem]
  (when (element? elem)
    (let [second-item (second elem)]
      (when (map? second-item)
        second-item))))

(defn element-children
  "Get the children of an element (everything after tag and attrs map).

   Example:
   ```clojure
   (element-children [:div {} \"Hello\" [:span {} \"World\"]])
   => [\"Hello\" [:span {} \"World\"]]
   ```"
  [elem]
  (when (element? elem)
    (let [second-item (second elem)]
      (if (map? second-item)
        (subvec elem 2)
        (subvec elem 1)))))

(defn element-attr
  "Get an attribute value from an element.

   Example:
   ```clojure
   (element-attr [:input {:type \"email\" :value \"x@y.com\"}] :type)
   => \"email\"
   ```"
  [elem attr-key]
  (get (element-attrs elem) attr-key))

(defn element-text
  "Get the text content of an element, recursively extracting all text.
   Returns a string with all text content concatenated.

   Example:
   ```clojure
   (element-text [:div {} \"Hello \" [:span {} \"World\"]])
   => \"Hello World\"
   ```"
  [elem]
  (cond
    (nil? elem) ""
    (string? elem) elem
    (number? elem) (str elem)
    (element? elem)
    (apply str (map element-text (element-children elem)))
    (sequential? elem)
    (apply str (map element-text elem))
    :else ""))

(defn element-classes
  "Get the CSS classes of an element as a set.
   Returns an empty set if no classes.

   Example:
   ```clojure
   (element-classes [:div {:className \"btn btn-primary\"}])
   => #{\"btn\" \"btn-primary\"}
   ```"
  [elem]
  (let [attrs     (element-attrs elem)
        class-str (or (:className attrs) (:class attrs) "")]
    (if (str/blank? class-str)
      #{}
      (set (str/split class-str #"\s+")))))

(defn has-class?
  "Returns true if the element has the given CSS class.

   Example:
   ```clojure
   (has-class? [:div {:className \"btn active\"}] \"active\") => true
   ```"
  [elem class-name]
  (contains? (element-classes elem) class-name))

;; =============================================================================
;; Element Finding
;; =============================================================================

(defn find-by-id
  "Find an element in the tree by its :id attribute.
   Returns the first matching element or nil.

   Example:
   ```clojure
   (find-by-id tree \"submit-btn\")
   => [:button {:id \"submit-btn\" :onClick #fn} \"Submit\"]
   ```"
  [tree id]
  (cond
    (nil? tree) nil

    (element? tree)
    (let [attrs (element-attrs tree)]
      (if (= id (:id attrs))
        tree
        (some #(find-by-id % id) (element-children tree))))

    (sequential? tree)
    (some #(find-by-id % id) tree)

    :else nil))

(defn find-all
  "Find all elements in the tree matching the predicate.
   The predicate receives each element (hiccup vector).
   Returns a vector of matching elements.

   Example:
   ```clojure
   (find-all tree #(= :button (element-tag %)))
   => [[:button {...} \"Click\"] [:button {...} \"Cancel\"]]
   ```"
  [tree pred]
  (cond
    (nil? tree) []

    (element? tree)
    (let [matches       (if (pred tree) [tree] [])
          child-matches (mapcat #(find-all % pred) (element-children tree))]
      (into matches child-matches))

    (sequential? tree)
    (vec (mapcat #(find-all % pred) tree))

    :else []))

(defn find-by-tag
  "Find all elements with the given tag keyword.

   Example:
   ```clojure
   (find-by-tag tree :input)
   => [[:input {:type \"text\"...}] [:input {:type \"email\"...}]]
   ```"
  [tree tag]
  (find-all tree #(= tag (element-tag %))))

(defn find-by-class
  "Find all elements with the given CSS class.

   Example:
   ```clojure
   (find-by-class tree \"btn-primary\")
   => [[:button {:className \"btn btn-primary\"} \"Submit\"]]
   ```"
  [tree class-name]
  (find-all tree #(has-class? % class-name)))

(defn find-first
  "Find the first element matching the predicate, or nil.

   Example:
   ```clojure
   (find-first tree #(= :form (element-tag %)))
   => [:form {:onSubmit #fn} ...]
   ```"
  [tree pred]
  (first (find-all tree pred)))

;; =============================================================================
;; Handler Invocation
;; =============================================================================

(defn click!
  "Invoke the :onClick handler of an element.
   The element should have been found via find-by-id or similar.

   The handler will have already closed over the app when the component rendered,
   so no app parameter is needed.

   Options:
   - :event - The synthetic event to pass (default: {})

   Returns the result of the handler, or nil if no handler.

   Example:
   ```clojure
   (click! (find-by-id tree \"submit-btn\"))
   (click! (find-by-id tree \"checkbox\") {:target {:checked true}})
   ```"
  ([elem] (click! elem {}))
  ([elem event]
   (when elem
     (when-let [on-click (element-attr elem :onClick)]
       (when (fn? on-click)
         (on-click event))))))

(defn invoke-handler!
  "Invoke a specific handler on an element.
   The handler key can be any attribute (e.g., :onClick, :onChange, :onSubmit).

   Returns the result of the handler, or nil if no handler.

   Any additional arguments are applied to the handler, so you can match the arity of the target.

   Example:
   ```clojure
   (invoke-handler! (find-by-id tree \"email\") :onChange {:target {:value \"x@y.com\"}})
   ```"
  [elem handler-key & args]
  (when elem
    (when-let [handler (element-attr elem handler-key)]
      (when (fn? handler)
        (apply handler args)))))

(defn type-text!
  "Invoke the :onChange handler with a text value.
   Convenience for simulating typing into an input.

   Example:
   ```clojure
   (type-text! (find-by-id tree \"username\") \"john.doe\")
   ```"
  [elem value]
  (invoke-handler! elem :onChange {:target {:value value}}))

;; =============================================================================
;; Assertions and Queries
;; =============================================================================

(defn element-exists?
  "Returns true if an element with the given ID exists in the tree.

   Example:
   ```clojure
   (exists? tree \"submit-btn\") => true
   ```"
  [tree id]
  (some? (find-by-id tree id)))

(defn text-of
  "Get the text content of the element with the given ID.
   Returns nil if not found.

   Example:
   ```clojure
   (text-of tree \"counter\") => \"42\"
   ```"
  [tree id]
  (when-let [elem (find-by-id tree id)]
    (element-text elem)))

(defn attr-of
  "Get an attribute value from the element with the given ID.
   Returns nil if element not found or attribute not present.

   Example:
   ```clojure
   (attr-of tree \"email-input\" :type) => \"email\"
   ```"
  [tree id attr-key]
  (when-let [elem (find-by-id tree id)]
    (element-attr elem attr-key)))

(defn classes-of
  "Get the CSS classes of the element with the given ID as a set.
   Returns nil if element not found.

   Example:
   ```clojure
   (classes-of tree \"my-btn\") => #{\"btn\" \"btn-primary\"}
   ```"
  [tree id]
  (when-let [elem (find-by-id tree id)]
    (element-classes elem)))
