(ns com.fulcrologic.fulcro.headless
  "Headless Fulcro application support for testing and server-side execution.

   This namespace provides utilities for running Fulcro applications in headless mode
   on the JVM. Key features:

   - Synchronous transaction processing (all operations complete before returning)
   - Render frame capture (inspect before/after state for assertions)
   - Raw dom-server Element tree capture (convert to hiccup or other formats as needed)
   - Controlled execution (step through transaction phases)
   - Integration with Ring handlers and Pathom parsers via loopback remotes

   Example:
   ```clojure
   (require '[com.fulcrologic.fulcro.headless :as h])
   (require '[com.fulcrologic.fulcro.headless.hiccup :as hic])
   (require '[com.fulcrologic.fulcro.headless.loopback-remotes :as lr])

   (def app (h/build-test-app
              {:root-class Root
               :remotes {:remote (lr/sync-remote my-handler)}}))

   ;; Transactions are synchronous
   (comp/transact! app [(my-mutation)])

   ;; Inspect render frames
   (let [frame (h/last-frame app)
         hiccup (hic/rendered-tree->hiccup (:rendered frame))]
     (is (= expected (:tree frame)))
     (hic/click! (hic/find-by-id hiccup \"my-button\")))
   ```"
  (:require
    [com.fulcrologic.fulcro.algorithms.denormalize :as fdn]
    [com.fulcrologic.fulcro.algorithms.lookup :as ah]
    [com.fulcrologic.fulcro.algorithms.normalized-state :as fnorm]
    [com.fulcrologic.fulcro.algorithms.tx-processing.synchronous-tx-processing :as stx]
    [com.fulcrologic.fulcro.components :as comp]
    [com.fulcrologic.fulcro.headless.hiccup :as hic]
    [com.fulcrologic.fulcro.raw.application :as rapp]
    [com.fulcrologic.fulcro.raw.components :as rc]
    [taoensso.timbre :as log]))

(def ^:dynamic *render-frame-max*
  "Default maximum number of render frames to keep in history."
  10)

(defn- render-app-tree
  "Render the app's root component to a dom-server Element tree.
   Binds the proper dynamic vars for component rendering."
  [app]
  (let [{:com.fulcrologic.fulcro.application/keys [state-atom runtime-atom]} app
        state-map @state-atom
        {:com.fulcrologic.fulcro.application/keys [root-class root-factory]} @runtime-atom
        factory   (or root-factory
                    (when root-class
                      (comp/factory root-class)))]
    (when (and root-class factory)
      (let [query (comp/get-query root-class state-map)
            tree  (fdn/db->tree query state-map state-map)]
        (binding [comp/*app*    app
                  comp/*parent* nil
                  comp/*shared* (comp/shared app)]
          (factory tree))))))

(defn- capture-frame
  "Capture a render frame from the current app state.
   The frame contains:
   - :state - The normalized state map at render time
   - :tree - The denormalized props tree
   - :rendered - The raw dom-server Element tree (convert to hiccup with headless.hiccup/rendered-tree->hiccup)
   - :timestamp - System time in milliseconds"
  [app]
  (let [{:com.fulcrologic.fulcro.application/keys [state-atom runtime-atom]} app
        state-map @state-atom
        {:com.fulcrologic.fulcro.application/keys [root-class]} @runtime-atom
        query     (when root-class (comp/get-query root-class state-map))
        tree      (when query (fdn/db->tree query state-map state-map))
        rendered  (try
                    (render-app-tree app)
                    (catch Exception e
                      (log/trace e "Could not render tree (root component may not have render fn)")
                      nil))]
    {:state     state-map
     :tree      tree
     :rendered  rendered
     :timestamp (System/currentTimeMillis)}))

(defn- add-frame!
  "Add a frame to the render history, maintaining max size."
  [app frame]
  (let [runtime-atom (:com.fulcrologic.fulcro.application/runtime-atom app)
        max-frames   (or (::render-frame-max @runtime-atom) *render-frame-max*)]
    (swap! runtime-atom update ::render-frames
      (fn [frames]
        (let [new-frames (vec (cons frame (or frames [])))]
          (if (> (count new-frames) max-frames)
            (subvec new-frames 0 max-frames)
            new-frames))))))

(defn- frame-capturing-render
  "A render function that captures frames to history.
   This replaces the optimized-render! algorithm."
  [app options]
  (let [frame (capture-frame app)]
    (add-frame! app frame)
    frame))

;; =============================================================================
;; Render History
;; =============================================================================

(defn render-history
  "Get the render history (newest first).
   Each entry is a map with:
   - :state - The state map at render time
   - :tree - The denormalized props tree
   - :rendered - The raw dom-server Element tree
   - :timestamp - System time in milliseconds"
  [app]
  (-> app :com.fulcrologic.fulcro.application/runtime-atom deref ::render-frames))

(defn last-frame
  "Get the most recent render frame, or nil if none."
  [app]
  (first (render-history app)))

(defn frame-at
  "Get a specific render frame by index.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (nth (render-history app) frame-index nil))

(defn clear-render-history!
  "Clear all captured render frames."
  [app]
  (swap! (:com.fulcrologic.fulcro.application/runtime-atom app)
    assoc ::render-frames []))

(defn render-frame!
  "Force a render and capture the frame in history.
   Returns the captured frame."
  [app]
  (let [render! (ah/app-algorithm app :render!)]
    (when render!
      (render! app {:force-root? true}))
    (last-frame app)))

(defn frames-since
  "Get all frames captured since the given timestamp."
  [app timestamp]
  (take-while #(> (:timestamp %) timestamp) (render-history app)))

;; =============================================================================
;; Convenience Accessors for Frame Data
;; =============================================================================

(defn state-at-render
  "Get the state map from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (:state (frame-at app frame-index)))

(defn tree-at-render
  "Get the denormalized tree from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest)."
  [app frame-index]
  (:tree (frame-at app frame-index)))

(defn rendered-at
  "Get the raw dom-server Element tree from a specific render frame.
   `frame-index` is 0-indexed from most recent (0 = latest).
   Use headless.hiccup/rendered-tree->hiccup to convert to hiccup."
  [app frame-index]
  (:rendered (frame-at app frame-index)))

(defn hiccup-frame
  "Get the rendered output as hiccup from a render frame.
   Convenience function that converts the dom-server Element tree to hiccup.

   With 1 arg: Returns hiccup from the most recent frame.
   With 2 args: Returns hiccup from the nth most recent frame (0 = latest).

   Example:
   ```clojure
   ;; Get most recent render as hiccup
   (hiccup-frame app)

   ;; Get the render from 2 frames ago
   (hiccup-frame app 2)
   ```"
  ([app]
   (hiccup-frame app 0))
  ([app n-steps-ago]
   (when-let [rendered (rendered-at app n-steps-ago)]
     (hic/rendered-tree->hiccup rendered))))

;; =============================================================================
;; Props Access
;; =============================================================================

(defn current-props
  "Get denormalized props for a component at the given ident.
   Useful for inspecting component state in tests."
  [app component-class ident]
  (fnorm/ui->props (rapp/current-state app) component-class ident))

(defn hiccup-for
  "Return the hiccup for the given component based on the `app` current state. You MUST supply
   the ident unless the component has a constant ident (or is Root)."
  ([app component-class]
   (hiccup-for app component-class (rc/get-ident component-class {})))
  ([app component-class ident]
   (let [state-map (rapp/current-state app)
         query     (comp/get-query component-class state-map)
         factory   (comp/factory component-class)
         tree      (fdn/db->tree query (if ident
                                         (get-in state-map ident)
                                         state-map) state-map)]
     (hic/rendered-tree->hiccup
       (binding [comp/*app*    app
                 comp/*parent* nil
                 comp/*shared* (comp/shared app)]
         (factory tree))))))

;; =============================================================================
;; App Building
;; =============================================================================

(defn build-test-app
  "Create a test application configured for synchronous testing.

   Options:
   - :root-class - Root component class (required for render frame capture)
   - :remotes - Map of remote-name to remote implementation
   - :initial-state - Initial state map (merged with root's initial state)
   - :render-history-size - Number of frames to keep (default 10)
   - :shared - Static shared props
   - :shared-fn - Function to compute shared props from root tree

   Returns a Fulcro app configured with:
   - Synchronous transaction processing
   - Frame-capturing render
   - The specified remotes"
  [{:keys [root-class remotes initial-state render-history-size shared shared-fn]
    :or   {render-history-size *render-frame-max*
           remotes             {}}}]
  (let [base-app (rapp/fulcro-app
                   {:initial-db        (or initial-state {})
                    :root-class        root-class
                    :remotes           (if (empty? remotes)
                                         {:remote {:transmit! (fn [{:com.fulcrologic.fulcro.algorithms.tx-processing/keys [result-handler]}]
                                                                (log/warn "No remote configured, returning empty response")
                                                                (result-handler {:status-code 200 :body {}}))}}
                                         remotes)
                    :shared            shared
                    :shared-fn         shared-fn
                    :optimized-render! frame-capturing-render
                    :core-render!      (fn [app options]
                                         (let [optimized-render! (ah/app-algorithm app :optimized-render!)]
                                           (binding [comp/*app*    app
                                                     comp/*parent* nil
                                                     comp/*shared* (comp/shared app)]
                                             (optimized-render! app options))))})
        app      (stx/with-synchronous-transactions base-app)]
    (when root-class
      (rapp/initialize-state! app root-class))
    (swap! (:com.fulcrologic.fulcro.application/runtime-atom app)
      assoc ::render-frame-max render-history-size
      ::render-frames [])
    app))

(defn set-remote!
  "Set or replace a remote on the test app.
   Convenience wrapper around rapp/set-remote!"
  [app remote-name remote]
  (rapp/set-remote! app remote-name remote))

;; =============================================================================
;; Transaction Processing
;; =============================================================================

(defn wait-for-idle!
  "Block until all pending work is complete.
   With synchronous tx processing, this is usually instant,
   but useful after operations that might queue follow-on work."
  [app]
  (loop [iterations 0]
    (when (< iterations 100)                                ; Safety limit
      (let [submission-queue (stx/submission-queue app)
            active-queue     (stx/active-queue app)]
        (when (or (seq submission-queue) (seq active-queue))
          (Thread/sleep 1)
          (recur (inc iterations)))))))

(defn pending-sends
  "Get the pending send queue for a remote.
   Useful for verifying what would be sent to the server."
  [app remote-name]
  (stx/send-queue app remote-name))

(defn has-pending-work?
  "Returns true if there is any pending transaction work."
  [app]
  (or (stx/available-work? app)
    (seq (stx/active-queue app))
    (stx/post-processing? app)))

;; =============================================================================
;; Test Utilities
;; =============================================================================

(defn reset-app!
  "Reset the app state to initial state from root component.
   Useful for test isolation."
  [app]
  (let [runtime-atom (:com.fulcrologic.fulcro.application/runtime-atom app)
        {:com.fulcrologic.fulcro.application/keys [root-class]} @runtime-atom]
    (when root-class
      (reset! (:com.fulcrologic.fulcro.application/state-atom app) {})
      (rapp/initialize-state! app root-class)
      (clear-render-history! app))))

(defn with-render-tracking
  "Execute body and return a map with:
   - :result - The return value of body
   - :frames - All frames captured during execution
   - :frame-count - Number of frames captured

   Example:
   ```clojure
   (let [{:keys [result frames]} (with-render-tracking app
                                   (comp/transact! app [(my-mutation)]))]
     (is (= 1 (count frames))))
   ```"
  [app body-fn]
  (let [start-count     (count (render-history app))
        result          (body-fn)
        end-count       (count (render-history app))
        new-frame-count (- end-count start-count)]
    {:result      result
     :frames      (take new-frame-count (render-history app))
     :frame-count new-frame-count}))

