(ns com.fulcrologic.fulcro.dom-hiccup
  "Server-side DOM rendering to hiccup data structures.

   Unlike dom-server which renders to HTML strings, this namespace
   produces hiccup vectors with real function values preserved.
   This enables testing of event handlers and DOM inspection.

   Usage: Create your UI in CLJC files, and require with conditional reader tags:

   (ns app.ui
     (:require
       #?(:clj [com.fulcrologic.fulcro.dom-hiccup :as dom] :cljs [com.fulcrologic.fulcro.dom :as dom])))

   The output is standard Clojure hiccup:
   [:div {:id \"my-div\" :onClick <actual-fn>}
     [:span {} \"Hello\"]
     [:button {:onClick <actual-fn>} \"Click me\"]]"
  (:refer-clojure :exclude [filter map mask meta select set symbol time use])
  (:require
    [clojure.spec.alpha :as s]
    [com.fulcrologic.fulcro.algorithms.do-not-use :as util]
    [com.fulcrologic.fulcro.dom-common :as cdom]
    [com.fulcrologic.fulcro.raw.components :as rc :refer [component-instance?]]
    [taoensso.timbre :as log]))

(declare a abbr address altGlyph altGlyphDef altGlyphItem animate animateColor animateMotion animateTransform area
  article aside audio b base bdi bdo big blockquote body br button canvas caption circle cite clipPath code
  col colgroup color-profile cursor data datalist dd defs del desc details dfn dialog discard div dl dt
  ellipse em embed feBlend feColorMatrix feComponentTransfer feComposite feConvolveMatrix feDiffuseLighting
  feDisplacementMap feDistantLight feDropShadow feFlood feFuncA feFuncB feFuncG feFuncR feGaussianBlur
  feImage feMerge feMergeNode feMorphology feOffset fePointLight feSpecularLighting feSpotLight feTile feTurbulence
  fieldset figcaption figure filter font font-face font-face-format font-face-name font-face-src font-face-uri
  footer foreignObject form g glyph glyphRef h1 h2 h3 h4 h5 h6 hatch hatchpath head header hkern hr html
  i iframe image img input ins kbd keygen label legend li line linearGradient link main map mark marker mask
  menu menuitem mesh meshgradient meshpatch meshrow meta metadata meter missing-glyph
  mpath nav noscript object ol optgroup option output p param path pattern picture polygon polyline pre progress q radialGradient
  rect rp rt ruby s samp script section select set small solidcolor source span stop strong style sub summary
  sup svg switch symbol table tbody td text textPath textarea tfoot th thead time title tr track tref tspan
  u ul unknown use var video view vkern wbr)

;; =============================================================================
;; Hiccup Element Protocol
;; =============================================================================

(defprotocol IHiccupElement
  "Protocol for elements that can be rendered to hiccup."
  (render-to-hiccup* [this]
    "Render this element to hiccup vector(s)."))

(declare render-element)

;; =============================================================================
;; Element Records
;; =============================================================================

(defrecord HiccupElement [tag attrs react-key children]
  IHiccupElement
  (render-to-hiccup* [this]
    (render-element this)))

(defrecord HiccupText [text]
  IHiccupElement
  (render-to-hiccup* [this]
    text))

(defrecord HiccupFragment [elements]
  IHiccupElement
  (render-to-hiccup* [this]
    (mapv render-to-hiccup* (:elements this))))

(defn hiccup-element?
  "Returns true if x is a hiccup element."
  [x]
  (satisfies? IHiccupElement x))

;; =============================================================================
;; Spec for DOM element args (same as dom-server)
;; =============================================================================

(s/def ::dom-element-args
  (s/cat
    :css (s/? keyword?)
    :attrs (s/? (s/or
                  :nil nil?
                  :map #(and (map? %)
                          (not (component-instance? %))
                          (not (hiccup-element? %)))))
    :children (s/* (s/or
                     :nil nil?
                     :string string?
                     :number number?
                     :collection #(or (vector? %) (seq? %))
                     :component component-instance?
                     :element hiccup-element?))))

;; =============================================================================
;; Component Rendering
;; =============================================================================

(defn- render-component
  "Render a Fulcro component to hiccup element(s)."
  [c]
  (if (or (nil? c) (hiccup-element? c))
    c
    (when-let [render (rc/component-options c :render)]
      (when-let [output (render c)]
        (if (vector? output)
          (mapv render-component output)
          (recur output))))))

;; =============================================================================
;; Element Creation and Rendering
;; =============================================================================

(defn element
  "Create a hiccup DOM element record."
  [{:keys [tag attrs react-key children] :as elem}]
  (assert (name tag) (str "Invalid tag in element: " elem))
  (assert (or (nil? attrs) (map? attrs)) (str "Invalid attrs in element: " elem))
  (let [children (flatten children)
        processed-children
        (reduce
          (fn [res c]
            (let [c' (cond
                       ;; Already a hiccup element
                       (hiccup-element? c) c

                       ;; Fulcro component instance - render it
                       (component-instance? c)
                       (let [rendered (render-component c)]
                         (if (vector? rendered)
                           (->HiccupFragment rendered)
                           rendered))

                       ;; String or number - wrap as text
                       (or (string? c) (number? c))
                       (->HiccupText (str c))

                       ;; nil - skip
                       (nil? c) nil

                       :else
                       (throw (IllegalArgumentException.
                                (str "Invalid child element: " (pr-str c)))))]
              (cond-> res
                (some? c') (conj c'))))
          []
          children)]
    (->HiccupElement (name tag) attrs react-key processed-children)))

(defn render-element
  "Render a HiccupElement to hiccup vector."
  [{:keys [tag attrs children]}]
  (let [rendered-children (mapv render-to-hiccup* children)
        ;; Flatten any fragments
        flattened-children (reduce
                             (fn [acc child]
                               (if (and (vector? child)
                                        (vector? (first child)))
                                 ;; This is a fragment (vector of vectors)
                                 (into acc child)
                                 (conj acc child)))
                             []
                             rendered-children)]
    (into [(keyword tag) (or attrs {})] flattened-children)))

;; =============================================================================
;; Top-level Rendering
;; =============================================================================

(defn render-to-hiccup
  "Render a Fulcro component or hiccup element to hiccup data structure.

   Accepts:
   - A Fulcro component instance
   - A HiccupElement record
   - A vector of the above

   Returns:
   - A hiccup vector [:tag {...attrs} ...children]
   - Or a vector of hiccup vectors for fragments"
  [x]
  (cond
    (nil? x) nil

    (hiccup-element? x)
    (render-to-hiccup* x)

    (component-instance? x)
    (let [rendered (render-component x)]
      (cond
        (nil? rendered) nil
        (hiccup-element? rendered) (render-to-hiccup* rendered)
        (vector? rendered) (mapv render-to-hiccup* rendered)
        :else rendered))

    (vector? x)
    (mapv render-to-hiccup x)

    :else
    (throw (IllegalArgumentException.
             (str "Cannot render to hiccup: " (type x))))))

;; =============================================================================
;; Tag Function Generation
;; =============================================================================

(defn gen-tag-fn
  "Generate a tag function that creates HiccupElement records."
  [tag]
  `(defn ~tag ~(cdom/gen-docstring tag false)
     [& ~'args]
     (try
       (let [conformed-args# (util/conform! ::dom-element-args ~'args)
             {attrs#    :attrs
              children# :children
              css#      :css} conformed-args#
             children#       (mapv second children#)
             attrs-value#    (or (second attrs#) {})]
         (element {:tag       '~tag
                   :attrs     (-> (cdom/interpret-classes attrs-value#)
                                (dissoc :ref :key)
                                (cdom/add-kwprops-to-props css#))
                   :react-key (:key attrs-value#)
                   :children  children#}))
       (catch Exception e#
         (log/error e# "Error creating hiccup element" '~tag ~'args)
         nil))))

(defmacro gen-all-tags
  "Generate all HTML/SVG tag functions."
  []
  (when-not (boolean (:ns &env))
    `(do
       ~@(clojure.core/map gen-tag-fn cdom/tags))))

(gen-all-tags)

;; =============================================================================
;; Utility Functions
;; =============================================================================

(defn create-element
  "Create a DOM element for which there exists no corresponding function.
   Useful to create DOM elements not included in the standard set."
  ([tag]
   (create-element tag nil))
  ([tag opts & children]
   (element {:tag       tag
             :attrs     (dissoc opts :ref :key)
             :react-key (:key opts)
             :children  children})))

(defn node
  "Returns the dom node associated with a component's React ref.
   This is a NO-OP function for hiccup rendering."
  ([component])
  ([component name]))

;; =============================================================================
;; Hiccup Tree Helpers
;; =============================================================================

(defn hiccup-element-node?
  "Returns true if x is a valid hiccup element vector (not a text node)."
  [x]
  (and (vector? x)
       (keyword? (first x))))

(defn hiccup-attrs
  "Get the attributes map from a hiccup element.
   Returns nil if not a valid hiccup element or has no attrs."
  [elem]
  (when (hiccup-element-node? elem)
    (let [second-item (second elem)]
      (when (map? second-item)
        second-item))))

(defn hiccup-children
  "Get the children of a hiccup element (everything after tag and optional attrs)."
  [elem]
  (when (hiccup-element-node? elem)
    (let [second-item (second elem)]
      (if (map? second-item)
        (subvec elem 2)
        (subvec elem 1)))))

(defn hiccup-tag
  "Get the tag keyword from a hiccup element."
  [elem]
  (when (hiccup-element-node? elem)
    (first elem)))

(defn find-element-by-id
  "Find an element in the hiccup tree by its :id attribute.
   Returns the first matching element or nil."
  [hiccup id]
  (cond
    (nil? hiccup) nil

    (hiccup-element-node? hiccup)
    (let [attrs (hiccup-attrs hiccup)]
      (if (= id (:id attrs))
        hiccup
        ;; Search children
        (some #(find-element-by-id % id) (hiccup-children hiccup))))

    (sequential? hiccup)
    (some #(find-element-by-id % id) hiccup)

    :else nil))

(defn find-elements
  "Find all elements in the hiccup tree matching the predicate.
   The predicate receives each hiccup element vector.
   Returns a vector of matching elements."
  [hiccup pred]
  (cond
    (nil? hiccup) []

    (hiccup-element-node? hiccup)
    (let [matches (if (pred hiccup) [hiccup] [])
          child-matches (mapcat #(find-elements % pred) (hiccup-children hiccup))]
      (into matches child-matches))

    (sequential? hiccup)
    (vec (mapcat #(find-elements % pred) hiccup))

    :else []))

(defn find-elements-by-tag
  "Find all elements with the given tag keyword."
  [hiccup tag]
  (find-elements hiccup #(= tag (hiccup-tag %))))

(defn find-elements-by-class
  "Find all elements that have the given class in their :className or :class attribute."
  [hiccup class-name]
  (find-elements hiccup
    (fn [elem]
      (let [attrs (hiccup-attrs elem)
            class-str (or (:className attrs) (:class attrs) "")]
        (some #(= class-name %) (clojure.string/split class-str #"\s+"))))))

(defn element-text
  "Get the text content of a hiccup element (recursively extracts all text).
   Returns a string with all text content concatenated."
  [elem]
  (cond
    (nil? elem) ""
    (string? elem) elem
    (number? elem) (str elem)
    (hiccup-element-node? elem)
    (apply str (clojure.core/map element-text (hiccup-children elem)))
    (sequential? elem)
    (apply str (clojure.core/map element-text elem))
    :else ""))

(defn element-attr
  "Get an attribute value from a hiccup element."
  [elem attr-key]
  (get (hiccup-attrs elem) attr-key))
