(ns com.fulcrologic.fulcro.dynamic-join
  "Utility for creating a static component that can point to an arbitrary component class in the Fulcro registry.

   This allows you to have a component that can be composed into your application (via dynamic generation or
   dynamic code loading) on demand. It is nothing more that a stock component whose query can be changed
   according to registry keys instead of you having to try to figure it all out."
  #?(:cljs (:require-macros com.fulcrologic.fulcro.dynamic-join))
  (:require
    [com.fulcrologic.fulcro.components :as comp]
    [com.fulcrologic.fulcro.raw.components :as rc]
    [com.fulcrologic.fulcro.mutations :refer [defmutation]]
    [com.fulcrologic.fulcro.algorithms.merge :as merge]
    [edn-query-language.core :as eql]
    [taoensso.encore :as enc]
    [taoensso.timbre :as log]))

(defn initialize-target-state*
  "Mutation helper. You probably want `initialize-dynamic-join*` or `set-dynamic-target!` instead.

   Returns an updated state map with target (a registry key or class) initialized placed on DynamicJoinComponent"
  [state-map DynamicJoinComponent target {:keys [initialize? initial-state-params initial-state]}]
  (if (false? initialize?)
    state-map
    (enc/catching
      (let [TargetClass   (cond-> target
                            (not (comp/component-class? target)) (comp/registry-key->class))
            initial-state (or
                            initial-state
                            (comp/get-initial-state TargetClass (or initial-state-params {})))]
        (merge/merge-component state-map TargetClass initial-state :replace (conj (comp/get-ident DynamicJoinComponent {}) :component/child)))
      e
      (do
        (log/error e "Cannot initialize target state for dynamic join component" (comp/component-name DynamicJoinComponent))
        state-map))))

(defn initialize-dynamic-target*
  "Mutation helper. Initialize a dynamic join. See `set-dynamic-target!`.

   options can contain:

    * :initialize? - Merge the initial state of the target into app state. Default TRUE.
    * :initial-state-params - Parameters to use with `get-initial-state` on the target for initializing.
    * :initial-state - Override the component's idea of initial state and use this map instead.
   "
  [state-map DynamicJoinComponent target options]
  (if-let [c (cond-> target
               (not (comp/component-class? target)) (comp/registry-key->class))]
    (-> state-map
      (initialize-target-state* DynamicJoinComponent c options)
      (rc/set-query* DynamicJoinComponent {:query [{:component/child (comp/get-query c)}]}))
    (log/error "Unable to find target in Fulcro's registry. Is it loaded?" target)))

(defmutation initialize-dynamic-target
  "MUTATION. See `set-dynamic-target!` instead, or use `initialize-dynamic-target*` from within mutations.

   params MUST include:

   * ::component - The dynamic join component itself
   * ::target - The registry key or class to set as the join's target

   params can contain:

   * :initialize? - Merge the initial state of the target into app state. Default TRUE.
   * :initial-state-params - Parameters to use with `get-initial-state` on the target for initializing (default {}).
   * :initial-state - Override the component's idea of initial state and use this map instead (default nil).
  "
  [{:keys  [initialize? initial-state-params initial-state]
    ::keys [component target] :as params}]
  (action [{:keys [state]}]
    (swap! state initialize-dynamic-target* component target params)))

(defn set-dynamic-target!
  "Set DynamicJoinComponent so that it will render `target`. `target` can be a component class or registry key.
   `options` is a map with any of the following:

   * :initialize? - Merge the initial state of the target into app state. Default TRUE.
   * :initial-state-params - Parameters to use with `get-initial-state` on the target for initializing.
   * :initial-state - Override the component's idea of initial state and use this map instead.
   "
  [app-ish DynamicJoinComponent target options]
  (rc/transact! app-ish [(initialize-dynamic-target (assoc options
                                                      ::component DynamicJoinComponent
                                                      ::target target))]))

(defn dynamic-join
  "Create a placeholder component that can use the support of set-dynamic-target! to change its query
   dynamically at runtime to point to an instance of the component in the fulcro registry.

   `component-options` will be included in the join component's options, but cannot override query, or
   ident.

   The resulting component can be composed into the state UI tree, and will render nothing until `set-dynamic-target!`
   is called.

   Computed props sent to the dynamic join will relay THROUGH to the child target.

   Returns a React component class."
  [registry-key component-options]
  (let [fixed-ident      [::id registry-key]
        self             (volatile! nil)
        target-component (fn [this]
                           (let [ast (eql/query->ast (comp/get-query this))]
                             (some-> ast :children first :component)))
        result           (comp/sc registry-key
                           (merge
                             {:initial-state (fn [_] {})}
                             component-options
                             {:query (fn [_] [:component/child])
                              :ident (fn [] fixed-ident)})
                           (fn [this props]
                             (enc/when-let [ChildClass (log/spy :info (target-component this))
                                            child      (log/spy :info (get props :component/child))
                                            factory    (comp/computed-factory ChildClass)]
                               (factory child (comp/get-computed props)))))]
    (vreset! self result)
    result))

#?(:clj
   (defmacro defdynamic-join
     "[sym fulcro-component-options]

      Define a placeholder component that can use the support of set-dynamic-target! to change its query
      dynamically at runtime to point to (and render) an instance of some component in the fulcro registry.

      `fulcro-component-options` are additional options (besides query and ident)
      that the resulting generated component will have (e.g., routing options).

      The resulting component IS a Fulcro component.
     "
     ([sym component-options]
      (let [nspc         (if (boolean? (:ns &env)) (-> &env :ns :name str) (name (ns-name *ns*)))
            registry-key (keyword nspc (str sym))]
        `(defonce ~sym
           (dynamic-join ~registry-key ~component-options))))))


