(ns com.fulcrologic.fulcro-i18n.gettext
  "A set of functions for working with GNU gettext translation files, including translation generation tools to go
  from PO files to cljc.

  WARNING: This used to use dynamic vars and a global var with dynamic code loading. Those functions are no longer
  used, but are left to prevent library breakage. The current versions treat translations as pure data that can
  be loaded on demand from the server as data into app state instead."
  (:require [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.pprint :as pp]
            [clojure.java.shell :refer [sh]])
  (:import (java.io File)))

(defn- cljc-output-dir
  "Given a base source path (no trailing /) and a ns, returns the path to the directory that should contain it."
  [src-base ns]
  (let [path-from-ns (-> ns (str/replace #"\." "/") (str/replace #"-" "_"))]
    (str src-base "/" path-from-ns)))

(defn strip-comments
  "Given a sequence of strings that possibly contain comments of the form # ... <newline>: Return the sequence of
  strings without those comments."
  [string]
  (str/replace string #"(?m)^#[^\n\r]*(\r\n|\n|\r|$)" ""))

(defn is-header? [entry]
  (str/starts-with? entry "msgid \"\"\nmsgstr \"\"\n\"Project-Id-V"))

(defn get-blocks
  [resource]
  (filter (comp not is-header?) (remove str/blank? (map strip-comments (str/split (slurp resource) #"\n\n")))))

(defn stripquotes [s]
  (-> s
    (str/replace #"^\"" "")
    (str/replace #"\"$" "")))

(defn block->translation
  [gettext-block]
  (let [lines (str/split-lines gettext-block)]
    (-> (reduce (fn [{:keys [msgid msgctxt msgstr section] :as acc} line]
                  (let [[_ k v :as keyline] (re-matches #"^(msgid|msgctxt|msgstr)\s+\"(.*)\"\s*$" line)
                        unescaped-value (when v (str/replace v #"\\" ""))]
                    (cond
                      (and line (.matches line "^\".*\"$")) (update acc section #(str % (stripquotes line)))
                      (and k (#{"msgid" "msgctxt" "msgstr"} k)) (-> acc
                                                                  (assoc :section (keyword k))
                                                                  (update (keyword k) #(str % unescaped-value)))
                      :else (do
                              (println "Unexpected input -->" line "<--")
                              acc)))) {} lines)
      (dissoc :section))))

(defn- map-translations
  "Map translated strings to lookup keys.

  Parameters:
  * `resource` - A resource to read the po from.

  Returns a map of msgstr values to msgctxt|msgid string keys."
  [file-or-resource]
  (let [translations (map block->translation (get-blocks
                                               (if (string? file-or-resource)
                                                 (io/as-file file-or-resource)
                                                 file-or-resource)))]
    (reduce (fn [acc translation]
              (let [{:keys [msgctxt msgid msgstr] :or {msgctxt "" msgid "" msgstr ""}} translation
                    msg (if (and (-> msgstr .trim .isEmpty) (-> msgid .trim .isEmpty not))
                          (do
                            (println (str "WARNING: Message '" msgid "' is missing a translation! Using the default locale's message instead of an empty string."))
                            msgid)
                          msgstr)]
                (assoc acc (str msgctxt "|" msgid) msg)))
      {} translations)))

(defn- wrap-translations-in-ns
  "Wrap a translation map with supporting clojurescript code

  Parameters:
  * `locale` - the locale which this translation targets
  * `translation` - a clojurescript map of translations
  * `ns` - the ns in which to load the translations
  * `dynamic?` - The ns for this locale will be dynamically loaded as a module.

  Returns a string of clojurescript code."
  [& {:keys [locale translation ns dynamic?]}]
  (let [trans-ns    (str ns "." locale)
        locale-kw   (if (keyword? locale) locale (keyword locale))
        str-locale  (name locale-kw)
        ns-decl     (str "(ns " trans-ns " (:require fulcro.i18n #?(:cljs cljs.loader)))")
        comment     ";; This file was generated by Fulcro."
        trans-def   (pp/write (list 'def 'translations translation) :stream nil)
        swap-decl   (pp/write (list 'swap! 'fulcro.i18n/*loaded-translations* 'assoc str-locale 'translations) :stream nil)
        loaded-decl (str "#?(:cljs (cljs.loader/set-loaded! " locale-kw "))")]
    (str/join "\n\n" (keep identity [ns-decl comment trans-def swap-decl (when dynamic? loaded-decl)]))))

(defn- po-path [{:keys [podir]} po-file] (.getAbsolutePath (new File podir po-file)))

(defn- find-po-files
  "Finds any existing po-files, and adds them to settings. Returns the new settings."
  [{:keys [podir] :as settings}]
  (assoc settings :existing-po-files (filter #(.endsWith % ".po") (str/split-lines (:out (sh "ls" (.getAbsolutePath podir)))))))

(defn- js-missing?
  "Checks for compiled js. Returns settings if all OK, nil otherwise."
  [{:keys [js-path] :as settings}]
  (if (.exists (io/as-file js-path))
    settings
    (do
      (println js-path "is missing. Did you compile?"))))

(defn- gettext-missing?
  "Checks for gettext. Returns settings if all OK, nil otherwise."
  [settings]
  (let [xgettext (:exit (sh "which" "xgettext"))
        msgmerge (:exit (sh "which" "msgmerge"))]
    (if (or (not= xgettext 0) (not= msgmerge 0))
      (do
        (println "Count not find xgettext or msgmerge on PATH")
        nil)
      settings)))

(defn- run
  "Run a shell command and logging the command and result."
  [& args]
  (println "Running: " (str/join " " args))
  (let [result (:exit (apply sh args))]
    (when (not= 0 result)
      (print "Command Failed: " (str/join " " args))
      (println result))))

(defn- clojure-ize-locale [po-filename]
  (-> po-filename
    (str/replace #"^([a-z]+_*[A-Z]*).po$" "$1")
    (str/replace #"_" "-")))

(defn- expand-settings
  "Adds defaults and some additional helpful config items"
  [{:keys [src ns po] :as settings}]
  (let [srcdir      ^File (some-> src (io/as-file))
        output-path (some-> src (cljc-output-dir ns))
        outdir      ^File (some-> output-path (io/as-file))
        podir       ^File (some-> po (io/as-file))]
    (merge settings
      {:messages-pot (some-> podir (File. "messages.pot") (.getAbsolutePath))
       :podir        podir
       :outdir       outdir
       :srcdir       srcdir
       :output-path  output-path})))

(defn- verify-source-folders
  "Verifies that the source folder (target of the translation cljc) and ..."
  [{:keys [^File srcdir ^File outdir] :as settings}]
  (cond
    (not (.exists srcdir)) (do
                             (println "The given source-folder does not exist")
                             nil)
    (not (.exists outdir)) (do (println "Making missing source folder " (.getAbsolutePath outdir))
                               (.mkdirs outdir)
                               settings)
    :else settings))

(defn- verify-po-folders
  "Verifies that po files can be generated. Returns settings if so, nil otherwise."
  [{:keys [^File podir] :as settings}]
  (cond
    (not (.exists podir)) (do
                            (println "Creating missing PO directory: " (.getAbsolutePath podir))
                            (.mkdirs podir)
                            settings)
    (not (.isDirectory podir)) (do
                                 (println "po-folder must be a directory.")
                                 nil)
    :else settings))

(defn extract-strings
  "Extract strings from a compiled js file (whitespace optimized) as a PO template. If existing translations exist
  then this function will auto-update those (using `msgmerge`) as well.

  Remember that you must first compile your application (*without* modules) and with `:whitespace` optimization to generate
  a single javascript file. The gettext tools know how to extract from Javascript, but not Clojurescript.

  Parameters:
  `:js-path` - The path to your generated javascript (defaults to `i18n/i18n.js` from project root)
  `:po` - The directory where your PO template and PO files should live (defaults to `i18n` from project root). "
  [{:keys [js-path po no-location? sort-output?] :or {js-path "i18n/i18n.js" po "i18n"}}]
  (when-let [{:keys [existing-po-files messages-pot]
              :as   settings} (some-> {:js-path js-path :po po}
                                      expand-settings
                                      js-missing?
                                      gettext-missing?
                                      verify-po-folders
                                      find-po-files)]
    (println "Extracting strings")
    (let [args (remove nil?
                       ["xgettext" "--from-code=UTF-8" "--debug"
                        (when no-location? "--no-location")
                        (when sort-output? "--sort-output")
                        "-k" "-kfulcro_tr:1" "-kfulcro_trc:1c,2" "-kfulcro_trf:1" "-o" messages-pot js-path])]
      (apply run args))
    (doseq [po (:existing-po-files settings)]
      (when (.exists (io/as-file (po-path settings po)))
        (println "Merging extracted PO template file to existing translations for " po)
        (run "msgmerge" "--force-po" "--no-wrap" "-U" (po-path settings po) messages-pot)))))

(defn deploy-translations
  "Scans for .po files and generates cljc for those translations in your app. At the moment, these must be output to
  the package `translations`.

  The settings map should contain:
  :src - The source folder (base) of where to emit the files. Defaults to `src`
  :po - The directory where you po files live. Defaults to `i18n`
  :as-modules? - If true, generates the locales so they can be dynamically loaded. NOTE: You must configure your build to support this as well
  by putting each locale in a module with the locale's cljs keyword (e.g. a module :de-AT { :entries #{translations/de-AT}})."
  [{:keys [src po as-modules?] :or {src "src" po "i18n" as-modules? false} :as settings}]
  (let [{:keys [existing-po-files output-path outdir]
         :as   settings} (some-> {:src src :ns "translations" :po po :as-modules? as-modules?}
                                 expand-settings
                                 verify-po-folders
                                 verify-source-folders
                                 find-po-files)
        replace-hyphen #(str/replace % #"-" "_")
        locales        (map clojure-ize-locale existing-po-files)]
    (println "po path is: " po)
    (println "Output path is: " output-path)
    (doseq [po existing-po-files]
      (let [locale            (clojure-ize-locale po)
            translation-map   (map-translations (po-path settings po))
            cljc-translations (wrap-translations-in-ns :ns "translations" :locale locale :translation translation-map :dynamic? as-modules?)
            cljc-trans-path   (str output-path "/" (replace-hyphen locale) ".cljc")]
        (println "Writing " cljc-trans-path)
        (spit cljc-trans-path cljc-translations)))
    (println "Deployed translations.")))
