(ns fooheads.build
  "from https://kozieiev.com/blog/packaging-clojure-into-jar-uberjar-with-tools-build/"
  (:require
    [clojure.edn :as edn]
    [clojure.java.shell :as shell]
    [clojure.string :as str]
    [clojure.tools.build.api :as b]
    [deps-deploy.deps-deploy :as dd]))


(def ^:private build-folder "target")
(def ^:private jar-content (str build-folder "/classes"))
(def ^:private basis (b/create-basis {:project "deps.edn"}))


(defn- git-remote-url
  "Get the git remote URL, normalized to HTTPS format"
  []
  (->
    (shell/sh "git" "remote" "get-url" "origin")
    :out
    str/trim
    (str/replace #"^git@github\.com:" "https://github.com/")))


(defn- ensure-git-suffix
  "Ensure URL ends with .git"
  [url]
  (str url (when-not (str/ends-with? url ".git") ".git")))


(defn- git-tag
  "Get the current git tag, if any"
  []
  (let [result (shell/sh "git" "describe" "--tags" "--exact-match" "HEAD")]
    (when (zero? (:exit result))
      (str/trim (:out result)))))


(defn- git-url
  []
  (->
    (git-remote-url)
    (str/replace #"\.git$" "")))


(defn- git-connection
  []
  (->
    (git-remote-url)
    (str/replace #"^https://github\.com/" "scm:git:https://github.com/")
    (ensure-git-suffix)))


(defn- git-developer-connection
  []
  (->
    (git-remote-url)
    (str/replace #"^https://github\.com/" "scm:git:ssh://git@github.com/")
    (ensure-git-suffix)))


(defn- lib-data
  []
  (edn/read-string (slurp "lib.edn")))


(defn- jar-file-name
  [lib-name version]
  (format "%s/%s-%s.jar" build-folder (name lib-name) version))


(defn clean
  [_]
  (b/delete {:path build-folder})
  (println (format "Build folder \"%s\" removed" build-folder)))


(defn jar
  [_]
  (let [{lib-name :name version :version} (lib-data)
        jar-file-name (jar-file-name lib-name version)]
    (clean nil)

    (b/copy-dir
      {:src-dirs   ["src" "resources"]
       :target-dir jar-content})

    (b/write-pom
      {:class-dir jar-content
       :lib       lib-name
       :version   version
       :basis     basis
       :src-dirs  ["src"]
       :scm
       (merge {:url (git-url)
               :connection (git-connection)
               :developerConnection (git-developer-connection)}
              (when-let [tag (git-tag)]
                {:tag tag}))})

    (b/jar
      {:class-dir jar-content
       :jar-file  jar-file-name lib-name version})

    (println (format "Jar file created: \"%s\"" jar-file-name))))


(defn deploy
  [_]
  (let [{lib-name :name
         version :version
         repository :repository} (lib-data)
        jar-file-name (jar-file-name lib-name version)]
    (jar nil)

    (dd/deploy {:installer :remote
                :artifact jar-file-name
                :pom-file (b/pom-path {:lib lib-name
                                       :class-dir jar-content})
                :repository {"releases" {:url repository}}})))


(defn help
  []
  (println "Available tasks:")
  (println "  clean   - Remove build folder")
  (println "  jar     - Builds a JAR file")
  (println "  deploy  - Builds a JAR and deploys to a remote repository")
  (println "  help    - Show this help message"))


(comment
  (clean "")
  (jar "")
  (deploy "")

  (let [{lib-name :name version :version} (lib-data)]
    (b/write-pom
      {:basis     basis
       :class-dir jar-content
       :lib       lib-name
       :version   version
       :src-dirs  ["src"]})))

