(ns flock.staff.logging.json-layout
  (:require [cheshire.core :as json]
            [flock.staff.logging.constants :as c])
  (:import [ch.qos.logback.classic.spi ILoggingEvent]
           [ch.qos.logback.classic Level])
  (:gen-class
   :init init
   :state state
   :constructors {[] []}
   :methods [[setPretty [Boolean] void]]
   :extends ch.qos.logback.core.spi.ContextAwareBase
   :implements [ch.qos.logback.core.Layout]
   :name flock.staff.logging.JsonLayout))

(defn- set-started!
  [this started?]
  (swap! (.state this) assoc :started started?))

(defn -init []
  [[] (atom {:started false})])

(defn -start
  [this]
  (set-started! this true)
  nil)

(defn -stop
  [this]
  (set-started! this false)
  nil)

(defn -isStarted
  [this]
  (:started @(.state this)))

(defn -setContext
  [this ^ch.qos.logback.core.Context context]
  (swap! (.state this) assoc :context context))

(defn -getContext
  [this]
  (:context @(.state this)))

(defn -setPretty
  [this ^Boolean pretty]
  (swap! (.state this) assoc :pretty pretty))

(defn- ->current-thread-context
  [event]
  (let [mdc (.getMDCPropertyMap event)
        context (-> mdc (get c/mdc-context-key {}) (json/parse-string keyword))
        request-id (get mdc c/mdc-request-id-key)]
    (if request-id
      (assoc context c/context-request-id-key request-id)
      context)))

(defn- ->throwable-proxy
  [t-proxy]
  (when t-proxy
    {:message (.getMessage t-proxy)
     :class-name (.getClassName t-proxy)
     :common-frames (.getCommonFrames t-proxy)
     :stack-trace (mapv str (.getStackTraceElementProxyArray t-proxy))}))

(defn- ->proxy
  [event]
  (when-let [t-proxy (.getThrowableProxy event)]
    (let [cause (.getCause t-proxy)
          suppressed (some->>
                      (.getSuppressed t-proxy)
                      (map #(.getMessage %))
                      not-empty)
          result (cond-> (->throwable-proxy t-proxy)
                   cause (assoc :cause (->throwable-proxy cause))
                   suppressed (assoc :suppressed suppressed))]
      {:throwable-proxy result})))

(defn ->caller
  [event]
  (when (= Level/ERROR (.getLevel event))
    {:caller (->>
              event
              (.getCallerData)
              (mapv str))}))

(defn -doLayout
  [this ^ILoggingEvent event]
  (let [ts (.getTimeStamp event)
        logger-context-vo (.getLoggerContextVO event)
        pretty? (:pretty @(.state this))
        json-opts {:date-format "HH:mm:ss.SSS" :pretty pretty?}]
    (->
     {:timestamp-human (java.util.Date. ts)
      :timestamp ts
      :thread (.getThreadName event)
      :level (str (.getLevel event))
      :logger (.getLoggerName event)
      :message (.getFormattedMessage event)
      :context (->current-thread-context event)}
     (merge (->caller event))
     (merge (->proxy event))
     (json/generate-string json-opts)
     (str "\n"))))

(defn -getContentType [this] "application/json")

(defn -getFileHeader [this])
(defn -getPresentationHeader [this])
(defn -getPresentationFooter [this])
(defn -getFileFooter [this])
