(ns flock.staff.logging
  (:require [clojure.tools.logging :as log]
            [clojure.tools.logging.impl :as impl]
            [flock.staff.logging.constants :as c]
            [cheshire.core :as json])
  (:import [flock.staff.logging JsonLayout]
           [org.slf4j MDC]
           ))

(def ^:dynamic *request-id* nil)

(defmacro with-binding-request-id
  [request-id & body]
  `(do
     (assert (string? ~request-id) "request-id must be string.")
     (binding [*request-id* ~request-id]
       ~@body)))

(defmacro with-mdc-request-id
  [request-id & body]
  (assert (string? request-id) "request-id must be string.")
  `(do
     (MDC/put c/mdc-request-id-key ~request-id)
     ~@body
     (MDC/remove c/mdc-request-id-key)))

(defn attach-binding-request-id
  [context request-id]
  (if request-id
    (assoc context c/context-request-id-key request-id)
    context))

(defn- stringify-context
  [x]
  (cond
    (map? x) (->> x (map (partial mapv stringify-context)) (into {}))
    (sequential? x) (mapv stringify-context x)
    (keyword? x) (name x)
    :else (str x)))

(defn safe-generate-json
  [context]
  (try
    (json/generate-string context)
    (catch Exception e
      (->> context
           stringify-context
           json/generate-string))))

(defmacro log
  {:arglists '([level context message & more] [level context throwable message & more])}
  [level context & args]
  `(do
     (when-not (map? ~context)
       (throw (Exception. "Context must be map.")))
     (->> *request-id*
          (attach-binding-request-id ~context)
          safe-generate-json
          (MDC/put c/mdc-context-key))
     (log/logp ~level ~@args)
     (MDC/remove c/mdc-context-key)))

;; level-specific macros

(defmacro trace
  {:arglists '([context message & more] [context throwable message & more])}
  [context & args]
  `(log :trace ~context ~@args))

(defmacro debug
  {:arglists '([context message & more] [context throwable message & more])}
  [context & args]
  `(log :debug ~context ~@args))

(defmacro info
  {:arglists '([context message & more] [context throwable message & more])}
  [context & args]
  `(log :info ~context ~@args))

(defmacro warn
  {:arglists '([context message & more] [context throwable message & more])}
  [context & args]
  `(log :warn ~context ~@args))

(defmacro error
  {:arglists '([context message & more] [context throwable message & more])}
  [context & args]
  `(log :error ~context ~@args))

;; Tests:

(comment
  (future
    (with-mdc-request-id "hjsdf123jk123"
      (info {:a 1} "test"))))

(comment
  (future
    (with-binding-request-id "34qwe123"
      (info {:b 1} "test"))))

(comment
  (info {:a 1 :b #{2 3} :f [4 5] :c {:d :e}} "test"))

(comment
  (doseq [i (range 10)]
    (future
      (with-binding-request-id (str "req-" i)
        (info {:param (str "future-" i)} {:i i})))))

(comment
  (doseq [i (range 10)]
    (future (info {:param (str "future-" i)} {:i i}))))

;; test json encoding error
(comment
  (safe-generate-json {:a (JsonLayout.)}))
