package com.flipkart.krystal.vajram.protobuf3.codegen;

import static com.flipkart.krystal.codegen.common.datatypes.StandardJavaType.BYTE;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ModelsProto3SchemaGen.validateModelType;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.isProtoTypeMap;
import static com.flipkart.krystal.vajram.protobuf3.codegen.ProtoGenUtils.isProtoTypeRepeated;
import static com.flipkart.krystal.vajram.protobuf3.codegen.VajramProtoConstants.MODELS_PROTO_MSG_SUFFIX;
import static com.squareup.javapoet.MethodSpec.methodBuilder;
import static javax.lang.model.element.Modifier.FINAL;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.lang.model.element.Modifier.STATIC;

import com.flipkart.krystal.codegen.common.datatypes.CodeGenType;
import com.flipkart.krystal.codegen.common.models.CodeGenUtility;
import com.flipkart.krystal.codegen.common.models.CodegenPhase;
import com.flipkart.krystal.codegen.common.models.DeclaredTypeVisitor;
import com.flipkart.krystal.codegen.common.spi.CodeGenerator;
import com.flipkart.krystal.codegen.common.spi.ModelsCodeGenContext;
import com.flipkart.krystal.model.IfAbsent.IfAbsentThen;
import com.flipkart.krystal.model.MandatoryFieldMissingException;
import com.flipkart.krystal.model.ModelRoot;
import com.flipkart.krystal.serial.SerializableModel;
import com.flipkart.krystal.vajram.protobuf3.Protobuf3;
import com.flipkart.krystal.vajram.protobuf3.SerializableProtoModel;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import lombok.extern.slf4j.Slf4j;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.checkerframework.checker.nullness.qual.Nullable;

/**
 * This code generator generates implementations in the model root hierarchy. More specifically, it
 * generates a final class name "[ModelName]_ImmutProto" which extends the "[ModelName]_Immut"
 * interface and the {@link SerializableModel} interface. This class wraps the proto java model
 * generated by protoc, whose name is "[ModelName]_Proto". All the getters in the class delegate to
 * the proto model. It also contains a lazily initialized byte array field named
 * "_serializedPayload". This is initialized in a constructor which accepts a byte array. This also
 * generates an inner class inside the ImmutProto class which extends the
 * "[ModelName]_Immut.Builder" interface and wraps a "[ModelName]_Proto.Builder" instance and all
 * setter and getter calls are delegated to this.
 */
@Slf4j
public class ModelsProto3Gen implements CodeGenerator {

  private final ModelsCodeGenContext codeGenContext;
  private final CodeGenUtility util;

  public ModelsProto3Gen(ModelsCodeGenContext codeGenContext) {
    this.codeGenContext = codeGenContext;
    this.util = codeGenContext.util();
  }

  @Override
  public void generate() {
    if (!isApplicable()) {
      return;
    }
    validate();
    generateProtoImplementation();
  }

  private boolean isApplicable() {
    if (!CodegenPhase.FINAL.equals(codeGenContext.codegenPhase())) {
      util.note("Skipping protobuf models codegen since current phase is not FINAL");
      return false;
    }
    if (!isProto3SerdeSupported()) {
      util.note(
          "Skipping protobuf models codegen since Protobuf3 is not a supported protocol of the model "
              + codeGenContext.modelRootType());
      return false;
    }

    TypeElement modelRootType = codeGenContext.modelRootType();
    // Check if the model root has the ModelRoot annotation
    ModelRoot modelRootAnnotation = modelRootType.getAnnotation(ModelRoot.class);
    if (modelRootAnnotation == null) {
      util.note(
          "Skipping class '%s' since it doesn't have @ModelRoot annotation"
              .formatted(modelRootType.getQualifiedName()));
      return false;
    }

    return true;
  }

  /**
   * Checks if the model root supports Json serialization.
   *
   * @return true if Json is supported, false otherwise
   */
  private boolean isProto3SerdeSupported() {
    return util.typeExplicitlySupportsProtocol(codeGenContext.modelRootType(), Protobuf3.class);
  }

  private void validate() {
    TypeElement modelRootType = codeGenContext.modelRootType();
    validateModelType(modelRootType, util);
  }

  private void generateProtoImplementation() {
    TypeElement modelRootType = codeGenContext.modelRootType();
    ClassName immutClassName = util.getImmutClassName(modelRootType);

    String protoClassName = getProtoClassName();
    String packageName = immutClassName.packageName();

    // Generate the implementation class using JavaPoet
    TypeSpec typeSpec = generateImplementationTypeSpec(modelRootType, packageName, protoClassName);

    // Create JavaFile and write to source file
    JavaFile javaFile = JavaFile.builder(packageName, typeSpec).build();

    util.generateSourceFile(packageName + "." + protoClassName, javaFile.toString(), modelRootType);

    log.info("Generated protobuf implementation class: {}", protoClassName);
  }

  private String getProtoClassName() {
    return util.getImmutClassName(codeGenContext.modelRootType()).simpleName()
        + Protobuf3.PROTOBUF_3.modelClassesSuffix();
  }

  private TypeSpec generateImplementationTypeSpec(
      TypeElement modelRootType, String packageName, String protoClassName) {
    ClassName immutableProtoType = ClassName.get(packageName, protoClassName);
    String modelRootName = modelRootType.getSimpleName().toString();
    ClassName immutModelName = util.getImmutClassName(modelRootType);
    String protoMsgClassName = modelRootName + MODELS_PROTO_MSG_SUFFIX;

    // Extract model methods
    List<ExecutableElement> modelMethods = util.extractAndValidateModelMethods(modelRootType);

    // Create class interfaces
    TypeName serializableTypeName =
        ParameterizedTypeName.get(
            ClassName.get(SerializableProtoModel.class),
            ClassName.get(packageName, protoMsgClassName));

    // Create field types
    TypeName byteArrayType = ArrayTypeName.of(TypeName.BYTE);
    ClassName protoMsgType = ClassName.get(packageName, protoMsgClassName);

    // Create class builder
    TypeSpec.Builder classBuilder =
        util.classBuilder(protoClassName, modelRootType.getQualifiedName().toString())
            .addModifiers(PUBLIC)
            .addSuperinterface(immutModelName)
            .addSuperinterface(serializableTypeName);

    // Add fields
    classBuilder.addField(
        FieldSpec.builder(
                byteArrayType.annotated(AnnotationSpec.builder(MonotonicNonNull.class).build()),
                "_serializedPayload",
                PRIVATE)
            .build());
    classBuilder.addField(
        FieldSpec.builder(
                protoMsgType.annotated(AnnotationSpec.builder(MonotonicNonNull.class).build()),
                "_proto",
                PRIVATE)
            .build());

    // Add constructor for serialized payload
    classBuilder.addMethod(
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC)
            .addParameter(byteArrayType, "_serializedPayload")
            .addStatement("this._serializedPayload = _serializedPayload")
            .addStatement("this._proto = null")
            .build());

    // Add constructor for proto message
    classBuilder.addMethod(
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC)
            .addParameter(protoMsgType, "_proto")
            .addStatement("this._proto = _proto")
            .addStatement("this._serializedPayload = null")
            .build());

    // Add _serialize method from Serializable interface with lazy initialization
    classBuilder.addMethod(
        MethodSpec.overriding(util.getMethod(SerializableModel.class, "_serialize", 0))
            .addCode(
"""
if (_serializedPayload == null){
  this._serializedPayload = _proto.toByteArray();
}
return _serializedPayload;
""")
            .build());

    // Add _build method from ImmutableModel interface
    classBuilder.addMethod(
        MethodSpec.methodBuilder("_build")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(immutModelName)
            .addStatement("return this")
            .build());

    // Add _asBuilder method from ImmutableModel interface
    classBuilder.addMethod(
        MethodSpec.methodBuilder("_asBuilder")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(ClassName.get("", "Builder"))
            .addStatement("return new Builder(_proto().toBuilder())")
            .build());

    // Add _newCopy method from ImmutableModel interface
    classBuilder.addMethod(
        MethodSpec.methodBuilder("_newCopy")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(immutableProtoType)
            .addCode(
                """
                if(_serializedPayload != null) {
                  return new $L(_serializedPayload);
                } else if(_proto != null){
                  return new $L(_proto);
                } else {
                  throw new $T("Both _proto and _serializedPayload are null");
                }
                """,
                protoClassName,
                protoClassName,
                IllegalStateException.class)
            .build());

    // Add method to lazily deserialize the proto message
    MethodSpec.Builder getProtoMsgBuilder =
        MethodSpec.overriding(util.getMethod(SerializableProtoModel.class, "_proto", 0))
            .returns(protoMsgType)
            .beginControlFlow("if (_proto == null && _serializedPayload != null)")
            .beginControlFlow("try")
            .addStatement("_proto = $T.parseFrom(_serializedPayload)", protoMsgType)
            .nextControlFlow("catch (Exception e)")
            .addStatement("throw new RuntimeException(\"Failed to deserialize proto message\", e)")
            .endControlFlow()
            .endControlFlow()
            .beginControlFlow("if (_proto == null)")
            .addStatement(
                "throw new IllegalStateException(\"Both _proto and _serializedPayload are null\")")
            .endControlFlow()
            .addStatement("return _proto");

    classBuilder.addMethod(getProtoMsgBuilder.build());

    // Add getters for all model methods
    for (ExecutableElement method : modelMethods) {

      // Get the return type
      TypeMirror returnType = method.getReturnType();
      CodeGenType dataType = new DeclaredTypeVisitor(util, method).visit(returnType);
      TypeName typeName = TypeName.get(returnType);

      classBuilder.addMethod(getterMethod(method, returnType, typeName, dataType).build());
    }

    // Add equals method that delegates to the proto object

    util.addImmutableModelObjectMethods(
        immutModelName,
        modelMethods.stream().map(ExecutableElement::getSimpleName).collect(Collectors.toSet()),
        classBuilder);

    // Add Builder inner class
    TypeSpec builderTypeSpec =
        generateBuilderTypeSpec(
            modelRootType,
            packageName,
            protoClassName,
            protoMsgClassName,
            immutModelName,
            modelMethods);
    classBuilder.addType(builderTypeSpec);
    classBuilder.addMethod(
        methodBuilder("_builder")
            .addModifiers(PUBLIC, STATIC)
            .returns(immutableProtoType.nestedClass("Builder"))
            .addStatement("return new $T()", immutableProtoType.nestedClass("Builder"))
            .build());
    return classBuilder.build();
  }

  private MethodSpec.Builder getterMethod(
      ExecutableElement method, TypeMirror returnType, TypeName typeName, CodeGenType dataType) {

    String methodName = method.getSimpleName().toString();

    MethodSpec.Builder getterBuilder =
        MethodSpec.methodBuilder(methodName).addAnnotation(Override.class).addModifiers(PUBLIC);

    // Only add @Nullable annotation if the field needs presence check, is not FAIL,
    // and the return type is not Optional
    if (needsPresenceCheckInModels(method)
        && !isMandatoryField(method)
        && !util.isOptional(returnType)) {
      getterBuilder.returns(typeName.annotated(AnnotationSpec.builder(Nullable.class).build()));
    } else {
      getterBuilder.returns(typeName);
    }

    addGetterCode(getterBuilder, method, dataType, methodName);
    return getterBuilder;
  }

  private void addGetterCode(
      MethodSpec.Builder getterBuilder,
      ExecutableElement method,
      CodeGenType dataType,
      String methodName) {

    if (isProtoTypeRepeated(dataType)) {
      // For repeated fields, use getXList() method
      getterBuilder.addStatement(
          "return _proto().get$LList()", CodeGenUtility.capitalizeFirstChar(methodName));
      return;
    }

    if (isProtoTypeMap(dataType)) {
      // For map fields, use getXMap() method
      getterBuilder.addStatement(
          "return _proto().get$LMap()", CodeGenUtility.capitalizeFirstChar(methodName));
      return;
    }
    boolean isOptionalReturnType = util.isOptional(method.getReturnType());

    // Return null for fields which can be inspected for presence/absence of value
    if (needsPresenceCheckInModels(method)) {
      CodeBlock protoPresenceCheck =
          CodeBlock.of(
              """
              if (!_proto().has$L()){
              """,
              CodeGenUtility.capitalizeFirstChar(methodName));

      // Add validation for mandatory fields
      if (isMandatoryField(method)) {
        getterBuilder
            .addCode(protoPresenceCheck)
            .addCode(
                """
                throw new $T($S, $S);
              }
              """,
                MandatoryFieldMissingException.class,
                getProtoClassName(),
                methodName);
      } else if (isOptionalReturnType) {
        getterBuilder
            .addCode(protoPresenceCheck)
            .addCode(
                """
                return Optional.empty();
              }
              """);
      } else {
        getterBuilder
            .addCode(protoPresenceCheck)
            .addCode(
                """
                return null;
              }
              """);
      }
    }

    // Get the value from the proto message
    if (isOptionalReturnType) {
      getterBuilder.addStatement(
          "return $T.of(_proto().get$L())",
          Optional.class,
          CodeGenUtility.capitalizeFirstChar(methodName));
    } else {
      getterBuilder.addStatement(
          "return _proto().get$L()", CodeGenUtility.capitalizeFirstChar(methodName));
    }
  }

  private TypeSpec generateBuilderTypeSpec(
      TypeElement modelRootType,
      String packageName,
      String protoClassName,
      String protoMsgClassName,
      ClassName immutInterfaceName,
      List<ExecutableElement> modelMethods) {
    ModelRoot modelRoot = modelRootType.getAnnotation(ModelRoot.class);

    ClassName immutableProtoType = ClassName.get(packageName, protoClassName);

    // Create class interfaces
    ClassName builderInterfaceClassName = immutInterfaceName.nestedClass("Builder");

    ClassName protoBuilderClassName =
        ClassName.get(packageName, protoMsgClassName).nestedClass("Builder");
    ClassName protoMsgClassNameObj = ClassName.get(packageName, protoMsgClassName);

    // Create Builder class
    TypeSpec.Builder builderClassBuilder =
        util.classBuilder("Builder", modelRootType.getQualifiedName().toString())
            .addModifiers(PUBLIC, STATIC)
            .addSuperinterface(builderInterfaceClassName);

    // Add Builder field
    builderClassBuilder.addField(
        FieldSpec.builder(protoBuilderClassName, "_proto", PRIVATE, FINAL).build());

    // Add Builder constructor with _proto parameter
    builderClassBuilder.addMethod(
        MethodSpec.constructorBuilder()
            .addParameter(protoBuilderClassName, "_proto")
            .addStatement("this._proto = _proto")
            .build());

    // Add Builder default constructor
    builderClassBuilder.addMethod(
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC)
            .addStatement("this._proto = $T.newBuilder()", protoMsgClassNameObj)
            .build());

    // Add _build method
    builderClassBuilder.addMethod(
        MethodSpec.methodBuilder("_build")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(immutableProtoType)
            .addStatement("return new $L(_proto.build())", protoClassName)
            .build());

    // Add _asBuilder method
    builderClassBuilder.addMethod(
        MethodSpec.methodBuilder("_asBuilder")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(ClassName.get("", "Builder"))
            .addStatement("return this")
            .build());

    // Add _newCopy method
    builderClassBuilder.addMethod(
        MethodSpec.methodBuilder("_newCopy")
            .addAnnotation(Override.class)
            .addModifiers(PUBLIC)
            .returns(ClassName.get("", "Builder"))
            .addStatement("return new Builder(_proto.clone())")
            .build());

    // Add getters and setters for all model methods
    for (ExecutableElement method : modelMethods) {
      String methodName = method.getSimpleName().toString();
      TypeMirror returnType = method.getReturnType();
      CodeGenType dataType = new DeclaredTypeVisitor(util, method).visit(returnType);

      // Check if the return type is Optional
      boolean isOptionalReturnType = util.isOptional(returnType);
      TypeName typeName;

      if (isOptionalReturnType) {
        // Use the inner type for the parameter if the return type is Optional
        TypeMirror innerType = util.getOptionalInnerType(returnType);
        typeName = TypeName.get(innerType);
      } else {
        typeName = TypeName.get(returnType);
      }

      if (typeName.isPrimitive()) {
        typeName = typeName.box();
      }

      // Add setter method
      MethodSpec.Builder setterBuilder =
          MethodSpec.methodBuilder(methodName)
              .addAnnotation(Override.class)
              .addModifiers(PUBLIC)
              .returns(ClassName.get("", "Builder"));

      // Add parameter
      ParameterSpec.Builder paramBuilder = ParameterSpec.builder(typeName, methodName);
      paramBuilder.addAnnotation(Nullable.class);

      // Check if the field is a repeated field (List) or a map field
      if (isProtoTypeRepeated(dataType)) {
        // For repeated fields, use clear and addAll pattern
        setterBuilder.addCode(
            """
                  _proto.clear$L();
                  if ($L == null){
                    return this;
                  }
                  _proto.addAll$L($L);
                """,
            CodeGenUtility.capitalizeFirstChar(methodName),
            methodName,
            CodeGenUtility.capitalizeFirstChar(methodName),
            methodName);
      } else if (isProtoTypeMap(dataType)) {
        // For map fields, use clear and putAll pattern
        setterBuilder.addCode(
            """
                  _proto.clear$L();
                  if ($L == null){
                    return this;
                  }
                  _proto.putAll$L($L);
                """,
            CodeGenUtility.capitalizeFirstChar(methodName),
            methodName,
            CodeGenUtility.capitalizeFirstChar(methodName),
            methodName);
      } else {
        // For regular fields
        setterBuilder.addCode(
            """
                  if ($L == null){
                    _proto.clear$L();
                    return this;
                  }
                """,
            methodName,
            CodeGenUtility.capitalizeFirstChar(methodName));

        setterBuilder.addStatement(
            dataType.equals(BYTE)
                ? "_proto.set$L(com.google.protobuf.ByteString.copyFrom(new byte[]{$L}))"
                : "_proto.set$L($L)",
            CodeGenUtility.capitalizeFirstChar(methodName),
            methodName);
      }

      setterBuilder.addStatement("return this");

      builderClassBuilder.addMethod(setterBuilder.addParameter(paramBuilder.build()).build());
      if (modelRoot.builderExtendsModelRoot()) {
        builderClassBuilder.addMethod(getterMethod(method, returnType, typeName, dataType).build());
      }
    }

    // Add method to access the proto builder
    return builderClassBuilder
        .addMethod(
            MethodSpec.methodBuilder("_proto")
                .addModifiers(PUBLIC)
                .returns(protoBuilderClassName)
                .addStatement("return _proto")
                .build())
        .build();
  }

  private boolean needsPresenceCheckInModels(ExecutableElement method) {
    TypeMirror returnType = method.getReturnType();
    CodeGenType dataType = new DeclaredTypeVisitor(util, method).visit(returnType, null);
    if (isProtoTypeRepeated(dataType)) {
      return false;
    }
    if (isProtoTypeMap(dataType)) {
      return false;
    }
    return !util.getIfAbsent(method).value().usePlatformDefault();
  }

  /**
   * Checks if a field should be treated as mandatory. Fields with @IfAbsent(FAIL) are treated as
   * mandatory.
   */
  private boolean isMandatoryField(ExecutableElement method) {
    return util.getIfAbsent(method).value() == IfAbsentThen.FAIL;
  }
}
