package com.flipkart.krystal.data;

import com.flipkart.krystal.except.KrystalCompletionException;
import com.flipkart.krystal.except.ThrowingCallable;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;

public sealed interface Errable<T> permits Success, Failure {

  /**
   * Returns a {@link CompletableFuture} which is completed exceptionally with the error if this is
   * a {@link Failure}, or completed normally with the value if this is a {@link Success}, or
   * completed normally with null if this is {@link Nil}
   */
  CompletableFuture<@Nullable T> toFuture();

  @Nullable T value();

  /**
   * Returns an {@link Optional} which is has the value inside this Errable. The returned Optional
   * is present only if and only if this Errable is of the type {@link NonNil}. In all other cases,
   * the returned Optional is empty.
   */
  Optional<T> valueOpt();

  /**
   * Returns an {@link Optional} which has the error which caused this Errable to be a {@link
   * Failure} if and only if this errable is a Failure. In all other cases, the returned Optional is
   * empty.
   */
  Optional<Throwable> errorOpt();

  /**
   * Returns a non-empty {@link Optional} if this is a {@link Success} or empty {@link Optional} it
   * this is {@link Nil}
   *
   * @throws RuntimeException if this is a {@link Failure}. If the error in {@link Failure} is not a
   *     {@link RuntimeException}, then the error is wrapped in a new {@link RuntimeException} and
   *     thrown.
   */
  Optional<T> valueOptOrThrow();

  /**
   *
   *
   * <ul>
   *   <li>NonNil: returns the non-null value
   *   <li>Nil: throws {@link NilValueException}
   *   <li>Failure: throws a {@link RuntimeException} representing the throwable which caused the
   *       failure. If the throwable is a {@link RuntimeException}, it is thrown as is. Else it is
   *       wrapped in a {@link KrystalCompletionException} and thrown.
   * </ul>
   */
  T valueOrThrow();

  default void handle(Consumer<Failure<T>> ifFailure, Consumer<NonNil<T>> ifNonNil) {
    handle(ifFailure, () -> {}, ifNonNil);
  }

  void handle(Consumer<Failure<T>> ifFailure, Runnable ifNil, Consumer<NonNil<T>> ifNonNil);

  <U> U map(Function<Failure<T>, U> ifFailure, Supplier<U> ifNil, Function<NonNil<T>, U> ifNonNil);

  /* ***********************************************************************************************/
  /* ************************************** Static utilities ***************************************/
  /* ***********************************************************************************************/

  static <T> Errable<T> nil() {
    return Nil.nil();
  }

  static <T> Errable<T> withValue(@Nullable T t) {
    return t != null ? new NonNil<>(t) : nil();
  }

  static <T> Errable<T> withError(Throwable t) {
    return new Failure<>(t);
  }

  static <T> Errable<T> errableFrom(ThrowingCallable<@Nullable T> valueProvider) {
    try {
      return withValue(valueProvider.call());
    } catch (Throwable e) {
      return withError(e);
    }
  }

  static <S, T> Function<S, Errable<@NonNull T>> computeErrableFrom(
      Function<S, @Nullable T> valueComputer) {
    return s -> errableFrom(() -> valueComputer.apply(s));
  }

  @SuppressWarnings("unchecked")
  static <T> Errable<T> errableFrom(@Nullable T value, @Nullable Throwable error) {
    if (value != null) {
      if (error != null) {
        throw illegalState();
      } else {
        return withValue(value);
      }
    } else if (error != null) {
      return withError(error);
    } else {
      return nil();
    }
  }

  private static IllegalArgumentException illegalState() {
    return new IllegalArgumentException("Both of 'value' and 'error' cannot be present together");
  }
}
