(ns exoscale.ticker
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as com])
  (:import (java.util.concurrent Executors
                                 ThreadFactory
                                 ScheduledExecutorService
                                 TimeUnit)
           (java.util.concurrent.atomic AtomicLong)))

(set! *warn-on-reflection* true)

(defonce thread-factory
  (let [thread-cnt (AtomicLong. 0)]
    (reify ThreadFactory
      (newThread [_ f]
        (doto (Thread. ^Runnable f)
          (.setName (format "exoscale-ticker-%s" (.getAndIncrement thread-cnt))))))))

(defn subscribe!
  "Registers an effect function `f` under key `k` to be regularly executed by the ticker.
  - `ticker`: The ticker instance returned by `ticker`.
  - `k`: A unique key for the effect.
  - `f`: A zero-argument function to be executed."
  [ticker k f]
  (swap! (::effects ticker) assoc k f))

(defn unsubscribe!
  "Removes the effect function associated with key `k` from the ticker.
  - `ticker`: The ticker instance.
  - `k`: The effect key to remove."
  [ticker k]
  (swap! (::effects ticker) dissoc k))

(defn ticker
  "Creates a ticker component that runs registered effects at a regular interval (in ms).
  Options:
    - `:interval` (default 15 minutes): interval in milliseconds between executions.
  Returns a component that implements `com/start` and `com/stop` and holds effect functions under `::effects`.
  Usage:
    (def t (com/start (ticker {:interval 1000})))
    (add-effect! t ::foo #(prn :yolo))
    (com/stop t)"
  [{:keys [interval]
    :or {interval (* 15 60 1000)}}]
  (with-meta {::effects (atom nil)}
    {`com/start
     (fn [{:as this ::keys [effects]}]
       (log/info "Starting ticker")
       (assoc this
              ::executor
              (doto (Executors/newSingleThreadScheduledExecutor thread-factory)
                (.scheduleWithFixedDelay
                 (fn runner* []
                   (run! (fn [[k f]]
                           (try
                             (f)
                             (catch Exception e
                               (log/error e
                                          (format "Ticker execution failure for %s"
                                                  (name k))))))
                         @effects))
                 interval
                 interval
                 TimeUnit/MILLISECONDS))))
     `com/stop
     (fn [{:as this ::keys [^ScheduledExecutorService executor effects]}]
       (log/info "Stopping ticker")
       (some-> executor .shutdown)
       (log/info "Ticker stopped")
       (reset! effects nil)
       (assoc this ::executor nil))}))

;; (def t (com/start (ticker {:interval (* 3 1000)})))
;; (add-effect! t ::foo #(prn :yolo))
;; (add-effect! t ::bar #(prn :bar))
;; (com/stop t)
