(ns exoscale.mania
  (:require [exoscale.mania.bc :as bc]
            [exoscale.mania.cli :as cli]
            [exoscale.mania.config :as config]
            [exoscale.mania.repl :as repl]
            [exoscale.mania.service :as service]
            [exoscale.mania.component :as component]
            [clojure.tools.logging :as log]
            [clojure.spec.alpha :as s]
            [signal.handler :as sig]))

(defn exit!
  [exit-code]
  (System/exit exit-code))

(defmacro safely!
  "runs body and allow to continue no matter if it throws, typically to
  be used on sigterm/sigint callbacks"
  [& body]
  `(try
     ~@body
     (catch Throwable e#
       (log/error e# "Unexpected error encountered, ignoring"))))

(defn set-signals-handlers
  [{:exoscale.mania.system/keys [stop start reload]}]
  (sig/with-handler :term
    (log/info "caught SIGTERM, quitting")
    (repl/stop!)
    (safely! (stop))
    (log/info "all components shut down")
    (exit! 0))

  (sig/with-handler :int
    (log/info "caught SIGINT, quitting")
    (repl/stop!)
    (safely! (stop))
    (log/info "all components shut down")
    (exit! 0))

  (sig/with-handler :hup
    (log/info "caught SIGHUP, reloading")
    (if (some? reload)
      (reload)
      (do (stop) (start)))
    (log/info "system reloaded")))

(s/def :exoscale.mania.system/init ifn?)
(s/def :exoscale.mania.system/start ifn?)
(s/def :exoscale.mania.system/stop ifn?)

(s/def :exoscale.mania.hooks/pre-init ifn?)
(s/def :exoscale.mania.hooks/on-init ifn?)
(s/def :exoscale.mania.hooks/on-start ifn?)
(s/def :exoscale.mania.hooks/on-error ifn?)

(s/def :exoscale.mania/options
  (s/keys :req [:exoscale.mania.service/name
                :exoscale.mania.cli/options
                :exoscale.mania.system/init
                :exoscale.mania.system/start
                :exoscale.mania.system/stop
                :exoscale.mania.config/type
                :exoscale.mania.config/spec]
          :opt [:exoscale.mania.config/profile
                :exoscale.mania.config/verbose-spec-error
                :exoscale.mania.service/version
                :exoscale.mania.hooks/pre-init
                :exoscale.mania.hooks/on-init
                :exoscale.mania.hooks/on-start
                :exoscale.mania.hooks/on-error
                :exoscale.mania.component/builder
                :exoscale.mania.repl/bind
                :exoscale.mania.repl/port]))

(def cli-inherited-keys
  "List of keys from args that will serve as default for local config"
  #{:exoscale.mania.config/file
    :exoscale.mania.config/profile
    :exoscale.mania.config/verbose-spec-error
    :exoscale.mania.repl/bind
    :exoscale.mania.repl/port})

(defn main-fn [opts]
  (let [{:as opts
         :exoscale.mania.service/keys [name version]
         :exoscale.mania.system/keys [init start]
         :exoscale.mania.hooks/keys [pre-init on-init on-start on-error]} (bc/adapt opts)
        options (conj cli/default-opts opts)
        ns' *ns*]
    (s/assert* :exoscale.mania/options options)
    (fn [& args-raw]
      (try
        (let [args (cli/parse-args options args-raw)
              args-options (-> args :options)
              version (or version (service/version ns'))]
          (when version
            (System/setProperty "exoscale.service.version"
                                version))

          (when (:version args-options)
            (println (or version "unknown"))
            (exit! 0))

          (when (:help args-options)
            (println "Options:")
            (println (:summary args))
            (exit! 0))

          (println (if version
                     (format "Starting %s version %s" name version)
                     (format "Starting %s" name)))

          (println "Loading config")
          (let [cli-opts+conf-opts (merge options
                                          (select-keys args-options cli-inherited-keys))

                {:keys [reporter logging] :as loaded-config} (config/load! cli-opts+conf-opts)
                config (merge loaded-config {:exoscale.mania.cli/options args-options})]

            (log/infof "Arguments: %s" args-options)

            (try
              ;; We do this with a dynamic resolve operation to avoid
              ;; depending on unilog
              (let [f (requiring-resolve 'unilog.config/start-logging!)]
                (log/info "Starting logging")
                (f logging))
              (catch Exception _))
            (when (:exoscale.mania.repl/port cli-opts+conf-opts)
              (repl/start! cli-opts+conf-opts))

            (try
              ;; We do this with a dynamic resolve operation to avoid
              ;; depending on reporter
              (let [f (requiring-resolve 'spootnik.reporter/initialize!)]
                (log/info "Starting reporter")
                (f reporter))
              (catch Exception _))

            (when (ifn? pre-init)
              (log/info "Starting pre-init")
              (pre-init config))

            (log/info "Initializing system")
            (init config)

            (when (ifn? on-init)
              (log/info "Starting on-init")
              (on-init config))

            (log/info "Setting signal handlers")
            (set-signals-handlers options)

            (log/info "Starting system")
            (start)
            (log/info "System started")

            (when (ifn? on-start)
              (log/info "Starting on-start")
              (on-start config))))

        (catch Exception e
          (binding [*out* *err*]
            (when (ifn? on-error) (on-error e))
            (println (format "Startup error: %s"
                             (ex-message e)))
            (println (format "Exiting %s" name)))
          (exit! 1))))))

(defmacro def-main
  [options]
  `(def ~'-main "main entry point generated by mania" (main-fn ~options)))

(defmacro def-component-main
  [options]
  `(let [options# (component/defsystem ~options)]
     (def-main options#)))
