(ns exoscale.mania.secret.file
  (:require [aero.core :as aero]
            [com.stuartsierra.component :as component]
            [clojure.spec.alpha :as s]
            [exoscale.cloak :as cloak]
            [exoscale.mania.reloadable :as reloadable]
            [exoscale.mania.secret :as secret]
            [clojure.tools.logging :as log]))

(defn- reload-edn-file
  "Reloads the edn `file`, validating it against `spec`.
  If the file is not present, it will use the base `seed` data."
  [seed file spec]
  (if (nil? file)
    seed
    (let [data (aero/read-config file)]
      (when-not (s/valid? spec data)
        (throw (ex-info (format "Provided file %s does not conform to spec" file)
                        (s/explain-data spec data))))
      (cloak/mask data))))


(defrecord FileSecretsProvider [file spec state-atom]
  component/Lifecycle
  (start [this]
    (swap! state-atom reload-edn-file file spec)
    this)
  (stop [this]
    this)
  reloadable/Reloadable
  (reload [this]
    (log/infof "Reloading EDN file from %s" file)
    (swap! state-atom reload-edn-file file spec)
    this)
  secret/SecretsProvider
  (lookup [this]
    (cloak/unmask @state-atom))
  (lookup [this k]
    (get (cloak/unmask @state-atom) k)))

(defn make-file-secrets-provider
  "Create a file-backed credential provider, with optional `seed` for initial data.
  The provider implements `exoscale.mania.reloadable/Reloadable`."
  ([file spec]
   (make-file-secrets-provider file {} spec))
  ([file seed spec]
   (->FileSecretsProvider file spec (atom seed))))

(defn make-static-secrets-provider
  "Create a static credentials provider with `seed` data."
  [seed spec]
  (make-file-secrets-provider nil seed spec))
