# Mania

A few helpers to build service/program/cronjob entrypoints.

The library will generate a `-main` function for you that has the
defaults we expect.

* Enforce cli parameter parsing with tools.cli
* Loads config from aero (edn) file in resource or from cli argument
* Setups/initializes logging
* Setups/initializes reporting
* Provides hooks to various stages of the startup
* Provides unified error handling/exit
* Provides hooks to perform init/stop/start/reload of a system
* Guarantees the presence of the production readiness properties

Leiningen: 

```clojure
[com.exoscale/mania "1.0.10"]
[exoscale/reporter "1.0.5"]
[spootnik/unilog "0.7.24"]
```

deps.edn:

```clojure
com.exoscale/mania {:mvn/version "1.0.10"}
exoscale/reporter {:mvn/version "1.0.5"}
spootnik/unilog {:mvn/version "0.7.24"}
```

## Usage

Call the library from your application's `main.clj`.
The namespace has to generate a Java class with `:gen-class`.

Below a minimal running system

```clj
(ns exoscale.server.main
  (:require [exoscale.mania :refer [def-component-main]]
            [com.stuartsierra.component :as component])
  (:gen-class))

;; global mutable var to hold the system
(defonce ^:redef system nil)

;; specs for the configuration
(s/def ::logging    (s/map-of keyword? (complement nil?)))
(s/def ::reporter   (s/map-of keyword? map?))
(s/def ::config     (s/keys :req-un [::reporter ::logging]))

(defrecord Service []
  component/Lifecycle
  (start [this])
  (stop [this]))

(defn build-system [config]
  (component/system-map
   :service (map->Service {})))
   
(def-component-main
  {:exoscale.mania.service/name         "Simple service"
   :exoscale.mania.component/system-var system
   :exoscale.mania.config/resource-file "config.edn"
   :exoscale.mania.config/spec          ::config
   :exoscale.mania.component/builder    build-system})

```

The following `config.edn` file should be present on the classpath (from the resources directory)

```edn
{:reporter
 {:sentry
  {:dsn #profile {:default "<DSN>"
                  :dev #or [#env SENTRY_DSN ":memory:"]}}}

 :logging {:console {:encoder :json}
           :level :info}}
```

### All options

```edn
{:exoscale.mania.service/name "foo"
 :exoscale.mania.service/version "1.10.1"               ;; if omited we'll infer it from the jar manifest
 :exoscale.mania.config/spec ::my-spec                  ;; defaults to a simple check for :logging and :reporter keys
 :exoscale.mania.config/file "foo.edn"                  ;; if specified will try to load the file from fs at <file>
 :exoscale.mania.config/resource-file "bar.edn"         ;; by default set to config.edn will load conf from resource if :exoscale.mania.config/file is not set or passed via cli
 :exoscale.mania.config/resolver aero/resource-resolver ;; required for aero #include directives to work with jar resources
 :exoscale.mania.cli/options                            ;; defaults to just -f <config-file>
 :exoscale.mania.system/init (fn [config] ...)          ;;
 :exoscale.mania.system/start (fn [] ...)               ;;
 :exoscale.mania.system/stop (fn [] ...)                ;;
 :exoscale.mania.system/reload-interval 3600            ;; number of seconds between each reload call
 :exoscale.mania.hooks/pre-init (fn [config] ...)       ;; runs before init
 :exoscale.mania.hooks/on-init (fn [config] ...)        ;; runs after init
 :exoscale.mania.hooks/on-start (fn [] ...)             ;; runs after start
 :exoscale.mania.hooks/on-error (fn [] ...)             ;; runs on error
 :exoscale.mania.repl/bind "127.0.0.1"                  ;; sets address for nREPL server
 :exoscale.mania.repl/port "7999"                       ;; sets port for nREPL server
 }

```

## Examples

- [Service](https://github.com/exoscale/mania/tree/master/examples/service.clj)
- [Cronjob](https://github.com/exoscale/mania/tree/master/examples/cronjob.clj)

## Config loading

It will by default try to load a config.edn file from the resources.  
You can also pass `-f <config-file>` to the cli and it will try to load this instead.

We expect the config file to have at least these keys:
* `:logging` unilog config format
* `:reporter` spootnik.reporter config format

We don't assume anything about how your start/manage your system with
init/start/stop, if you redef a var somewhere it's your prerogative
(we just call these functions, mania will not (re)set a system var for
you for instance).

In the case of aero you can pass `--profile` or `-p` and have its
value used as `#profile`.

You can also change the cli options via :cli/options config.


### Config secrets

We do support a custom aero reader tag to hide secrets:

``` clj
{:some-key #secret "launch-codes:1234"}
```

They will print `"<<-secret->>"` (a string, not to break serialisation
everywhere) instead of the actual value.
To get to the value you can deref the value of the key

``` clj
(prn (:some-key cfg))
=> "<<-secret->>"

(-> cfg :some-key deref)
=> "launch-codes:1234"
```

You can also use `(exoscale.mania.secret/unmask x)`, that would return x
with all Secrets unmasked, no matter the depth. **Be careful** when
using that, that would be something you'd want to use at the deepest
level, closest possible to the place where the value(s) is/are
actually needed. Also **be aware** of logging and what you expose from
this value.


You can also build secrets from code with `exoscale.mania.secret/mask`.

### `#env-file`

We support a custom aero reader `#env-file` which retrieves the value of the environment variable named by the key. If the variable is unset, but the same variable ending in `_FILE` is set, the contents of the file will be returned. Otherwise `nil` is returned


``` clj
;; {:file-content #env-file "ENV"}

(System/getenv "ENV")
=> nil
(System/getenv "ENV_FILE")
=> "/tmp/test-env-file"

(prn cfg)
=> {:file-content "file content"}
```

## System reloading

At times it is necessary to reload parts of the system without necessarily restarting the entire system. The canonical use case for this is to reload short lived secrets from a file when they are refreshed by an external tool.

For this purpose mania provides the `Reloadable` protocol

```clj
(defprotocol Reloadable
  (reload [this]
    "Cause the given component to reload its config.
     Returns a new updated instance of the component"))
```

As you can see this protocol is very straightforward: a component that wishes to support reloading just needs to implement the single `reload` function.
A default nop implementation is provided for components that do not explicitly implement the protocol.

As an example, here we have the reload implementation of a component that caches the s3 client configuration.
Because its s3 dependency may have changed when reload has been called, it has to recompute its client config cache.

```clj
(ns my-namespace
  (:require [exoscale.mania.reloadable :as reload]
            [com.stuartsierra.component :as component])

(defrecord my-component [s3 zone]
  component/Lifecycle
  (start [this]
    ;; Cache our zone s3 config
    (assoc this :zone-config (config-for s3 zone)))
  (stop [this] (assoc this :zone-config nil))
  reloadable/Reloadable
  (reload [this]
    ;; Our s3 dependency might have changed, recompute the cached config
    (assoc this :zone-config (config-for s3 zone)))))
```

Next to the the `start` and `stop` lifecycle functions, mania also generates a `reload` lifecycle function.
This function will invoke the `reload` protocol on each component in dependency order. It also ensures that updates dependencies are `assoc`'d into the component before `reload` is called.

By default, the reload function is invoked by sending `SIGHUP` to the process. The application will also reload
every 3600s seconds by default, and  a `:exoscale.mania.component/reloader` of type `IntervalReloader` component will be added to your system

You can define the key `exoscale.mania.config/reload-interval <number-of-seconds>` to ensure your 
application reloads at a different interval, or alternatively **a value of `0` will opt out of auto reload.**

If you prefer or want to do some action before reloading, a component is also provided 
which can schedule `reload` of the system on a timer.

```clj
;; We assume the namespace in which the mania (def-component) is run is called your-app.main
;; We need the dance around resolve: trying to resolve the reload! function at compile time
;; will typically result in circular import statements
(component/system-map
  ;; Reload the system every hour
  {:reloader (reloadable/map->IntervalReloader {:interval 3600
                                                :action (fn []
                                                           (log/info "Reloading system")
                                                           ((resolve 'your-app.main/reload!)))})
   :other-component (make-other-component)}
```

## Logging

refer to [unilog](https://github.com/pyr/unilog)

## Reporting

refer to [exoscale/reporter](https://github.com/exoscale/reporter)

## Version handling

We'll infer version at startup from the jar `MANIFEST.MF` file, but
you can also supply your own via `:program/version`. We'll
additionally set the java property `exoscale.service.version` to the
detected/supplied version, that can be useful if you need to use that
from a running service (ex: /version http endpoint).

## REPL support

You can start your service with for instance `--repl "localhost:7999"`
and get a nREPL server start with your app, which would allow you then
to connect to it with your editor/client (ex via `cider-connect` on
emacs).

This can also be setup via the following keys at config time:

```clj
:exoscale.mania.repl/bind "127.0.0.1"    ;; sets address for nREPL server
:exoscale.mania.repl/port "7999"         ;; sets port for nREPL server
```

However you should note that this could be a security risk if done via
config in production, you then **must ensure** that these ports are
not accessible from outside the host and rely on something like a ssh
tunnel to connect to it.

The default port is 7999, the default address is 127.0.0.1.

## available cli-parameters

```shell
  -f, --file <FILE>           Configuration file path
  -p, --profile <PROFILE_ID>  aero profile
  -r, --repl <ADDRESS:port>   nREPL server address <ip:port>
  -v, --version               Show version
  -h, --help                  Show Help
```

## TODO

- [ ] Improve version handling : https://github.com/exoscale/bundes/blob/master/src/bundes/version.clj
- [ ] Ensure REPL will always be disabled on production
- [ ] Add explicit dependencies to reporter/unilog - production readiness

## License

Copyright © 2024 [Exoscale](https://exoscale.com)
