;; by Joel Boehland http://github.com/jolby/colors
;; April 16, 2010

;; Copyright (c) Joel Boehland, 2010. All rights reserved.  The use
;; and distribution terms for this software are covered by the Eclipse
;; Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file epl-v10.html at the root of this
;; distribution.  By using this software in any fashion, you are
;; agreeing to be bound by the terms of this license.  You must not
;; remove this notice, or any other, from this software.


;; ================================================================================
;; Usage of colorbrewer schemes covered under an Apache licence:
;; Apache-Style Software License for ColorBrewer software and ColorBrewer Color Schemes, Version 1.1

;; Copyright (c) 2002 Cynthia Brewer, Mark Harrower, and The
;; Pennsylvania State University. All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:

;;    1. Redistributions as source code must retain the above
;;    copyright notice, this list of conditions and the following
;;    disclaimer.  2. The end-user documentation included with the
;;    redistribution, if any, must include the following
;;    acknowledgment:

;;       This product includes color specifications and designs
;;       developed by Cynthia Brewer (http://colorbrewer.org/).

;;       Alternately, this acknowledgment may appear in the software
;;    itself, if and wherever such third-party acknowledgments
;;    normally appear.  3. The name "ColorBrewer" must not be used to
;;    endorse or promote products derived from this software without
;;    prior written permission. For written permission, please contact
;;    Cynthia Brewer at cbrewer@psu.edu.  4. Products derived from
;;    this software may not be called "ColorBrewer", nor may
;;    "ColorBrewer" appear in their name, without prior written
;;    permission of Cynthia Brewer.

;; THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
;; WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
;; OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
;; DISCLAIMED. IN NO EVENT SHALL CYNTHIA BREWER, MARK HARROWER, OR THE
;; PENNSYLVANIA STATE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT,
;; INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
;; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
;; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
;; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
;; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
;; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
;; OF THE POSSIBILITY OF SUCH DAMAGE.

(ns com.evocomputing.colors.palettes.color-brewer
  (:use [com.evocomputing.colors :only (create-color)])
  (:require
   [clojure.contrib.str-utils :as su]
   [clojure.contrib.duck-streams :as ds]))


(declare color-brewer-palettes)

(def divergent-labels ["BrBG" "PiYG" "PRGn" "PuOr" "RdBu" "RdGy" "RdYlBu" "RdYlGn" "Spectral"])
(def qualitative-labels ["Accent" "Dark2" "Paired" "Pastel1" "Pastel2" "Set1" "Set2" "Set3"])
(def sequential-labels ["Blues" "BuGn" "BuPu" "GnBu" "Greens" "Greys" "Oranges" "OrRd" "PuBu"
                        "PuBuGn" "PuRd" "Purples" "RdPu" "Reds" "YlGn" "YlGnBu" "YlOrBr" "YlOrRd"])

(defn process-csv-line [header-line line]
  (let [[color-name num-colors type
         crit-val color-num color-letter
         r g b scheme-type] (su/re-split #"," line)
         qs #(when-not (or (nil? %) (= "" %)) (.replace % "\"" ""))
         nf (java.text.NumberFormat/getInstance)
         nums #(when-not (or (nil? %) (= "" %)) (.parse nf %))
         is-header (not (nil? (qs color-name)))
         cvals (vector (or (qs color-name) (header-line 0))
                       (or (nums num-colors) (header-line 1))
                       (or (qs type) (header-line 2))
                       (if is-header (nums crit-val) (header-line 3))
                       (nums color-num)
                       (qs color-letter)
                       (nums r) (nums g) (nums b)
                       (or (qs scheme-type) (header-line 9)))]
    cvals))

(defn read-colors-csv [path]
  (with-open [reader (java.io.BufferedReader. (java.io.FileReader. path))]
    (loop [colors []
           header-line (into [] (repeat 10 nil))]
      (let [line (.readLine reader)]
        (if-not line
          colors
          (let [cvals (process-csv-line header-line line)]
            (recur (conj colors cvals) (if (cvals 0) cvals header-line))))))))


(defn rec-to-color
  "Create a color object from a color brewer record"
  [rec]
  (create-color :r (rec 6) :g (rec 7) :b (rec 8)))

(defn get-color-brewer-palette
  "Get the colors associated with the provided palette and number"
  [palette-label number]
  (map #'rec-to-color
       (filter #(and (= palette-label (% 0)) (= number (% 1)))
               color-brewer-palettes)))

(def col-headers ["ColorName","NumOfColors","Type","CritVal",
                   "ColorNum","ColorLetter","R","G","B","SchemeType"])

(def color-brewer-palettes
[["Accent" 3 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 3 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 3 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 4 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 4 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 4 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 4 "qual" nil 4 "D" 255 255 153 "Qualitative"]
 ["Accent" 5 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 5 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 5 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 5 "qual" nil 4 "D" 255 255 153 "Qualitative"]
 ["Accent" 5 "qual" nil 5 "E" 56 108 176 "Qualitative"]
 ["Accent" 6 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 6 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 6 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 6 "qual" nil 4 "D" 255 255 153 "Qualitative"]
 ["Accent" 6 "qual" nil 5 "E" 56 108 176 "Qualitative"]
 ["Accent" 6 "qual" nil 6 "F" 240 2 127 "Qualitative"]
 ["Accent" 7 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 7 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 7 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 7 "qual" nil 4 "D" 255 255 153 "Qualitative"]
 ["Accent" 7 "qual" nil 5 "E" 56 108 176 "Qualitative"]
 ["Accent" 7 "qual" nil 6 "F" 240 2 127 "Qualitative"]
 ["Accent" 7 "qual" nil 7 "G" 191 91 23 "Qualitative"]
 ["Accent" 8 "qual" nil 1 "A" 127 201 127 "Qualitative"]
 ["Accent" 8 "qual" nil 2 "B" 190 174 212 "Qualitative"]
 ["Accent" 8 "qual" nil 3 "C" 253 192 134 "Qualitative"]
 ["Accent" 8 "qual" nil 4 "D" 255 255 153 "Qualitative"]
 ["Accent" 8 "qual" nil 5 "E" 56 108 176 "Qualitative"]
 ["Accent" 8 "qual" nil 6 "F" 240 2 127 "Qualitative"]
 ["Accent" 8 "qual" nil 7 "G" 191 91 23 "Qualitative"]
 ["Accent" 8 "qual" nil 8 "H" 102 102 102 "Qualitative"]
 ["Blues" 3 "seq" nil 1 "C" 222 235 247 "Sequential"]
 ["Blues" 3 "seq" nil 2 "F" 158 202 225 "Sequential"]
 ["Blues" 3 "seq" nil 3 "I" 49 130 189 "Sequential"]
 ["Blues" 4 "seq" nil 1 "B" 239 243 255 "Sequential"]
 ["Blues" 4 "seq" nil 2 "E" 189 215 231 "Sequential"]
 ["Blues" 4 "seq" nil 3 "G" 107 174 214 "Sequential"]
 ["Blues" 4 "seq" nil 4 "J" 33 113 181 "Sequential"]
 ["Blues" 5 "seq" nil 1 "B" 239 243 255 "Sequential"]
 ["Blues" 5 "seq" nil 2 "E" 189 215 231 "Sequential"]
 ["Blues" 5 "seq" nil 3 "G" 107 174 214 "Sequential"]
 ["Blues" 5 "seq" nil 4 "I" 49 130 189 "Sequential"]
 ["Blues" 5 "seq" nil 5 "K" 8 81 156 "Sequential"]
 ["Blues" 6 "seq" nil 1 "B" 239 243 255 "Sequential"]
 ["Blues" 6 "seq" nil 2 "D" 198 219 239 "Sequential"]
 ["Blues" 6 "seq" nil 3 "F" 158 202 225 "Sequential"]
 ["Blues" 6 "seq" nil 4 "G" 107 174 214 "Sequential"]
 ["Blues" 6 "seq" nil 5 "I" 49 130 189 "Sequential"]
 ["Blues" 6 "seq" nil 6 "K" 8 81 156 "Sequential"]
 ["Blues" 7 "seq" nil 1 "B" 239 243 255 "Sequential"]
 ["Blues" 7 "seq" nil 2 "D" 198 219 239 "Sequential"]
 ["Blues" 7 "seq" nil 3 "F" 158 202 225 "Sequential"]
 ["Blues" 7 "seq" nil 4 "G" 107 174 214 "Sequential"]
 ["Blues" 7 "seq" nil 5 "H" 66 146 198 "Sequential"]
 ["Blues" 7 "seq" nil 6 "J" 33 113 181 "Sequential"]
 ["Blues" 7 "seq" nil 7 "L" 8 69 148 "Sequential"]
 ["Blues" 8 "seq" nil 1 "A" 247 251 255 "Sequential"]
 ["Blues" 8 "seq" nil 2 "C" 222 235 247 "Sequential"]
 ["Blues" 8 "seq" nil 3 "D" 198 219 239 "Sequential"]
 ["Blues" 8 "seq" nil 4 "F" 158 202 225 "Sequential"]
 ["Blues" 8 "seq" nil 5 "G" 107 174 214 "Sequential"]
 ["Blues" 8 "seq" nil 6 "H" 66 146 198 "Sequential"]
 ["Blues" 8 "seq" nil 7 "J" 33 113 181 "Sequential"]
 ["Blues" 8 "seq" nil 8 "L" 8 69 148 "Sequential"]
 ["Blues" 9 "seq" nil 1 "A" 247 251 255 "Sequential"]
 ["Blues" 9 "seq" nil 2 "C" 222 235 247 "Sequential"]
 ["Blues" 9 "seq" nil 3 "D" 198 219 239 "Sequential"]
 ["Blues" 9 "seq" nil 4 "F" 158 202 225 "Sequential"]
 ["Blues" 9 "seq" nil 5 "G" 107 174 214 "Sequential"]
 ["Blues" 9 "seq" nil 6 "H" 66 146 198 "Sequential"]
 ["Blues" 9 "seq" nil 7 "J" 33 113 181 "Sequential"]
 ["Blues" 9 "seq" nil 8 "K" 8 81 156 "Sequential"]
 ["Blues" 9 "seq" nil 9 "M" 8 48 107 "Sequential"]
 ["BrBG" 3 "div" 2 1 "E" 216 179 101 "Diverging"]
 ["BrBG" 3 "div" 2 2 "H" 245 245 245 "Diverging"]
 ["BrBG" 3 "div" 2 3 "K" 90 180 172 "Diverging"]
 ["BrBG" 4 "div" 2.5 1 "C" 166 97 26 "Diverging"]
 ["BrBG" 4 "div" 2.5 2 "F" 223 194 125 "Diverging"]
 ["BrBG" 4 "div" 2.5 3 "J" 128 205 193 "Diverging"]
 ["BrBG" 4 "div" 2.5 4 "M" 1 133 113 "Diverging"]
 ["BrBG" 5 "div" 3 1 "C" 166 97 26 "Diverging"]
 ["BrBG" 5 "div" 3 2 "F" 223 194 125 "Diverging"]
 ["BrBG" 5 "div" 3 3 "H" 245 245 245 "Diverging"]
 ["BrBG" 5 "div" 3 4 "J" 128 205 193 "Diverging"]
 ["BrBG" 5 "div" 3 5 "M" 1 133 113 "Diverging"]
 ["BrBG" 6 "div" 3.5 1 "B" 140 81 10 "Diverging"]
 ["BrBG" 6 "div" 3.5 2 "E" 216 179 101 "Diverging"]
 ["BrBG" 6 "div" 3.5 3 "G" 246 232 195 "Diverging"]
 ["BrBG" 6 "div" 3.5 4 "I" 199 234 229 "Diverging"]
 ["BrBG" 6 "div" 3.5 5 "K" 90 180 172 "Diverging"]
 ["BrBG" 6 "div" 3.5 6 "N" 1 102 94 "Diverging"]
 ["BrBG" 7 "div" 4 1 "B" 140 81 10 "Diverging"]
 ["BrBG" 7 "div" 4 2 "E" 216 179 101 "Diverging"]
 ["BrBG" 7 "div" 4 3 "G" 246 232 195 "Diverging"]
 ["BrBG" 7 "div" 4 4 "H" 245 245 245 "Diverging"]
 ["BrBG" 7 "div" 4 5 "I" 199 234 229 "Diverging"]
 ["BrBG" 7 "div" 4 6 "K" 90 180 172 "Diverging"]
 ["BrBG" 7 "div" 4 7 "N" 1 102 94 "Diverging"]
 ["BrBG" 8 "div" 4.5 1 "B" 140 81 10 "Diverging"]
 ["BrBG" 8 "div" 4.5 2 "D" 191 129 45 "Diverging"]
 ["BrBG" 8 "div" 4.5 3 "F" 223 194 125 "Diverging"]
 ["BrBG" 8 "div" 4.5 4 "G" 246 232 195 "Diverging"]
 ["BrBG" 8 "div" 4.5 5 "I" 199 234 229 "Diverging"]
 ["BrBG" 8 "div" 4.5 6 "J" 128 205 193 "Diverging"]
 ["BrBG" 8 "div" 4.5 7 "L" 53 151 143 "Diverging"]
 ["BrBG" 8 "div" 4.5 8 "N" 1 102 94 "Diverging"]
 ["BrBG" 9 "div" 5 1 "B" 140 81 10 "Diverging"]
 ["BrBG" 9 "div" 5 2 "D" 191 129 45 "Diverging"]
 ["BrBG" 9 "div" 5 3 "F" 223 194 125 "Diverging"]
 ["BrBG" 9 "div" 5 4 "G" 246 232 195 "Diverging"]
 ["BrBG" 9 "div" 5 5 "H" 245 245 245 "Diverging"]
 ["BrBG" 9 "div" 5 6 "I" 199 234 229 "Diverging"]
 ["BrBG" 9 "div" 5 7 "J" 128 205 193 "Diverging"]
 ["BrBG" 9 "div" 5 8 "L" 53 151 143 "Diverging"]
 ["BrBG" 9 "div" 5 9 "N" 1 102 94 "Diverging"]
 ["BrBG" 10 "div" 5.5 1 "A" 84 48 5 "Diverging"]
 ["BrBG" 10 "div" 5.5 2 "B" 140 81 10 "Diverging"]
 ["BrBG" 10 "div" 5.5 3 "D" 191 129 45 "Diverging"]
 ["BrBG" 10 "div" 5.5 4 "F" 223 194 125 "Diverging"]
 ["BrBG" 10 "div" 5.5 5 "G" 246 232 195 "Diverging"]
 ["BrBG" 10 "div" 5.5 6 "I" 199 234 229 "Diverging"]
 ["BrBG" 10 "div" 5.5 7 "J" 128 205 193 "Diverging"]
 ["BrBG" 10 "div" 5.5 8 "L" 53 151 143 "Diverging"]
 ["BrBG" 10 "div" 5.5 9 "N" 1 102 94 "Diverging"]
 ["BrBG" 10 "div" 5.5 10 "O" 0 60 48 "Diverging"]
 ["BrBG" 11 "div" 6 1 "A" 84 48 5 "Diverging"]
 ["BrBG" 11 "div" 6 2 "B" 140 81 10 "Diverging"]
 ["BrBG" 11 "div" 6 3 "D" 191 129 45 "Diverging"]
 ["BrBG" 11 "div" 6 4 "F" 223 194 125 "Diverging"]
 ["BrBG" 11 "div" 6 5 "G" 246 232 195 "Diverging"]
 ["BrBG" 11 "div" 6 6 "H" 245 245 245 "Diverging"]
 ["BrBG" 11 "div" 6 7 "I" 199 234 229 "Diverging"]
 ["BrBG" 11 "div" 6 8 "J" 128 205 193 "Diverging"]
 ["BrBG" 11 "div" 6 9 "L" 53 151 143 "Diverging"]
 ["BrBG" 11 "div" 6 10 "N" 1 102 94 "Diverging"]
 ["BrBG" 11 "div" 6 11 "O" 0 60 48 "Diverging"]
 ["BuGn" 3 "seq" nil 1 "C" 229 245 249 "Sequential"]
 ["BuGn" 3 "seq" nil 2 "F" 153 216 201 "Sequential"]
 ["BuGn" 3 "seq" nil 3 "I" 44 162 95 "Sequential"]
 ["BuGn" 4 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuGn" 4 "seq" nil 2 "E" 178 226 226 "Sequential"]
 ["BuGn" 4 "seq" nil 3 "G" 102 194 164 "Sequential"]
 ["BuGn" 4 "seq" nil 4 "J" 35 139 69 "Sequential"]
 ["BuGn" 5 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuGn" 5 "seq" nil 2 "E" 178 226 226 "Sequential"]
 ["BuGn" 5 "seq" nil 3 "G" 102 194 164 "Sequential"]
 ["BuGn" 5 "seq" nil 4 "I" 44 162 95 "Sequential"]
 ["BuGn" 5 "seq" nil 5 "K" 0 109 44 "Sequential"]
 ["BuGn" 6 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuGn" 6 "seq" nil 2 "D" 204 236 230 "Sequential"]
 ["BuGn" 6 "seq" nil 3 "F" 153 216 201 "Sequential"]
 ["BuGn" 6 "seq" nil 4 "G" 102 194 164 "Sequential"]
 ["BuGn" 6 "seq" nil 5 "I" 44 162 95 "Sequential"]
 ["BuGn" 6 "seq" nil 6 "K" 0 109 44 "Sequential"]
 ["BuGn" 7 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuGn" 7 "seq" nil 2 "D" 204 236 230 "Sequential"]
 ["BuGn" 7 "seq" nil 3 "F" 153 216 201 "Sequential"]
 ["BuGn" 7 "seq" nil 4 "G" 102 194 164 "Sequential"]
 ["BuGn" 7 "seq" nil 5 "H" 65 174 118 "Sequential"]
 ["BuGn" 7 "seq" nil 6 "J" 35 139 69 "Sequential"]
 ["BuGn" 7 "seq" nil 7 "L" 0 88 36 "Sequential"]
 ["BuGn" 8 "seq" nil 1 "A" 247 252 253 "Sequential"]
 ["BuGn" 8 "seq" nil 2 "C" 229 245 249 "Sequential"]
 ["BuGn" 8 "seq" nil 3 "D" 204 236 230 "Sequential"]
 ["BuGn" 8 "seq" nil 4 "F" 153 216 201 "Sequential"]
 ["BuGn" 8 "seq" nil 5 "G" 102 194 164 "Sequential"]
 ["BuGn" 8 "seq" nil 6 "H" 65 174 118 "Sequential"]
 ["BuGn" 8 "seq" nil 7 "J" 35 139 69 "Sequential"]
 ["BuGn" 8 "seq" nil 8 "L" 0 88 36 "Sequential"]
 ["BuGn" 9 "seq" nil 1 "A" 247 252 253 "Sequential"]
 ["BuGn" 9 "seq" nil 2 "C" 229 245 249 "Sequential"]
 ["BuGn" 9 "seq" nil 3 "D" 204 236 230 "Sequential"]
 ["BuGn" 9 "seq" nil 4 "F" 153 216 201 "Sequential"]
 ["BuGn" 9 "seq" nil 5 "G" 102 194 164 "Sequential"]
 ["BuGn" 9 "seq" nil 6 "H" 65 174 118 "Sequential"]
 ["BuGn" 9 "seq" nil 7 "J" 35 139 69 "Sequential"]
 ["BuGn" 9 "seq" nil 8 "K" 0 109 44 "Sequential"]
 ["BuGn" 9 "seq" nil 9 "M" 0 68 27 "Sequential"]
 ["BuPu" 3 "seq" nil 1 "C" 224 236 244 "Sequential"]
 ["BuPu" 3 "seq" nil 2 "F" 158 188 218 "Sequential"]
 ["BuPu" 3 "seq" nil 3 "I" 136 86 167 "Sequential"]
 ["BuPu" 4 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuPu" 4 "seq" nil 2 "E" 179 205 227 "Sequential"]
 ["BuPu" 4 "seq" nil 3 "G" 140 150 198 "Sequential"]
 ["BuPu" 4 "seq" nil 4 "J" 136 65 157 "Sequential"]
 ["BuPu" 5 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuPu" 5 "seq" nil 2 "E" 179 205 227 "Sequential"]
 ["BuPu" 5 "seq" nil 3 "G" 140 150 198 "Sequential"]
 ["BuPu" 5 "seq" nil 4 "I" 136 86 167 "Sequential"]
 ["BuPu" 5 "seq" nil 5 "K" 129 15 124 "Sequential"]
 ["BuPu" 6 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuPu" 6 "seq" nil 2 "D" 191 211 230 "Sequential"]
 ["BuPu" 6 "seq" nil 3 "F" 158 188 218 "Sequential"]
 ["BuPu" 6 "seq" nil 4 "G" 140 150 198 "Sequential"]
 ["BuPu" 6 "seq" nil 5 "I" 136 86 167 "Sequential"]
 ["BuPu" 6 "seq" nil 6 "K" 129 15 124 "Sequential"]
 ["BuPu" 7 "seq" nil 1 "B" 237 248 251 "Sequential"]
 ["BuPu" 7 "seq" nil 2 "D" 191 211 230 "Sequential"]
 ["BuPu" 7 "seq" nil 3 "F" 158 188 218 "Sequential"]
 ["BuPu" 7 "seq" nil 4 "G" 140 150 198 "Sequential"]
 ["BuPu" 7 "seq" nil 5 "H" 140 107 177 "Sequential"]
 ["BuPu" 7 "seq" nil 6 "J" 136 65 157 "Sequential"]
 ["BuPu" 7 "seq" nil 7 "L" 110 1 107 "Sequential"]
 ["BuPu" 8 "seq" nil 1 "A" 247 252 253 "Sequential"]
 ["BuPu" 8 "seq" nil 2 "C" 224 236 244 "Sequential"]
 ["BuPu" 8 "seq" nil 3 "D" 191 211 230 "Sequential"]
 ["BuPu" 8 "seq" nil 4 "F" 158 188 218 "Sequential"]
 ["BuPu" 8 "seq" nil 5 "G" 140 150 198 "Sequential"]
 ["BuPu" 8 "seq" nil 6 "H" 140 107 177 "Sequential"]
 ["BuPu" 8 "seq" nil 7 "J" 136 65 157 "Sequential"]
 ["BuPu" 8 "seq" nil 8 "L" 110 1 107 "Sequential"]
 ["BuPu" 9 "seq" nil 1 "A" 247 252 253 "Sequential"]
 ["BuPu" 9 "seq" nil 2 "C" 224 236 244 "Sequential"]
 ["BuPu" 9 "seq" nil 3 "D" 191 211 230 "Sequential"]
 ["BuPu" 9 "seq" nil 4 "F" 158 188 218 "Sequential"]
 ["BuPu" 9 "seq" nil 5 "G" 140 150 198 "Sequential"]
 ["BuPu" 9 "seq" nil 6 "H" 140 107 177 "Sequential"]
 ["BuPu" 9 "seq" nil 7 "J" 136 65 157 "Sequential"]
 ["BuPu" 9 "seq" nil 8 "K" 129 15 124 "Sequential"]
 ["BuPu" 9 "seq" nil 9 "M" 77 0 75 "Sequential"]
 ["Dark2" 3 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 3 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 3 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 4 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 4 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 4 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 4 "qual" nil 4 "D" 231 41 138 "Qualitative"]
 ["Dark2" 5 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 5 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 5 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 5 "qual" nil 4 "D" 231 41 138 "Qualitative"]
 ["Dark2" 5 "qual" nil 5 "E" 102 166 30 "Qualitative"]
 ["Dark2" 6 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 6 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 6 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 6 "qual" nil 4 "D" 231 41 138 "Qualitative"]
 ["Dark2" 6 "qual" nil 5 "E" 102 166 30 "Qualitative"]
 ["Dark2" 6 "qual" nil 6 "F" 230 171 2 "Qualitative"]
 ["Dark2" 7 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 7 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 7 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 7 "qual" nil 4 "D" 231 41 138 "Qualitative"]
 ["Dark2" 7 "qual" nil 5 "E" 102 166 30 "Qualitative"]
 ["Dark2" 7 "qual" nil 6 "F" 230 171 2 "Qualitative"]
 ["Dark2" 7 "qual" nil 7 "G" 166 118 29 "Qualitative"]
 ["Dark2" 8 "qual" nil 1 "A" 27 158 119 "Qualitative"]
 ["Dark2" 8 "qual" nil 2 "B" 217 95 2 "Qualitative"]
 ["Dark2" 8 "qual" nil 3 "C" 117 112 179 "Qualitative"]
 ["Dark2" 8 "qual" nil 4 "D" 231 41 138 "Qualitative"]
 ["Dark2" 8 "qual" nil 5 "E" 102 166 30 "Qualitative"]
 ["Dark2" 8 "qual" nil 6 "F" 230 171 2 "Qualitative"]
 ["Dark2" 8 "qual" nil 7 "G" 166 118 29 "Qualitative"]
 ["Dark2" 8 "qual" nil 8 "H" 102 102 102 "Qualitative"]
 ["GnBu" 3 "seq" nil 1 "C" 224 243 219 "Sequential"]
 ["GnBu" 3 "seq" nil 2 "F" 168 221 181 "Sequential"]
 ["GnBu" 3 "seq" nil 3 "I" 67 162 202 "Sequential"]
 ["GnBu" 4 "seq" nil 1 "B" 240 249 232 "Sequential"]
 ["GnBu" 4 "seq" nil 2 "E" 186 228 188 "Sequential"]
 ["GnBu" 4 "seq" nil 3 "G" 123 204 196 "Sequential"]
 ["GnBu" 4 "seq" nil 4 "J" 43 140 190 "Sequential"]
 ["GnBu" 5 "seq" nil 1 "B" 240 249 232 "Sequential"]
 ["GnBu" 5 "seq" nil 2 "E" 186 228 188 "Sequential"]
 ["GnBu" 5 "seq" nil 3 "G" 123 204 196 "Sequential"]
 ["GnBu" 5 "seq" nil 4 "I" 67 162 202 "Sequential"]
 ["GnBu" 5 "seq" nil 5 "K" 8 104 172 "Sequential"]
 ["GnBu" 6 "seq" nil 1 "B" 240 249 232 "Sequential"]
 ["GnBu" 6 "seq" nil 2 "D" 204 235 197 "Sequential"]
 ["GnBu" 6 "seq" nil 3 "F" 168 221 181 "Sequential"]
 ["GnBu" 6 "seq" nil 4 "G" 123 204 196 "Sequential"]
 ["GnBu" 6 "seq" nil 5 "I" 67 162 202 "Sequential"]
 ["GnBu" 6 "seq" nil 6 "K" 8 104 172 "Sequential"]
 ["GnBu" 7 "seq" nil 1 "B" 240 249 232 "Sequential"]
 ["GnBu" 7 "seq" nil 2 "D" 204 235 197 "Sequential"]
 ["GnBu" 7 "seq" nil 3 "F" 168 221 181 "Sequential"]
 ["GnBu" 7 "seq" nil 4 "G" 123 204 196 "Sequential"]
 ["GnBu" 7 "seq" nil 5 "H" 78 179 211 "Sequential"]
 ["GnBu" 7 "seq" nil 6 "J" 43 140 190 "Sequential"]
 ["GnBu" 7 "seq" nil 7 "L" 8 88 158 "Sequential"]
 ["GnBu" 8 "seq" nil 1 "A" 247 252 240 "Sequential"]
 ["GnBu" 8 "seq" nil 2 "C" 224 243 219 "Sequential"]
 ["GnBu" 8 "seq" nil 3 "D" 204 235 197 "Sequential"]
 ["GnBu" 8 "seq" nil 4 "F" 168 221 181 "Sequential"]
 ["GnBu" 8 "seq" nil 5 "G" 123 204 196 "Sequential"]
 ["GnBu" 8 "seq" nil 6 "H" 78 179 211 "Sequential"]
 ["GnBu" 8 "seq" nil 7 "J" 43 140 190 "Sequential"]
 ["GnBu" 8 "seq" nil 8 "L" 8 88 158 "Sequential"]
 ["GnBu" 9 "seq" nil 1 "A" 247 252 240 "Sequential"]
 ["GnBu" 9 "seq" nil 2 "C" 224 243 219 "Sequential"]
 ["GnBu" 9 "seq" nil 3 "D" 204 235 197 "Sequential"]
 ["GnBu" 9 "seq" nil 4 "F" 168 221 181 "Sequential"]
 ["GnBu" 9 "seq" nil 5 "G" 123 204 196 "Sequential"]
 ["GnBu" 9 "seq" nil 6 "H" 78 179 211 "Sequential"]
 ["GnBu" 9 "seq" nil 7 "J" 43 140 190 "Sequential"]
 ["GnBu" 9 "seq" nil 8 "K" 8 104 172 "Sequential"]
 ["GnBu" 9 "seq" nil 9 "M" 8 64 129 "Sequential"]
 ["Greens" 3 "seq" nil 1 "C" 229 245 224 "Sequential"]
 ["Greens" 3 "seq" nil 2 "F" 161 217 155 "Sequential"]
 ["Greens" 3 "seq" nil 3 "I" 49 163 84 "Sequential"]
 ["Greens" 4 "seq" nil 1 "B" 237 248 233 "Sequential"]
 ["Greens" 4 "seq" nil 2 "E" 186 228 179 "Sequential"]
 ["Greens" 4 "seq" nil 3 "G" 116 196 118 "Sequential"]
 ["Greens" 4 "seq" nil 4 "J" 35 139 69 "Sequential"]
 ["Greens" 5 "seq" nil 1 "B" 237 248 233 "Sequential"]
 ["Greens" 5 "seq" nil 2 "E" 186 228 179 "Sequential"]
 ["Greens" 5 "seq" nil 3 "G" 116 196 118 "Sequential"]
 ["Greens" 5 "seq" nil 4 "I" 49 163 84 "Sequential"]
 ["Greens" 5 "seq" nil 5 "K" 0 109 44 "Sequential"]
 ["Greens" 6 "seq" nil 1 "B" 237 248 233 "Sequential"]
 ["Greens" 6 "seq" nil 2 "D" 199 233 192 "Sequential"]
 ["Greens" 6 "seq" nil 3 "F" 161 217 155 "Sequential"]
 ["Greens" 6 "seq" nil 4 "G" 116 196 118 "Sequential"]
 ["Greens" 6 "seq" nil 5 "I" 49 163 84 "Sequential"]
 ["Greens" 6 "seq" nil 6 "K" 0 109 44 "Sequential"]
 ["Greens" 7 "seq" nil 1 "B" 237 248 233 "Sequential"]
 ["Greens" 7 "seq" nil 2 "D" 199 233 192 "Sequential"]
 ["Greens" 7 "seq" nil 3 "F" 161 217 155 "Sequential"]
 ["Greens" 7 "seq" nil 4 "G" 116 196 118 "Sequential"]
 ["Greens" 7 "seq" nil 5 "H" 65 171 93 "Sequential"]
 ["Greens" 7 "seq" nil 6 "J" 35 139 69 "Sequential"]
 ["Greens" 7 "seq" nil 7 "L" 0 90 50 "Sequential"]
 ["Greens" 8 "seq" nil 1 "A" 247 252 245 "Sequential"]
 ["Greens" 8 "seq" nil 2 "C" 229 245 224 "Sequential"]
 ["Greens" 8 "seq" nil 3 "D" 199 233 192 "Sequential"]
 ["Greens" 8 "seq" nil 4 "F" 161 217 155 "Sequential"]
 ["Greens" 8 "seq" nil 5 "G" 116 196 118 "Sequential"]
 ["Greens" 8 "seq" nil 6 "H" 65 171 93 "Sequential"]
 ["Greens" 8 "seq" nil 7 "J" 35 139 69 "Sequential"]
 ["Greens" 8 "seq" nil 8 "L" 0 90 50 "Sequential"]
 ["Greens" 9 "seq" nil 1 "A" 247 252 245 "Sequential"]
 ["Greens" 9 "seq" nil 2 "C" 229 245 224 "Sequential"]
 ["Greens" 9 "seq" nil 3 "D" 199 233 192 "Sequential"]
 ["Greens" 9 "seq" nil 4 "F" 161 217 155 "Sequential"]
 ["Greens" 9 "seq" nil 5 "G" 116 196 118 "Sequential"]
 ["Greens" 9 "seq" nil 6 "H" 65 171 93 "Sequential"]
 ["Greens" 9 "seq" nil 7 "J" 35 139 69 "Sequential"]
 ["Greens" 9 "seq" nil 8 "K" 0 109 44 "Sequential"]
 ["Greens" 9 "seq" nil 9 "M" 0 68 27 "Sequential"]
 ["Greys" 3 "seq" nil 1 "C" 240 240 240 "Sequential"]
 ["Greys" 3 "seq" nil 2 "F" 189 189 189 "Sequential"]
 ["Greys" 3 "seq" nil 3 "I" 99 99 99 "Sequential"]
 ["Greys" 4 "seq" nil 1 "B" 247 247 247 "Sequential"]
 ["Greys" 4 "seq" nil 2 "E" 204 204 204 "Sequential"]
 ["Greys" 4 "seq" nil 3 "G" 150 150 150 "Sequential"]
 ["Greys" 4 "seq" nil 4 "J" 82 82 82 "Sequential"]
 ["Greys" 5 "seq" nil 1 "B" 247 247 247 "Sequential"]
 ["Greys" 5 "seq" nil 2 "E" 204 204 204 "Sequential"]
 ["Greys" 5 "seq" nil 3 "G" 150 150 150 "Sequential"]
 ["Greys" 5 "seq" nil 4 "I" 99 99 99 "Sequential"]
 ["Greys" 5 "seq" nil 5 "K" 37 37 37 "Sequential"]
 ["Greys" 6 "seq" nil 1 "B" 247 247 247 "Sequential"]
 ["Greys" 6 "seq" nil 2 "D" 217 217 217 "Sequential"]
 ["Greys" 6 "seq" nil 3 "F" 189 189 189 "Sequential"]
 ["Greys" 6 "seq" nil 4 "G" 150 150 150 "Sequential"]
 ["Greys" 6 "seq" nil 5 "I" 99 99 99 "Sequential"]
 ["Greys" 6 "seq" nil 6 "K" 37 37 37 "Sequential"]
 ["Greys" 7 "seq" nil 1 "B" 247 247 247 "Sequential"]
 ["Greys" 7 "seq" nil 2 "D" 217 217 217 "Sequential"]
 ["Greys" 7 "seq" nil 3 "F" 189 189 189 "Sequential"]
 ["Greys" 7 "seq" nil 4 "G" 150 150 150 "Sequential"]
 ["Greys" 7 "seq" nil 5 "H" 115 115 115 "Sequential"]
 ["Greys" 7 "seq" nil 6 "J" 82 82 82 "Sequential"]
 ["Greys" 7 "seq" nil 7 "L" 37 37 37 "Sequential"]
 ["Greys" 8 "seq" nil 1 "A" 255 255 255 "Sequential"]
 ["Greys" 8 "seq" nil 2 "C" 240 240 240 "Sequential"]
 ["Greys" 8 "seq" nil 3 "D" 217 217 217 "Sequential"]
 ["Greys" 8 "seq" nil 4 "F" 189 189 189 "Sequential"]
 ["Greys" 8 "seq" nil 5 "G" 150 150 150 "Sequential"]
 ["Greys" 8 "seq" nil 6 "H" 115 115 115 "Sequential"]
 ["Greys" 8 "seq" nil 7 "J" 82 82 82 "Sequential"]
 ["Greys" 8 "seq" nil 8 "L" 37 37 37 "Sequential"]
 ["Greys" 9 "seq" nil 1 "A" 255 255 255 "Sequential"]
 ["Greys" 9 "seq" nil 2 "C" 240 240 240 "Sequential"]
 ["Greys" 9 "seq" nil 3 "D" 217 217 217 "Sequential"]
 ["Greys" 9 "seq" nil 4 "F" 189 189 189 "Sequential"]
 ["Greys" 9 "seq" nil 5 "G" 150 150 150 "Sequential"]
 ["Greys" 9 "seq" nil 6 "H" 115 115 115 "Sequential"]
 ["Greys" 9 "seq" nil 7 "J" 82 82 82 "Sequential"]
 ["Greys" 9 "seq" nil 8 "K" 37 37 37 "Sequential"]
 ["Greys" 9 "seq" nil 9 "M" 0 0 0 "Sequential"]
 ["Oranges" 3 "seq" nil 1 "C" 254 230 206 "Sequential"]
 ["Oranges" 3 "seq" nil 2 "F" 253 174 107 "Sequential"]
 ["Oranges" 3 "seq" nil 3 "I" 230 85 13 "Sequential"]
 ["Oranges" 4 "seq" nil 1 "B" 254 237 222 "Sequential"]
 ["Oranges" 4 "seq" nil 2 "E" 253 190 133 "Sequential"]
 ["Oranges" 4 "seq" nil 3 "G" 253 141 60 "Sequential"]
 ["Oranges" 4 "seq" nil 4 "J" 217 71 1 "Sequential"]
 ["Oranges" 5 "seq" nil 1 "B" 254 237 222 "Sequential"]
 ["Oranges" 5 "seq" nil 2 "E" 253 190 133 "Sequential"]
 ["Oranges" 5 "seq" nil 3 "G" 253 141 60 "Sequential"]
 ["Oranges" 5 "seq" nil 4 "I" 230 85 13 "Sequential"]
 ["Oranges" 5 "seq" nil 5 "K" 166 54 3 "Sequential"]
 ["Oranges" 6 "seq" nil 1 "B" 254 237 222 "Sequential"]
 ["Oranges" 6 "seq" nil 2 "D" 253 208 162 "Sequential"]
 ["Oranges" 6 "seq" nil 3 "F" 253 174 107 "Sequential"]
 ["Oranges" 6 "seq" nil 4 "G" 253 141 60 "Sequential"]
 ["Oranges" 6 "seq" nil 5 "I" 230 85 13 "Sequential"]
 ["Oranges" 6 "seq" nil 6 "K" 166 54 3 "Sequential"]
 ["Oranges" 7 "seq" nil 1 "B" 254 237 222 "Sequential"]
 ["Oranges" 7 "seq" nil 2 "D" 253 208 162 "Sequential"]
 ["Oranges" 7 "seq" nil 3 "F" 253 174 107 "Sequential"]
 ["Oranges" 7 "seq" nil 4 "G" 253 141 60 "Sequential"]
 ["Oranges" 7 "seq" nil 5 "H" 241 105 19 "Sequential"]
 ["Oranges" 7 "seq" nil 6 "J" 217 72 1 "Sequential"]
 ["Oranges" 7 "seq" nil 7 "L" 140 45 4 "Sequential"]
 ["Oranges" 8 "seq" nil 1 "A" 255 245 235 "Sequential"]
 ["Oranges" 8 "seq" nil 2 "C" 254 230 206 "Sequential"]
 ["Oranges" 8 "seq" nil 3 "D" 253 208 162 "Sequential"]
 ["Oranges" 8 "seq" nil 4 "F" 253 174 107 "Sequential"]
 ["Oranges" 8 "seq" nil 5 "G" 253 141 60 "Sequential"]
 ["Oranges" 8 "seq" nil 6 "H" 241 105 19 "Sequential"]
 ["Oranges" 8 "seq" nil 7 "J" 217 72 1 "Sequential"]
 ["Oranges" 8 "seq" nil 8 "L" 140 45 4 "Sequential"]
 ["Oranges" 9 "seq" nil 1 "A" 255 245 235 "Sequential"]
 ["Oranges" 9 "seq" nil 2 "C" 254 230 206 "Sequential"]
 ["Oranges" 9 "seq" nil 3 "D" 253 208 162 "Sequential"]
 ["Oranges" 9 "seq" nil 4 "F" 253 174 107 "Sequential"]
 ["Oranges" 9 "seq" nil 5 "G" 253 141 60 "Sequential"]
 ["Oranges" 9 "seq" nil 6 "H" 241 105 19 "Sequential"]
 ["Oranges" 9 "seq" nil 7 "J" 217 72 1 "Sequential"]
 ["Oranges" 9 "seq" nil 8 "K" 166 54 3 "Sequential"]
 ["Oranges" 9 "seq" nil 9 "M" 127 39 4 "Sequential"]
 ["OrRd" 3 "seq" nil 1 "C" 254 232 200 "Sequential"]
 ["OrRd" 3 "seq" nil 2 "F" 253 187 132 "Sequential"]
 ["OrRd" 3 "seq" nil 3 "I" 227 74 51 "Sequential"]
 ["OrRd" 4 "seq" nil 1 "B" 254 240 217 "Sequential"]
 ["OrRd" 4 "seq" nil 2 "E" 253 204 138 "Sequential"]
 ["OrRd" 4 "seq" nil 3 "G" 252 141 89 "Sequential"]
 ["OrRd" 4 "seq" nil 4 "J" 215 48 31 "Sequential"]
 ["OrRd" 5 "seq" nil 1 "B" 254 240 217 "Sequential"]
 ["OrRd" 5 "seq" nil 2 "E" 253 204 138 "Sequential"]
 ["OrRd" 5 "seq" nil 3 "G" 252 141 89 "Sequential"]
 ["OrRd" 5 "seq" nil 4 "I" 227 74 51 "Sequential"]
 ["OrRd" 5 "seq" nil 5 "K" 179 0 0 "Sequential"]
 ["OrRd" 6 "seq" nil 1 "B" 254 240 217 "Sequential"]
 ["OrRd" 6 "seq" nil 2 "D" 253 212 158 "Sequential"]
 ["OrRd" 6 "seq" nil 3 "F" 253 187 132 "Sequential"]
 ["OrRd" 6 "seq" nil 4 "G" 252 141 89 "Sequential"]
 ["OrRd" 6 "seq" nil 5 "I" 227 74 51 "Sequential"]
 ["OrRd" 6 "seq" nil 6 "K" 179 0 0 "Sequential"]
 ["OrRd" 7 "seq" nil 1 "B" 254 240 217 "Sequential"]
 ["OrRd" 7 "seq" nil 2 "D" 253 212 158 "Sequential"]
 ["OrRd" 7 "seq" nil 3 "F" 253 187 132 "Sequential"]
 ["OrRd" 7 "seq" nil 4 "G" 252 141 89 "Sequential"]
 ["OrRd" 7 "seq" nil 5 "H" 239 101 72 "Sequential"]
 ["OrRd" 7 "seq" nil 6 "J" 215 48 31 "Sequential"]
 ["OrRd" 7 "seq" nil 7 "L" 153 0 0 "Sequential"]
 ["OrRd" 8 "seq" nil 1 "A" 255 247 236 "Sequential"]
 ["OrRd" 8 "seq" nil 2 "C" 254 232 200 "Sequential"]
 ["OrRd" 8 "seq" nil 3 "D" 253 212 158 "Sequential"]
 ["OrRd" 8 "seq" nil 4 "F" 253 187 132 "Sequential"]
 ["OrRd" 8 "seq" nil 5 "G" 252 141 89 "Sequential"]
 ["OrRd" 8 "seq" nil 6 "H" 239 101 72 "Sequential"]
 ["OrRd" 8 "seq" nil 7 "J" 215 48 31 "Sequential"]
 ["OrRd" 8 "seq" nil 8 "L" 153 0 0 "Sequential"]
 ["OrRd" 9 "seq" nil 1 "A" 255 247 236 "Sequential"]
 ["OrRd" 9 "seq" nil 2 "C" 254 232 200 "Sequential"]
 ["OrRd" 9 "seq" nil 3 "D" 253 212 158 "Sequential"]
 ["OrRd" 9 "seq" nil 4 "F" 253 187 132 "Sequential"]
 ["OrRd" 9 "seq" nil 5 "G" 252 141 89 "Sequential"]
 ["OrRd" 9 "seq" nil 6 "H" 239 101 72 "Sequential"]
 ["OrRd" 9 "seq" nil 7 "J" 215 48 31 "Sequential"]
 ["OrRd" 9 "seq" nil 8 "K" 179 0 0 "Sequential"]
 ["OrRd" 9 "seq" nil 9 "M" 127 0 0 "Sequential"]
 ["Paired" 3 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 3 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 3 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 4 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 4 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 4 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 4 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 5 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 5 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 5 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 5 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 5 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 6 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 6 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 6 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 6 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 6 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 6 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 7 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 7 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 7 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 7 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 7 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 7 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 7 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 8 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 8 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 8 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 8 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 8 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 8 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 8 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 8 "qual" nil 8 "H" 255 127 0 "Qualitative"]
 ["Paired" 9 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 9 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 9 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 9 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 9 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 9 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 9 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 9 "qual" nil 8 "H" 255 127 0 "Qualitative"]
 ["Paired" 9 "qual" nil 9 "I" 202 178 214 "Qualitative"]
 ["Paired" 10 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 10 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 10 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 10 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 10 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 10 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 10 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 10 "qual" nil 8 "H" 255 127 0 "Qualitative"]
 ["Paired" 10 "qual" nil 9 "I" 202 178 214 "Qualitative"]
 ["Paired" 10 "qual" nil 10 "J" 106 61 154 "Qualitative"]
 ["Paired" 11 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 11 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 11 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 11 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 11 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 11 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 11 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 11 "qual" nil 8 "H" 255 127 0 "Qualitative"]
 ["Paired" 11 "qual" nil 9 "I" 202 178 214 "Qualitative"]
 ["Paired" 11 "qual" nil 10 "J" 106 61 154 "Qualitative"]
 ["Paired" 11 "qual" nil 11 "K" 255 255 153 "Qualitative"]
 ["Paired" 12 "qual" nil 1 "A" 166 206 227 "Qualitative"]
 ["Paired" 12 "qual" nil 2 "B" 31 120 180 "Qualitative"]
 ["Paired" 12 "qual" nil 3 "C" 178 223 138 "Qualitative"]
 ["Paired" 12 "qual" nil 4 "D" 51 160 44 "Qualitative"]
 ["Paired" 12 "qual" nil 5 "E" 251 154 153 "Qualitative"]
 ["Paired" 12 "qual" nil 6 "F" 227 26 28 "Qualitative"]
 ["Paired" 12 "qual" nil 7 "G" 253 191 111 "Qualitative"]
 ["Paired" 12 "qual" nil 8 "H" 255 127 0 "Qualitative"]
 ["Paired" 12 "qual" nil 9 "I" 202 178 214 "Qualitative"]
 ["Paired" 12 "qual" nil 10 "J" 106 61 154 "Qualitative"]
 ["Paired" 12 "qual" nil 11 "K" 255 255 153 "Qualitative"]
 ["Paired" 12 "qual" nil 12 "L" 177 89 40 "Qualitative"]
 ["Pastel1" 3 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 3 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 3 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 4 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 4 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 4 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 4 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 5 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 5 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 5 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 5 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 5 "qual" nil 5 "E" 254 217 166 "Qualitative"]
 ["Pastel1" 6 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 6 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 6 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 6 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 6 "qual" nil 5 "E" 254 217 166 "Qualitative"]
 ["Pastel1" 6 "qual" nil 6 "F" 255 255 204 "Qualitative"]
 ["Pastel1" 7 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 7 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 7 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 7 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 7 "qual" nil 5 "E" 254 217 166 "Qualitative"]
 ["Pastel1" 7 "qual" nil 6 "F" 255 255 204 "Qualitative"]
 ["Pastel1" 7 "qual" nil 7 "G" 229 216 189 "Qualitative"]
 ["Pastel1" 8 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 8 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 8 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 8 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 8 "qual" nil 5 "E" 254 217 166 "Qualitative"]
 ["Pastel1" 8 "qual" nil 6 "F" 255 255 204 "Qualitative"]
 ["Pastel1" 8 "qual" nil 7 "G" 229 216 189 "Qualitative"]
 ["Pastel1" 8 "qual" nil 8 "H" 253 218 236 "Qualitative"]
 ["Pastel1" 9 "qual" nil 1 "A" 251 180 174 "Qualitative"]
 ["Pastel1" 9 "qual" nil 2 "B" 179 205 227 "Qualitative"]
 ["Pastel1" 9 "qual" nil 3 "C" 204 235 197 "Qualitative"]
 ["Pastel1" 9 "qual" nil 4 "D" 222 203 228 "Qualitative"]
 ["Pastel1" 9 "qual" nil 5 "E" 254 217 166 "Qualitative"]
 ["Pastel1" 9 "qual" nil 6 "F" 255 255 204 "Qualitative"]
 ["Pastel1" 9 "qual" nil 7 "G" 229 216 189 "Qualitative"]
 ["Pastel1" 9 "qual" nil 8 "H" 253 218 236 "Qualitative"]
 ["Pastel1" 9 "qual" nil 9 "I" 242 242 242 "Qualitative"]
 ["Pastel2" 3 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 3 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 3 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 4 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 4 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 4 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 4 "qual" nil 4 "D" 244 202 228 "Qualitative"]
 ["Pastel2" 5 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 5 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 5 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 5 "qual" nil 4 "D" 244 202 228 "Qualitative"]
 ["Pastel2" 5 "qual" nil 5 "E" 230 245 201 "Qualitative"]
 ["Pastel2" 6 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 6 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 6 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 6 "qual" nil 4 "D" 244 202 228 "Qualitative"]
 ["Pastel2" 6 "qual" nil 5 "E" 230 245 201 "Qualitative"]
 ["Pastel2" 6 "qual" nil 6 "F" 255 242 174 "Qualitative"]
 ["Pastel2" 7 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 7 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 7 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 7 "qual" nil 4 "D" 244 202 228 "Qualitative"]
 ["Pastel2" 7 "qual" nil 5 "E" 230 245 201 "Qualitative"]
 ["Pastel2" 7 "qual" nil 6 "F" 255 242 174 "Qualitative"]
 ["Pastel2" 7 "qual" nil 7 "G" 241 226 204 "Qualitative"]
 ["Pastel2" 8 "qual" nil 1 "A" 179 226 205 "Qualitative"]
 ["Pastel2" 8 "qual" nil 2 "B" 253 205 172 "Qualitative"]
 ["Pastel2" 8 "qual" nil 3 "C" 203 213 232 "Qualitative"]
 ["Pastel2" 8 "qual" nil 4 "D" 244 202 228 "Qualitative"]
 ["Pastel2" 8 "qual" nil 5 "E" 230 245 201 "Qualitative"]
 ["Pastel2" 8 "qual" nil 6 "F" 255 242 174 "Qualitative"]
 ["Pastel2" 8 "qual" nil 7 "G" 241 226 204 "Qualitative"]
 ["Pastel2" 8 "qual" nil 8 "H" 204 204 204 "Qualitative"]
 ["PiYG" 3 "div" 2 1 "E" 233 163 201 "Diverging"]
 ["PiYG" 3 "div" 2 2 "H" 247 247 247 "Diverging"]
 ["PiYG" 3 "div" 2 3 "K" 161 215 106 "Diverging"]
 ["PiYG" 4 "div" 2.5 1 "C" 208 28 139 "Diverging"]
 ["PiYG" 4 "div" 2.5 2 "F" 241 182 218 "Diverging"]
 ["PiYG" 4 "div" 2.5 3 "J" 184 225 134 "Diverging"]
 ["PiYG" 4 "div" 2.5 4 "M" 77 172 38 "Diverging"]
 ["PiYG" 5 "div" 3 1 "C" 208 28 139 "Diverging"]
 ["PiYG" 5 "div" 3 2 "F" 241 182 218 "Diverging"]
 ["PiYG" 5 "div" 3 3 "H" 247 247 247 "Diverging"]
 ["PiYG" 5 "div" 3 4 "J" 184 225 134 "Diverging"]
 ["PiYG" 5 "div" 3 5 "M" 77 172 38 "Diverging"]
 ["PiYG" 6 "div" 3.5 1 "B" 197 27 125 "Diverging"]
 ["PiYG" 6 "div" 3.5 2 "E" 233 163 201 "Diverging"]
 ["PiYG" 6 "div" 3.5 3 "G" 253 224 239 "Diverging"]
 ["PiYG" 6 "div" 3.5 4 "I" 230 245 208 "Diverging"]
 ["PiYG" 6 "div" 3.5 5 "K" 161 215 106 "Diverging"]
 ["PiYG" 6 "div" 3.5 6 "N" 77 146 33 "Diverging"]
 ["PiYG" 7 "div" 4 1 "B" 197 27 125 "Diverging"]
 ["PiYG" 7 "div" 4 2 "E" 233 163 201 "Diverging"]
 ["PiYG" 7 "div" 4 3 "G" 253 224 239 "Diverging"]
 ["PiYG" 7 "div" 4 4 "H" 247 247 247 "Diverging"]
 ["PiYG" 7 "div" 4 5 "I" 230 245 208 "Diverging"]
 ["PiYG" 7 "div" 4 6 "K" 161 215 106 "Diverging"]
 ["PiYG" 7 "div" 4 7 "N" 77 146 33 "Diverging"]
 ["PiYG" 8 "div" 4.5 1 "B" 197 27 125 "Diverging"]
 ["PiYG" 8 "div" 4.5 2 "D" 222 119 174 "Diverging"]
 ["PiYG" 8 "div" 4.5 3 "F" 241 182 218 "Diverging"]
 ["PiYG" 8 "div" 4.5 4 "G" 253 224 239 "Diverging"]
 ["PiYG" 8 "div" 4.5 5 "I" 230 245 208 "Diverging"]
 ["PiYG" 8 "div" 4.5 6 "J" 184 225 134 "Diverging"]
 ["PiYG" 8 "div" 4.5 7 "L" 127 188 65 "Diverging"]
 ["PiYG" 8 "div" 4.5 8 "N" 77 146 33 "Diverging"]
 ["PiYG" 9 "div" 5 1 "B" 197 27 125 "Diverging"]
 ["PiYG" 9 "div" 5 2 "D" 222 119 174 "Diverging"]
 ["PiYG" 9 "div" 5 3 "F" 241 182 218 "Diverging"]
 ["PiYG" 9 "div" 5 4 "G" 253 224 239 "Diverging"]
 ["PiYG" 9 "div" 5 5 "H" 247 247 247 "Diverging"]
 ["PiYG" 9 "div" 5 6 "I" 230 245 208 "Diverging"]
 ["PiYG" 9 "div" 5 7 "J" 184 225 134 "Diverging"]
 ["PiYG" 9 "div" 5 8 "L" 127 188 65 "Diverging"]
 ["PiYG" 9 "div" 5 9 "N" 77 146 33 "Diverging"]
 ["PiYG" 10 "div" 5.5 1 "A" 142 1 82 "Diverging"]
 ["PiYG" 10 "div" 5.5 2 "B" 197 27 125 "Diverging"]
 ["PiYG" 10 "div" 5.5 3 "D" 222 119 174 "Diverging"]
 ["PiYG" 10 "div" 5.5 4 "F" 241 182 218 "Diverging"]
 ["PiYG" 10 "div" 5.5 5 "G" 253 224 239 "Diverging"]
 ["PiYG" 10 "div" 5.5 6 "I" 230 245 208 "Diverging"]
 ["PiYG" 10 "div" 5.5 7 "J" 184 225 134 "Diverging"]
 ["PiYG" 10 "div" 5.5 8 "L" 127 188 65 "Diverging"]
 ["PiYG" 10 "div" 5.5 9 "N" 77 146 33 "Diverging"]
 ["PiYG" 10 "div" 5.5 10 "O" 39 100 25 "Diverging"]
 ["PiYG" 11 "div" 6 1 "A" 142 1 82 "Diverging"]
 ["PiYG" 11 "div" 6 2 "B" 197 27 125 "Diverging"]
 ["PiYG" 11 "div" 6 3 "D" 222 119 174 "Diverging"]
 ["PiYG" 11 "div" 6 4 "F" 241 182 218 "Diverging"]
 ["PiYG" 11 "div" 6 5 "G" 253 224 239 "Diverging"]
 ["PiYG" 11 "div" 6 6 "H" 247 247 247 "Diverging"]
 ["PiYG" 11 "div" 6 7 "I" 230 245 208 "Diverging"]
 ["PiYG" 11 "div" 6 8 "J" 184 225 134 "Diverging"]
 ["PiYG" 11 "div" 6 9 "L" 127 188 65 "Diverging"]
 ["PiYG" 11 "div" 6 10 "N" 77 146 33 "Diverging"]
 ["PiYG" 11 "div" 6 11 "O" 39 100 25 "Diverging"]
 ["PRGn" 3 "div" 2 1 "E" 175 141 195 "Diverging"]
 ["PRGn" 3 "div" 2 2 "H" 247 247 247 "Diverging"]
 ["PRGn" 3 "div" 2 3 "K" 127 191 123 "Diverging"]
 ["PRGn" 4 "div" 2.5 1 "C" 123 50 148 "Diverging"]
 ["PRGn" 4 "div" 2.5 2 "F" 194 165 207 "Diverging"]
 ["PRGn" 4 "div" 2.5 3 "J" 166 219 160 "Diverging"]
 ["PRGn" 4 "div" 2.5 4 "M" 0 136 55 "Diverging"]
 ["PRGn" 5 "div" 3 1 "C" 123 50 148 "Diverging"]
 ["PRGn" 5 "div" 3 2 "F" 194 165 207 "Diverging"]
 ["PRGn" 5 "div" 3 3 "H" 247 247 247 "Diverging"]
 ["PRGn" 5 "div" 3 4 "J" 166 219 160 "Diverging"]
 ["PRGn" 5 "div" 3 5 "M" 0 136 55 "Diverging"]
 ["PRGn" 6 "div" 3.5 1 "B" 118 42 131 "Diverging"]
 ["PRGn" 6 "div" 3.5 2 "E" 175 141 195 "Diverging"]
 ["PRGn" 6 "div" 3.5 3 "G" 231 212 232 "Diverging"]
 ["PRGn" 6 "div" 3.5 4 "I" 217 240 211 "Diverging"]
 ["PRGn" 6 "div" 3.5 5 "K" 127 191 123 "Diverging"]
 ["PRGn" 6 "div" 3.5 6 "N" 27 120 55 "Diverging"]
 ["PRGn" 7 "div" 4 1 "B" 118 42 131 "Diverging"]
 ["PRGn" 7 "div" 4 2 "E" 175 141 195 "Diverging"]
 ["PRGn" 7 "div" 4 3 "G" 231 212 232 "Diverging"]
 ["PRGn" 7 "div" 4 4 "H" 247 247 247 "Diverging"]
 ["PRGn" 7 "div" 4 5 "I" 217 240 211 "Diverging"]
 ["PRGn" 7 "div" 4 6 "K" 127 191 123 "Diverging"]
 ["PRGn" 7 "div" 4 7 "N" 27 120 55 "Diverging"]
 ["PRGn" 8 "div" 4.5 1 "B" 118 42 131 "Diverging"]
 ["PRGn" 8 "div" 4.5 2 "D" 153 112 171 "Diverging"]
 ["PRGn" 8 "div" 4.5 3 "F" 194 165 207 "Diverging"]
 ["PRGn" 8 "div" 4.5 4 "G" 231 212 232 "Diverging"]
 ["PRGn" 8 "div" 4.5 5 "I" 217 240 211 "Diverging"]
 ["PRGn" 8 "div" 4.5 6 "J" 166 219 160 "Diverging"]
 ["PRGn" 8 "div" 4.5 7 "L" 90 174 97 "Diverging"]
 ["PRGn" 8 "div" 4.5 8 "N" 27 120 55 "Diverging"]
 ["PRGn" 9 "div" 5 1 "B" 118 42 131 "Diverging"]
 ["PRGn" 9 "div" 5 2 "D" 153 112 171 "Diverging"]
 ["PRGn" 9 "div" 5 3 "F" 194 165 207 "Diverging"]
 ["PRGn" 9 "div" 5 4 "G" 231 212 232 "Diverging"]
 ["PRGn" 9 "div" 5 5 "H" 247 247 247 "Diverging"]
 ["PRGn" 9 "div" 5 6 "I" 217 240 211 "Diverging"]
 ["PRGn" 9 "div" 5 7 "J" 166 219 160 "Diverging"]
 ["PRGn" 9 "div" 5 8 "L" 90 174 97 "Diverging"]
 ["PRGn" 9 "div" 5 9 "N" 27 120 55 "Diverging"]
 ["PRGn" 10 "div" 5.5 1 "A" 64 0 75 "Diverging"]
 ["PRGn" 10 "div" 5.5 2 "B" 118 42 131 "Diverging"]
 ["PRGn" 10 "div" 5.5 3 "D" 153 112 171 "Diverging"]
 ["PRGn" 10 "div" 5.5 4 "F" 194 165 207 "Diverging"]
 ["PRGn" 10 "div" 5.5 5 "G" 231 212 232 "Diverging"]
 ["PRGn" 10 "div" 5.5 6 "I" 217 240 211 "Diverging"]
 ["PRGn" 10 "div" 5.5 7 "J" 166 219 160 "Diverging"]
 ["PRGn" 10 "div" 5.5 8 "L" 90 174 97 "Diverging"]
 ["PRGn" 10 "div" 5.5 9 "N" 27 120 55 "Diverging"]
 ["PRGn" 10 "div" 5.5 10 "O" 0 68 27 "Diverging"]
 ["PRGn" 11 "div" 6 1 "A" 64 0 75 "Diverging"]
 ["PRGn" 11 "div" 6 2 "B" 118 42 131 "Diverging"]
 ["PRGn" 11 "div" 6 3 "D" 153 112 171 "Diverging"]
 ["PRGn" 11 "div" 6 4 "F" 194 165 207 "Diverging"]
 ["PRGn" 11 "div" 6 5 "G" 231 212 232 "Diverging"]
 ["PRGn" 11 "div" 6 6 "H" 247 247 247 "Diverging"]
 ["PRGn" 11 "div" 6 7 "I" 217 240 211 "Diverging"]
 ["PRGn" 11 "div" 6 8 "J" 166 219 160 "Diverging"]
 ["PRGn" 11 "div" 6 9 "L" 90 174 97 "Diverging"]
 ["PRGn" 11 "div" 6 10 "N" 27 120 55 "Diverging"]
 ["PRGn" 11 "div" 6 11 "O" 0 68 27 "Diverging"]
 ["PuBu" 3 "seq" nil 1 "C" 236 231 242 "Sequential"]
 ["PuBu" 3 "seq" nil 2 "F" 166 189 219 "Sequential"]
 ["PuBu" 3 "seq" nil 3 "I" 43 140 190 "Sequential"]
 ["PuBu" 4 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuBu" 4 "seq" nil 2 "E" 189 201 225 "Sequential"]
 ["PuBu" 4 "seq" nil 3 "G" 116 169 207 "Sequential"]
 ["PuBu" 4 "seq" nil 4 "J" 5 112 176 "Sequential"]
 ["PuBu" 5 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuBu" 5 "seq" nil 2 "E" 189 201 225 "Sequential"]
 ["PuBu" 5 "seq" nil 3 "G" 116 169 207 "Sequential"]
 ["PuBu" 5 "seq" nil 4 "I" 43 140 190 "Sequential"]
 ["PuBu" 5 "seq" nil 5 "K" 4 90 141 "Sequential"]
 ["PuBu" 6 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuBu" 6 "seq" nil 2 "D" 208 209 230 "Sequential"]
 ["PuBu" 6 "seq" nil 3 "F" 166 189 219 "Sequential"]
 ["PuBu" 6 "seq" nil 4 "G" 116 169 207 "Sequential"]
 ["PuBu" 6 "seq" nil 5 "I" 43 140 190 "Sequential"]
 ["PuBu" 6 "seq" nil 6 "K" 4 90 141 "Sequential"]
 ["PuBu" 7 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuBu" 7 "seq" nil 2 "D" 208 209 230 "Sequential"]
 ["PuBu" 7 "seq" nil 3 "F" 166 189 219 "Sequential"]
 ["PuBu" 7 "seq" nil 4 "G" 116 169 207 "Sequential"]
 ["PuBu" 7 "seq" nil 5 "H" 54 144 192 "Sequential"]
 ["PuBu" 7 "seq" nil 6 "J" 5 112 176 "Sequential"]
 ["PuBu" 7 "seq" nil 7 "L" 3 78 123 "Sequential"]
 ["PuBu" 8 "seq" nil 1 "A" 255 247 251 "Sequential"]
 ["PuBu" 8 "seq" nil 2 "C" 236 231 242 "Sequential"]
 ["PuBu" 8 "seq" nil 3 "D" 208 209 230 "Sequential"]
 ["PuBu" 8 "seq" nil 4 "F" 166 189 219 "Sequential"]
 ["PuBu" 8 "seq" nil 5 "G" 116 169 207 "Sequential"]
 ["PuBu" 8 "seq" nil 6 "H" 54 144 192 "Sequential"]
 ["PuBu" 8 "seq" nil 7 "J" 5 112 176 "Sequential"]
 ["PuBu" 8 "seq" nil 8 "L" 3 78 123 "Sequential"]
 ["PuBu" 9 "seq" nil 1 "A" 255 247 251 "Sequential"]
 ["PuBu" 9 "seq" nil 2 "C" 236 231 242 "Sequential"]
 ["PuBu" 9 "seq" nil 3 "D" 208 209 230 "Sequential"]
 ["PuBu" 9 "seq" nil 4 "F" 166 189 219 "Sequential"]
 ["PuBu" 9 "seq" nil 5 "G" 116 169 207 "Sequential"]
 ["PuBu" 9 "seq" nil 6 "H" 54 144 192 "Sequential"]
 ["PuBu" 9 "seq" nil 7 "J" 5 112 176 "Sequential"]
 ["PuBu" 9 "seq" nil 8 "K" 4 90 141 "Sequential"]
 ["PuBu" 9 "seq" nil 9 "M" 2 56 88 "Sequential"]
 ["PuBuGn" 3 "seq" nil 1 "C" 236 226 240 "Sequential"]
 ["PuBuGn" 3 "seq" nil 2 "F" 166 189 219 "Sequential"]
 ["PuBuGn" 3 "seq" nil 3 "I" 28 144 153 "Sequential"]
 ["PuBuGn" 4 "seq" nil 1 "B" 246 239 247 "Sequential"]
 ["PuBuGn" 4 "seq" nil 2 "E" 189 201 225 "Sequential"]
 ["PuBuGn" 4 "seq" nil 3 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 4 "seq" nil 4 "J" 2 129 138 "Sequential"]
 ["PuBuGn" 5 "seq" nil 1 "B" 246 239 247 "Sequential"]
 ["PuBuGn" 5 "seq" nil 2 "E" 189 201 225 "Sequential"]
 ["PuBuGn" 5 "seq" nil 3 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 5 "seq" nil 4 "I" 28 144 153 "Sequential"]
 ["PuBuGn" 5 "seq" nil 5 "K" 1 108 89 "Sequential"]
 ["PuBuGn" 6 "seq" nil 1 "B" 246 239 247 "Sequential"]
 ["PuBuGn" 6 "seq" nil 2 "D" 208 209 230 "Sequential"]
 ["PuBuGn" 6 "seq" nil 3 "F" 166 189 219 "Sequential"]
 ["PuBuGn" 6 "seq" nil 4 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 6 "seq" nil 5 "I" 28 144 153 "Sequential"]
 ["PuBuGn" 6 "seq" nil 6 "K" 1 108 89 "Sequential"]
 ["PuBuGn" 7 "seq" nil 1 "B" 246 239 247 "Sequential"]
 ["PuBuGn" 7 "seq" nil 2 "D" 208 209 230 "Sequential"]
 ["PuBuGn" 7 "seq" nil 3 "F" 166 189 219 "Sequential"]
 ["PuBuGn" 7 "seq" nil 4 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 7 "seq" nil 5 "H" 54 144 192 "Sequential"]
 ["PuBuGn" 7 "seq" nil 6 "J" 2 129 138 "Sequential"]
 ["PuBuGn" 7 "seq" nil 7 "L" 1 100 80 "Sequential"]
 ["PuBuGn" 8 "seq" nil 1 "A" 255 247 251 "Sequential"]
 ["PuBuGn" 8 "seq" nil 2 "C" 236 226 240 "Sequential"]
 ["PuBuGn" 8 "seq" nil 3 "D" 208 209 230 "Sequential"]
 ["PuBuGn" 8 "seq" nil 4 "F" 166 189 219 "Sequential"]
 ["PuBuGn" 8 "seq" nil 5 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 8 "seq" nil 6 "H" 54 144 192 "Sequential"]
 ["PuBuGn" 8 "seq" nil 7 "J" 2 129 138 "Sequential"]
 ["PuBuGn" 8 "seq" nil 8 "L" 1 100 80 "Sequential"]
 ["PuBuGn" 9 "seq" nil 1 "A" 255 247 251 "Sequential"]
 ["PuBuGn" 9 "seq" nil 2 "C" 236 226 240 "Sequential"]
 ["PuBuGn" 9 "seq" nil 3 "D" 208 209 230 "Sequential"]
 ["PuBuGn" 9 "seq" nil 4 "F" 166 189 219 "Sequential"]
 ["PuBuGn" 9 "seq" nil 5 "G" 103 169 207 "Sequential"]
 ["PuBuGn" 9 "seq" nil 6 "H" 54 144 192 "Sequential"]
 ["PuBuGn" 9 "seq" nil 7 "J" 2 129 138 "Sequential"]
 ["PuBuGn" 9 "seq" nil 8 "K" 1 108 89 "Sequential"]
 ["PuBuGn" 9 "seq" nil 9 "M" 1 70 54 "Sequential"]
 ["PuOr" 3 "div" 2 1 "E" 241 163 64 "Diverging"]
 ["PuOr" 3 "div" 2 2 "H" 247 247 247 "Diverging"]
 ["PuOr" 3 "div" 2 3 "K" 153 142 195 "Diverging"]
 ["PuOr" 4 "div" 2.5 1 "C" 230 97 1 "Diverging"]
 ["PuOr" 4 "div" 2.5 2 "F" 253 184 99 "Diverging"]
 ["PuOr" 4 "div" 2.5 3 "J" 178 171 210 "Diverging"]
 ["PuOr" 4 "div" 2.5 4 "M" 94 60 153 "Diverging"]
 ["PuOr" 5 "div" 3 1 "C" 230 97 1 "Diverging"]
 ["PuOr" 5 "div" 3 2 "F" 253 184 99 "Diverging"]
 ["PuOr" 5 "div" 3 3 "H" 247 247 247 "Diverging"]
 ["PuOr" 5 "div" 3 4 "J" 178 171 210 "Diverging"]
 ["PuOr" 5 "div" 3 5 "M" 94 60 153 "Diverging"]
 ["PuOr" 6 "div" 3.5 1 "B" 179 88 6 "Diverging"]
 ["PuOr" 6 "div" 3.5 2 "E" 241 163 64 "Diverging"]
 ["PuOr" 6 "div" 3.5 3 "G" 254 224 182 "Diverging"]
 ["PuOr" 6 "div" 3.5 4 "I" 216 218 235 "Diverging"]
 ["PuOr" 6 "div" 3.5 5 "K" 153 142 195 "Diverging"]
 ["PuOr" 6 "div" 3.5 6 "N" 84 39 136 "Diverging"]
 ["PuOr" 7 "div" 4 1 "B" 179 88 6 "Diverging"]
 ["PuOr" 7 "div" 4 2 "E" 241 163 64 "Diverging"]
 ["PuOr" 7 "div" 4 3 "G" 254 224 182 "Diverging"]
 ["PuOr" 7 "div" 4 4 "H" 247 247 247 "Diverging"]
 ["PuOr" 7 "div" 4 5 "I" 216 218 235 "Diverging"]
 ["PuOr" 7 "div" 4 6 "K" 153 142 195 "Diverging"]
 ["PuOr" 7 "div" 4 7 "N" 84 39 136 "Diverging"]
 ["PuOr" 8 "div" 4.5 1 "B" 179 88 6 "Diverging"]
 ["PuOr" 8 "div" 4.5 2 "D" 224 130 20 "Diverging"]
 ["PuOr" 8 "div" 4.5 3 "F" 253 184 99 "Diverging"]
 ["PuOr" 8 "div" 4.5 4 "G" 254 224 182 "Diverging"]
 ["PuOr" 8 "div" 4.5 5 "I" 216 218 235 "Diverging"]
 ["PuOr" 8 "div" 4.5 6 "J" 178 171 210 "Diverging"]
 ["PuOr" 8 "div" 4.5 7 "L" 128 115 172 "Diverging"]
 ["PuOr" 8 "div" 4.5 8 "N" 84 39 136 "Diverging"]
 ["PuOr" 9 "div" 5 1 "B" 179 88 6 "Diverging"]
 ["PuOr" 9 "div" 5 2 "D" 224 130 20 "Diverging"]
 ["PuOr" 9 "div" 5 3 "F" 253 184 99 "Diverging"]
 ["PuOr" 9 "div" 5 4 "G" 254 224 182 "Diverging"]
 ["PuOr" 9 "div" 5 5 "H" 247 247 247 "Diverging"]
 ["PuOr" 9 "div" 5 6 "I" 216 218 235 "Diverging"]
 ["PuOr" 9 "div" 5 7 "J" 178 171 210 "Diverging"]
 ["PuOr" 9 "div" 5 8 "L" 128 115 172 "Diverging"]
 ["PuOr" 9 "div" 5 9 "N" 84 39 136 "Diverging"]
 ["PuOr" 10 "div" 5.5 1 "A" 127 59 8 "Diverging"]
 ["PuOr" 10 "div" 5.5 2 "B" 179 88 6 "Diverging"]
 ["PuOr" 10 "div" 5.5 3 "D" 224 130 20 "Diverging"]
 ["PuOr" 10 "div" 5.5 4 "F" 253 184 99 "Diverging"]
 ["PuOr" 10 "div" 5.5 5 "G" 254 224 182 "Diverging"]
 ["PuOr" 10 "div" 5.5 6 "I" 216 218 235 "Diverging"]
 ["PuOr" 10 "div" 5.5 7 "J" 178 171 210 "Diverging"]
 ["PuOr" 10 "div" 5.5 8 "L" 128 115 172 "Diverging"]
 ["PuOr" 10 "div" 5.5 9 "N" 84 39 136 "Diverging"]
 ["PuOr" 10 "div" 5.5 10 "O" 45 0 75 "Diverging"]
 ["PuOr" 11 "div" 6 1 "A" 127 59 8 "Diverging"]
 ["PuOr" 11 "div" 6 2 "B" 179 88 6 "Diverging"]
 ["PuOr" 11 "div" 6 3 "D" 224 130 20 "Diverging"]
 ["PuOr" 11 "div" 6 4 "F" 253 184 99 "Diverging"]
 ["PuOr" 11 "div" 6 5 "G" 254 224 182 "Diverging"]
 ["PuOr" 11 "div" 6 6 "H" 247 247 247 "Diverging"]
 ["PuOr" 11 "div" 6 7 "I" 216 218 235 "Diverging"]
 ["PuOr" 11 "div" 6 8 "J" 178 171 210 "Diverging"]
 ["PuOr" 11 "div" 6 9 "L" 128 115 172 "Diverging"]
 ["PuOr" 11 "div" 6 10 "N" 84 39 136 "Diverging"]
 ["PuOr" 11 "div" 6 11 "O" 45 0 75 "Diverging"]
 ["PuRd" 3 "seq" nil 1 "C" 231 225 239 "Sequential"]
 ["PuRd" 3 "seq" nil 2 "F" 201 148 199 "Sequential"]
 ["PuRd" 3 "seq" nil 3 "I" 221 28 119 "Sequential"]
 ["PuRd" 4 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuRd" 4 "seq" nil 2 "E" 215 181 216 "Sequential"]
 ["PuRd" 4 "seq" nil 3 "G" 223 101 176 "Sequential"]
 ["PuRd" 4 "seq" nil 4 "J" 206 18 86 "Sequential"]
 ["PuRd" 5 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuRd" 5 "seq" nil 2 "E" 215 181 216 "Sequential"]
 ["PuRd" 5 "seq" nil 3 "G" 223 101 176 "Sequential"]
 ["PuRd" 5 "seq" nil 4 "I" 221 28 119 "Sequential"]
 ["PuRd" 5 "seq" nil 5 "K" 152 0 67 "Sequential"]
 ["PuRd" 6 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuRd" 6 "seq" nil 2 "D" 212 185 218 "Sequential"]
 ["PuRd" 6 "seq" nil 3 "F" 201 148 199 "Sequential"]
 ["PuRd" 6 "seq" nil 4 "G" 223 101 176 "Sequential"]
 ["PuRd" 6 "seq" nil 5 "I" 221 28 119 "Sequential"]
 ["PuRd" 6 "seq" nil 6 "K" 152 0 67 "Sequential"]
 ["PuRd" 7 "seq" nil 1 "B" 241 238 246 "Sequential"]
 ["PuRd" 7 "seq" nil 2 "D" 212 185 218 "Sequential"]
 ["PuRd" 7 "seq" nil 3 "F" 201 148 199 "Sequential"]
 ["PuRd" 7 "seq" nil 4 "G" 223 101 176 "Sequential"]
 ["PuRd" 7 "seq" nil 5 "H" 231 41 138 "Sequential"]
 ["PuRd" 7 "seq" nil 6 "J" 206 18 86 "Sequential"]
 ["PuRd" 7 "seq" nil 7 "L" 145 0 63 "Sequential"]
 ["PuRd" 8 "seq" nil 1 "A" 247 244 249 "Sequential"]
 ["PuRd" 8 "seq" nil 2 "C" 231 225 239 "Sequential"]
 ["PuRd" 8 "seq" nil 3 "D" 212 185 218 "Sequential"]
 ["PuRd" 8 "seq" nil 4 "F" 201 148 199 "Sequential"]
 ["PuRd" 8 "seq" nil 5 "G" 223 101 176 "Sequential"]
 ["PuRd" 8 "seq" nil 6 "H" 231 41 138 "Sequential"]
 ["PuRd" 8 "seq" nil 7 "J" 206 18 86 "Sequential"]
 ["PuRd" 8 "seq" nil 8 "L" 145 0 63 "Sequential"]
 ["PuRd" 9 "seq" nil 1 "A" 247 244 249 "Sequential"]
 ["PuRd" 9 "seq" nil 2 "C" 231 225 239 "Sequential"]
 ["PuRd" 9 "seq" nil 3 "D" 212 185 218 "Sequential"]
 ["PuRd" 9 "seq" nil 4 "F" 201 148 199 "Sequential"]
 ["PuRd" 9 "seq" nil 5 "G" 223 101 176 "Sequential"]
 ["PuRd" 9 "seq" nil 6 "H" 231 41 138 "Sequential"]
 ["PuRd" 9 "seq" nil 7 "J" 206 18 86 "Sequential"]
 ["PuRd" 9 "seq" nil 8 "K" 152 0 67 "Sequential"]
 ["PuRd" 9 "seq" nil 9 "M" 103 0 31 "Sequential"]
 ["Purples" 3 "seq" nil 1 "C" 239 237 245 "Sequential"]
 ["Purples" 3 "seq" nil 2 "F" 188 189 220 "Sequential"]
 ["Purples" 3 "seq" nil 3 "I" 117 107 177 "Sequential"]
 ["Purples" 4 "seq" nil 1 "B" 242 240 247 "Sequential"]
 ["Purples" 4 "seq" nil 2 "E" 203 201 226 "Sequential"]
 ["Purples" 4 "seq" nil 3 "G" 158 154 200 "Sequential"]
 ["Purples" 4 "seq" nil 4 "J" 106 81 163 "Sequential"]
 ["Purples" 5 "seq" nil 1 "B" 242 240 247 "Sequential"]
 ["Purples" 5 "seq" nil 2 "E" 203 201 226 "Sequential"]
 ["Purples" 5 "seq" nil 3 "G" 158 154 200 "Sequential"]
 ["Purples" 5 "seq" nil 4 "I" 117 107 177 "Sequential"]
 ["Purples" 5 "seq" nil 5 "K" 84 39 143 "Sequential"]
 ["Purples" 6 "seq" nil 1 "B" 242 240 247 "Sequential"]
 ["Purples" 6 "seq" nil 2 "D" 218 218 235 "Sequential"]
 ["Purples" 6 "seq" nil 3 "F" 188 189 220 "Sequential"]
 ["Purples" 6 "seq" nil 4 "G" 158 154 200 "Sequential"]
 ["Purples" 6 "seq" nil 5 "I" 117 107 177 "Sequential"]
 ["Purples" 6 "seq" nil 6 "K" 84 39 143 "Sequential"]
 ["Purples" 7 "seq" nil 1 "B" 242 240 247 "Sequential"]
 ["Purples" 7 "seq" nil 2 "D" 218 218 235 "Sequential"]
 ["Purples" 7 "seq" nil 3 "F" 188 189 220 "Sequential"]
 ["Purples" 7 "seq" nil 4 "G" 158 154 200 "Sequential"]
 ["Purples" 7 "seq" nil 5 "H" 128 125 186 "Sequential"]
 ["Purples" 7 "seq" nil 6 "J" 106 81 163 "Sequential"]
 ["Purples" 7 "seq" nil 7 "L" 74 20 134 "Sequential"]
 ["Purples" 8 "seq" nil 1 "A" 252 251 253 "Sequential"]
 ["Purples" 8 "seq" nil 2 "C" 239 237 245 "Sequential"]
 ["Purples" 8 "seq" nil 3 "D" 218 218 235 "Sequential"]
 ["Purples" 8 "seq" nil 4 "F" 188 189 220 "Sequential"]
 ["Purples" 8 "seq" nil 5 "G" 158 154 200 "Sequential"]
 ["Purples" 8 "seq" nil 6 "H" 128 125 186 "Sequential"]
 ["Purples" 8 "seq" nil 7 "J" 106 81 163 "Sequential"]
 ["Purples" 8 "seq" nil 8 "L" 74 20 134 "Sequential"]
 ["Purples" 9 "seq" nil 1 "A" 252 251 253 "Sequential"]
 ["Purples" 9 "seq" nil 2 "C" 239 237 245 "Sequential"]
 ["Purples" 9 "seq" nil 3 "D" 218 218 235 "Sequential"]
 ["Purples" 9 "seq" nil 4 "F" 188 189 220 "Sequential"]
 ["Purples" 9 "seq" nil 5 "G" 158 154 200 "Sequential"]
 ["Purples" 9 "seq" nil 6 "H" 128 125 186 "Sequential"]
 ["Purples" 9 "seq" nil 7 "J" 106 81 163 "Sequential"]
 ["Purples" 9 "seq" nil 8 "K" 84 39 143 "Sequential"]
 ["Purples" 9 "seq" nil 9 "M" 63 0 125 "Sequential"]
 ["RdBu" 3 "div" 2 1 "E" 239 138 98 "Diverging"]
 ["RdBu" 3 "div" 2 2 "H" 247 247 247 "Diverging"]
 ["RdBu" 3 "div" 2 3 "K" 103 169 207 "Diverging"]
 ["RdBu" 4 "div" 2.5 1 "C" 202 0 32 "Diverging"]
 ["RdBu" 4 "div" 2.5 2 "F" 244 165 130 "Diverging"]
 ["RdBu" 4 "div" 2.5 3 "J" 146 197 222 "Diverging"]
 ["RdBu" 4 "div" 2.5 4 "M" 5 113 176 "Diverging"]
 ["RdBu" 5 "div" 3 1 "C" 202 0 32 "Diverging"]
 ["RdBu" 5 "div" 3 2 "F" 244 165 130 "Diverging"]
 ["RdBu" 5 "div" 3 3 "H" 247 247 247 "Diverging"]
 ["RdBu" 5 "div" 3 4 "J" 146 197 222 "Diverging"]
 ["RdBu" 5 "div" 3 5 "M" 5 113 176 "Diverging"]
 ["RdBu" 6 "div" 3.5 1 "B" 178 24 43 "Diverging"]
 ["RdBu" 6 "div" 3.5 2 "E" 239 138 98 "Diverging"]
 ["RdBu" 6 "div" 3.5 3 "G" 253 219 199 "Diverging"]
 ["RdBu" 6 "div" 3.5 4 "I" 209 229 240 "Diverging"]
 ["RdBu" 6 "div" 3.5 5 "K" 103 169 207 "Diverging"]
 ["RdBu" 6 "div" 3.5 6 "N" 33 102 172 "Diverging"]
 ["RdBu" 7 "div" 4 1 "B" 178 24 43 "Diverging"]
 ["RdBu" 7 "div" 4 2 "E" 239 138 98 "Diverging"]
 ["RdBu" 7 "div" 4 3 "G" 253 219 199 "Diverging"]
 ["RdBu" 7 "div" 4 4 "H" 247 247 247 "Diverging"]
 ["RdBu" 7 "div" 4 5 "I" 209 229 240 "Diverging"]
 ["RdBu" 7 "div" 4 6 "K" 103 169 207 "Diverging"]
 ["RdBu" 7 "div" 4 7 "N" 33 102 172 "Diverging"]
 ["RdBu" 8 "div" 4.5 1 "B" 178 24 43 "Diverging"]
 ["RdBu" 8 "div" 4.5 2 "D" 214 96 77 "Diverging"]
 ["RdBu" 8 "div" 4.5 3 "F" 244 165 130 "Diverging"]
 ["RdBu" 8 "div" 4.5 4 "G" 253 219 199 "Diverging"]
 ["RdBu" 8 "div" 4.5 5 "I" 209 229 240 "Diverging"]
 ["RdBu" 8 "div" 4.5 6 "J" 146 197 222 "Diverging"]
 ["RdBu" 8 "div" 4.5 7 "L" 67 147 195 "Diverging"]
 ["RdBu" 8 "div" 4.5 8 "N" 33 102 172 "Diverging"]
 ["RdBu" 9 "div" 5 1 "B" 178 24 43 "Diverging"]
 ["RdBu" 9 "div" 5 2 "D" 214 96 77 "Diverging"]
 ["RdBu" 9 "div" 5 3 "F" 244 165 130 "Diverging"]
 ["RdBu" 9 "div" 5 4 "G" 253 219 199 "Diverging"]
 ["RdBu" 9 "div" 5 5 "H" 247 247 247 "Diverging"]
 ["RdBu" 9 "div" 5 6 "I" 209 229 240 "Diverging"]
 ["RdBu" 9 "div" 5 7 "J" 146 197 222 "Diverging"]
 ["RdBu" 9 "div" 5 8 "L" 67 147 195 "Diverging"]
 ["RdBu" 9 "div" 5 9 "N" 33 102 172 "Diverging"]
 ["RdBu" 10 "div" 5.5 1 "A" 103 0 31 "Diverging"]
 ["RdBu" 10 "div" 5.5 2 "B" 178 24 43 "Diverging"]
 ["RdBu" 10 "div" 5.5 3 "D" 214 96 77 "Diverging"]
 ["RdBu" 10 "div" 5.5 4 "F" 244 165 130 "Diverging"]
 ["RdBu" 10 "div" 5.5 5 "G" 253 219 199 "Diverging"]
 ["RdBu" 10 "div" 5.5 6 "I" 209 229 240 "Diverging"]
 ["RdBu" 10 "div" 5.5 7 "J" 146 197 222 "Diverging"]
 ["RdBu" 10 "div" 5.5 8 "L" 67 147 195 "Diverging"]
 ["RdBu" 10 "div" 5.5 9 "N" 33 102 172 "Diverging"]
 ["RdBu" 10 "div" 5.5 10 "O" 5 48 97 "Diverging"]
 ["RdBu" 11 "div" 6 1 "A" 103 0 31 "Diverging"]
 ["RdBu" 11 "div" 6 2 "B" 178 24 43 "Diverging"]
 ["RdBu" 11 "div" 6 3 "D" 214 96 77 "Diverging"]
 ["RdBu" 11 "div" 6 4 "F" 244 165 130 "Diverging"]
 ["RdBu" 11 "div" 6 5 "G" 253 219 199 "Diverging"]
 ["RdBu" 11 "div" 6 6 "H" 247 247 247 "Diverging"]
 ["RdBu" 11 "div" 6 7 "I" 209 229 240 "Diverging"]
 ["RdBu" 11 "div" 6 8 "J" 146 197 222 "Diverging"]
 ["RdBu" 11 "div" 6 9 "L" 67 147 195 "Diverging"]
 ["RdBu" 11 "div" 6 10 "N" 33 102 172 "Diverging"]
 ["RdBu" 11 "div" 6 11 "O" 5 48 97 "Diverging"]
 ["RdGy" 3 "div" 2 1 "E" 239 138 98 "Diverging"]
 ["RdGy" 3 "div" 2 2 "H" 255 255 255 "Diverging"]
 ["RdGy" 3 "div" 2 3 "K" 153 153 153 "Diverging"]
 ["RdGy" 4 "div" 2.5 1 "C" 202 0 32 "Diverging"]
 ["RdGy" 4 "div" 2.5 2 "F" 244 165 130 "Diverging"]
 ["RdGy" 4 "div" 2.5 3 "J" 186 186 186 "Diverging"]
 ["RdGy" 4 "div" 2.5 4 "M" 64 64 64 "Diverging"]
 ["RdGy" 5 "div" 3 1 "C" 202 0 32 "Diverging"]
 ["RdGy" 5 "div" 3 2 "F" 244 165 130 "Diverging"]
 ["RdGy" 5 "div" 3 3 "H" 255 255 255 "Diverging"]
 ["RdGy" 5 "div" 3 4 "J" 186 186 186 "Diverging"]
 ["RdGy" 5 "div" 3 5 "M" 64 64 64 "Diverging"]
 ["RdGy" 6 "div" 3.5 1 "B" 178 24 43 "Diverging"]
 ["RdGy" 6 "div" 3.5 2 "E" 239 138 98 "Diverging"]
 ["RdGy" 6 "div" 3.5 3 "G" 253 219 199 "Diverging"]
 ["RdGy" 6 "div" 3.5 4 "I" 224 224 224 "Diverging"]
 ["RdGy" 6 "div" 3.5 5 "K" 153 153 153 "Diverging"]
 ["RdGy" 6 "div" 3.5 6 "N" 77 77 77 "Diverging"]
 ["RdGy" 7 "div" 4 1 "B" 178 24 43 "Diverging"]
 ["RdGy" 7 "div" 4 2 "E" 239 138 98 "Diverging"]
 ["RdGy" 7 "div" 4 3 "G" 253 219 199 "Diverging"]
 ["RdGy" 7 "div" 4 4 "H" 255 255 255 "Diverging"]
 ["RdGy" 7 "div" 4 5 "I" 224 224 224 "Diverging"]
 ["RdGy" 7 "div" 4 6 "K" 153 153 153 "Diverging"]
 ["RdGy" 7 "div" 4 7 "N" 77 77 77 "Diverging"]
 ["RdGy" 8 "div" 4.5 1 "B" 178 24 43 "Diverging"]
 ["RdGy" 8 "div" 4.5 2 "D" 214 96 77 "Diverging"]
 ["RdGy" 8 "div" 4.5 3 "F" 244 165 130 "Diverging"]
 ["RdGy" 8 "div" 4.5 4 "G" 253 219 199 "Diverging"]
 ["RdGy" 8 "div" 4.5 5 "I" 224 224 224 "Diverging"]
 ["RdGy" 8 "div" 4.5 6 "J" 186 186 186 "Diverging"]
 ["RdGy" 8 "div" 4.5 7 "L" 135 135 135 "Diverging"]
 ["RdGy" 8 "div" 4.5 8 "N" 77 77 77 "Diverging"]
 ["RdGy" 9 "div" 5 1 "B" 178 24 43 "Diverging"]
 ["RdGy" 9 "div" 5 2 "D" 214 96 77 "Diverging"]
 ["RdGy" 9 "div" 5 3 "F" 244 165 130 "Diverging"]
 ["RdGy" 9 "div" 5 4 "G" 253 219 199 "Diverging"]
 ["RdGy" 9 "div" 5 5 "H" 255 255 255 "Diverging"]
 ["RdGy" 9 "div" 5 6 "I" 224 224 224 "Diverging"]
 ["RdGy" 9 "div" 5 7 "J" 186 186 186 "Diverging"]
 ["RdGy" 9 "div" 5 8 "L" 135 135 135 "Diverging"]
 ["RdGy" 9 "div" 5 9 "N" 77 77 77 "Diverging"]
 ["RdGy" 10 "div" 5.5 1 "A" 103 0 31 "Diverging"]
 ["RdGy" 10 "div" 5.5 2 "B" 178 24 43 "Diverging"]
 ["RdGy" 10 "div" 5.5 3 "D" 214 96 77 "Diverging"]
 ["RdGy" 10 "div" 5.5 4 "F" 244 165 130 "Diverging"]
 ["RdGy" 10 "div" 5.5 5 "G" 253 219 199 "Diverging"]
 ["RdGy" 10 "div" 5.5 6 "I" 224 224 224 "Diverging"]
 ["RdGy" 10 "div" 5.5 7 "J" 186 186 186 "Diverging"]
 ["RdGy" 10 "div" 5.5 8 "L" 135 135 135 "Diverging"]
 ["RdGy" 10 "div" 5.5 9 "N" 77 77 77 "Diverging"]
 ["RdGy" 10 "div" 5.5 10 "O" 26 26 26 "Diverging"]
 ["RdGy" 11 "div" 6 1 "A" 103 0 31 "Diverging"]
 ["RdGy" 11 "div" 6 2 "B" 178 24 43 "Diverging"]
 ["RdGy" 11 "div" 6 3 "D" 214 96 77 "Diverging"]
 ["RdGy" 11 "div" 6 4 "F" 244 165 130 "Diverging"]
 ["RdGy" 11 "div" 6 5 "G" 253 219 199 "Diverging"]
 ["RdGy" 11 "div" 6 6 "H" 255 255 255 "Diverging"]
 ["RdGy" 11 "div" 6 7 "I" 224 224 224 "Diverging"]
 ["RdGy" 11 "div" 6 8 "J" 186 186 186 "Diverging"]
 ["RdGy" 11 "div" 6 9 "L" 135 135 135 "Diverging"]
 ["RdGy" 11 "div" 6 10 "N" 77 77 77 "Diverging"]
 ["RdGy" 11 "div" 6 11 "O" 26 26 26 "Diverging"]
 ["RdPu" 3 "seq" nil 1 "C" 253 224 221 "Sequential"]
 ["RdPu" 3 "seq" nil 2 "F" 250 159 181 "Sequential"]
 ["RdPu" 3 "seq" nil 3 "I" 197 27 138 "Sequential"]
 ["RdPu" 4 "seq" nil 1 "B" 254 235 226 "Sequential"]
 ["RdPu" 4 "seq" nil 2 "E" 251 180 185 "Sequential"]
 ["RdPu" 4 "seq" nil 3 "G" 247 104 161 "Sequential"]
 ["RdPu" 4 "seq" nil 4 "J" 174 1 126 "Sequential"]
 ["RdPu" 5 "seq" nil 1 "B" 254 235 226 "Sequential"]
 ["RdPu" 5 "seq" nil 2 "E" 251 180 185 "Sequential"]
 ["RdPu" 5 "seq" nil 3 "G" 247 104 161 "Sequential"]
 ["RdPu" 5 "seq" nil 4 "I" 197 27 138 "Sequential"]
 ["RdPu" 5 "seq" nil 5 "K" 122 1 119 "Sequential"]
 ["RdPu" 6 "seq" nil 1 "B" 254 235 226 "Sequential"]
 ["RdPu" 6 "seq" nil 2 "D" 252 197 192 "Sequential"]
 ["RdPu" 6 "seq" nil 3 "F" 250 159 181 "Sequential"]
 ["RdPu" 6 "seq" nil 4 "G" 247 104 161 "Sequential"]
 ["RdPu" 6 "seq" nil 5 "I" 197 27 138 "Sequential"]
 ["RdPu" 6 "seq" nil 6 "K" 122 1 119 "Sequential"]
 ["RdPu" 7 "seq" nil 1 "B" 254 235 226 "Sequential"]
 ["RdPu" 7 "seq" nil 2 "D" 252 197 192 "Sequential"]
 ["RdPu" 7 "seq" nil 3 "F" 250 159 181 "Sequential"]
 ["RdPu" 7 "seq" nil 4 "G" 247 104 161 "Sequential"]
 ["RdPu" 7 "seq" nil 5 "H" 221 52 151 "Sequential"]
 ["RdPu" 7 "seq" nil 6 "J" 174 1 126 "Sequential"]
 ["RdPu" 7 "seq" nil 7 "L" 122 1 119 "Sequential"]
 ["RdPu" 8 "seq" nil 1 "A" 255 247 243 "Sequential"]
 ["RdPu" 8 "seq" nil 2 "C" 253 224 221 "Sequential"]
 ["RdPu" 8 "seq" nil 3 "D" 252 197 192 "Sequential"]
 ["RdPu" 8 "seq" nil 4 "F" 250 159 181 "Sequential"]
 ["RdPu" 8 "seq" nil 5 "G" 247 104 161 "Sequential"]
 ["RdPu" 8 "seq" nil 6 "H" 221 52 151 "Sequential"]
 ["RdPu" 8 "seq" nil 7 "J" 174 1 126 "Sequential"]
 ["RdPu" 8 "seq" nil 8 "L" 122 1 119 "Sequential"]
 ["RdPu" 9 "seq" nil 1 "A" 255 247 243 "Sequential"]
 ["RdPu" 9 "seq" nil 2 "C" 253 224 221 "Sequential"]
 ["RdPu" 9 "seq" nil 3 "D" 252 197 192 "Sequential"]
 ["RdPu" 9 "seq" nil 4 "F" 250 159 181 "Sequential"]
 ["RdPu" 9 "seq" nil 5 "G" 247 104 161 "Sequential"]
 ["RdPu" 9 "seq" nil 6 "H" 221 52 151 "Sequential"]
 ["RdPu" 9 "seq" nil 7 "J" 174 1 126 "Sequential"]
 ["RdPu" 9 "seq" nil 8 "K" 122 1 119 "Sequential"]
 ["RdPu" 9 "seq" nil 9 "M" 73 0 106 "Sequential"]
 ["Reds" 3 "seq" nil 1 "C" 254 224 210 "Sequential"]
 ["Reds" 3 "seq" nil 2 "F" 252 146 114 "Sequential"]
 ["Reds" 3 "seq" nil 3 "I" 222 45 38 "Sequential"]
 ["Reds" 4 "seq" nil 1 "B" 254 229 217 "Sequential"]
 ["Reds" 4 "seq" nil 2 "E" 252 174 145 "Sequential"]
 ["Reds" 4 "seq" nil 3 "G" 251 106 74 "Sequential"]
 ["Reds" 4 "seq" nil 4 "J" 203 24 29 "Sequential"]
 ["Reds" 5 "seq" nil 1 "B" 254 229 217 "Sequential"]
 ["Reds" 5 "seq" nil 2 "E" 252 174 145 "Sequential"]
 ["Reds" 5 "seq" nil 3 "G" 251 106 74 "Sequential"]
 ["Reds" 5 "seq" nil 4 "I" 222 45 38 "Sequential"]
 ["Reds" 5 "seq" nil 5 "K" 165 15 21 "Sequential"]
 ["Reds" 6 "seq" nil 1 "B" 254 229 217 "Sequential"]
 ["Reds" 6 "seq" nil 2 "D" 252 187 161 "Sequential"]
 ["Reds" 6 "seq" nil 3 "F" 252 146 114 "Sequential"]
 ["Reds" 6 "seq" nil 4 "G" 251 106 74 "Sequential"]
 ["Reds" 6 "seq" nil 5 "I" 222 45 38 "Sequential"]
 ["Reds" 6 "seq" nil 6 "K" 165 15 21 "Sequential"]
 ["Reds" 7 "seq" nil 1 "B" 254 229 217 "Sequential"]
 ["Reds" 7 "seq" nil 2 "D" 252 187 161 "Sequential"]
 ["Reds" 7 "seq" nil 3 "F" 252 146 114 "Sequential"]
 ["Reds" 7 "seq" nil 4 "G" 251 106 74 "Sequential"]
 ["Reds" 7 "seq" nil 5 "H" 239 59 44 "Sequential"]
 ["Reds" 7 "seq" nil 6 "J" 203 24 29 "Sequential"]
 ["Reds" 7 "seq" nil 7 "L" 153 0 13 "Sequential"]
 ["Reds" 8 "seq" nil 1 "A" 255 245 240 "Sequential"]
 ["Reds" 8 "seq" nil 2 "C" 254 224 210 "Sequential"]
 ["Reds" 8 "seq" nil 3 "D" 252 187 161 "Sequential"]
 ["Reds" 8 "seq" nil 4 "F" 252 146 114 "Sequential"]
 ["Reds" 8 "seq" nil 5 "G" 251 106 74 "Sequential"]
 ["Reds" 8 "seq" nil 6 "H" 239 59 44 "Sequential"]
 ["Reds" 8 "seq" nil 7 "J" 203 24 29 "Sequential"]
 ["Reds" 8 "seq" nil 8 "L" 153 0 13 "Sequential"]
 ["Reds" 9 "seq" nil 1 "A" 255 245 240 "Sequential"]
 ["Reds" 9 "seq" nil 2 "C" 254 224 210 "Sequential"]
 ["Reds" 9 "seq" nil 3 "D" 252 187 161 "Sequential"]
 ["Reds" 9 "seq" nil 4 "F" 252 146 114 "Sequential"]
 ["Reds" 9 "seq" nil 5 "G" 251 106 74 "Sequential"]
 ["Reds" 9 "seq" nil 6 "H" 239 59 44 "Sequential"]
 ["Reds" 9 "seq" nil 7 "J" 203 24 29 "Sequential"]
 ["Reds" 9 "seq" nil 8 "K" 165 15 21 "Sequential"]
 ["Reds" 9 "seq" nil 9 "M" 103 0 13 "Sequential"]
 ["RdYlBu" 3 "div" 2 1 "E" 252 141 89 "Diverging"]
 ["RdYlBu" 3 "div" 2 2 "H" 255 255 191 "Diverging"]
 ["RdYlBu" 3 "div" 2 3 "K" 145 191 219 "Diverging"]
 ["RdYlBu" 4 "div" 2.5 1 "C" 215 25 28 "Diverging"]
 ["RdYlBu" 4 "div" 2.5 2 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 4 "div" 2.5 3 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 4 "div" 2.5 4 "M" 44 123 182 "Diverging"]
 ["RdYlBu" 5 "div" 3 1 "C" 215 25 28 "Diverging"]
 ["RdYlBu" 5 "div" 3 2 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 5 "div" 3 3 "H" 255 255 191 "Diverging"]
 ["RdYlBu" 5 "div" 3 4 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 5 "div" 3 5 "M" 44 123 182 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 1 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 2 "E" 252 141 89 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 3 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 4 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 5 "K" 145 191 219 "Diverging"]
 ["RdYlBu" 6 "div" 3.5 6 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 7 "div" 4 1 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 7 "div" 4 2 "E" 252 141 89 "Diverging"]
 ["RdYlBu" 7 "div" 4 3 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 7 "div" 4 4 "H" 255 255 191 "Diverging"]
 ["RdYlBu" 7 "div" 4 5 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 7 "div" 4 6 "K" 145 191 219 "Diverging"]
 ["RdYlBu" 7 "div" 4 7 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 1 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 2 "D" 244 109 67 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 3 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 4 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 5 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 6 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 7 "L" 116 173 209 "Diverging"]
 ["RdYlBu" 8 "div" 4.5 8 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 9 "div" 5 1 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 9 "div" 5 2 "D" 244 109 67 "Diverging"]
 ["RdYlBu" 9 "div" 5 3 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 9 "div" 5 4 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 9 "div" 5 5 "H" 255 255 191 "Diverging"]
 ["RdYlBu" 9 "div" 5 6 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 9 "div" 5 7 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 9 "div" 5 8 "L" 116 173 209 "Diverging"]
 ["RdYlBu" 9 "div" 5 9 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 1 "A" 165 0 38 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 2 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 3 "D" 244 109 67 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 4 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 5 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 6 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 7 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 8 "L" 116 173 209 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 9 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 10 "div" 5.5 10 "O" 49 54 149 "Diverging"]
 ["RdYlBu" 11 "div" 6 1 "A" 165 0 38 "Diverging"]
 ["RdYlBu" 11 "div" 6 2 "B" 215 48 39 "Diverging"]
 ["RdYlBu" 11 "div" 6 3 "D" 244 109 67 "Diverging"]
 ["RdYlBu" 11 "div" 6 4 "F" 253 174 97 "Diverging"]
 ["RdYlBu" 11 "div" 6 5 "G" 254 224 144 "Diverging"]
 ["RdYlBu" 11 "div" 6 6 "H" 255 255 191 "Diverging"]
 ["RdYlBu" 11 "div" 6 7 "I" 224 243 248 "Diverging"]
 ["RdYlBu" 11 "div" 6 8 "J" 171 217 233 "Diverging"]
 ["RdYlBu" 11 "div" 6 9 "L" 116 173 209 "Diverging"]
 ["RdYlBu" 11 "div" 6 10 "N" 69 117 180 "Diverging"]
 ["RdYlBu" 11 "div" 6 11 "O" 49 54 149 "Diverging"]
 ["RdYlGn" 3 "div" 2 1 "E" 252 141 89 "Diverging"]
 ["RdYlGn" 3 "div" 2 2 "H" 255 255 191 "Diverging"]
 ["RdYlGn" 3 "div" 2 3 "K" 145 207 96 "Diverging"]
 ["RdYlGn" 4 "div" 2.5 1 "C" 215 25 28 "Diverging"]
 ["RdYlGn" 4 "div" 2.5 2 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 4 "div" 2.5 3 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 4 "div" 2.5 4 "M" 26 150 65 "Diverging"]
 ["RdYlGn" 5 "div" 3 1 "C" 215 25 28 "Diverging"]
 ["RdYlGn" 5 "div" 3 2 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 5 "div" 3 3 "H" 255 255 191 "Diverging"]
 ["RdYlGn" 5 "div" 3 4 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 5 "div" 3 5 "M" 26 150 65 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 1 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 2 "E" 252 141 89 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 3 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 4 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 5 "K" 145 207 96 "Diverging"]
 ["RdYlGn" 6 "div" 3.5 6 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 7 "div" 4 1 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 7 "div" 4 2 "E" 252 141 89 "Diverging"]
 ["RdYlGn" 7 "div" 4 3 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 7 "div" 4 4 "H" 255 255 191 "Diverging"]
 ["RdYlGn" 7 "div" 4 5 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 7 "div" 4 6 "K" 145 207 96 "Diverging"]
 ["RdYlGn" 7 "div" 4 7 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 1 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 2 "D" 244 109 67 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 3 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 4 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 5 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 6 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 7 "L" 102 189 99 "Diverging"]
 ["RdYlGn" 8 "div" 4.5 8 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 9 "div" 5 1 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 9 "div" 5 2 "D" 244 109 67 "Diverging"]
 ["RdYlGn" 9 "div" 5 3 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 9 "div" 5 4 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 9 "div" 5 5 "H" 255 255 191 "Diverging"]
 ["RdYlGn" 9 "div" 5 6 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 9 "div" 5 7 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 9 "div" 5 8 "L" 102 189 99 "Diverging"]
 ["RdYlGn" 9 "div" 5 9 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 1 "A" 165 0 38 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 2 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 3 "D" 244 109 67 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 4 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 5 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 6 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 7 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 8 "L" 102 189 99 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 9 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 10 "div" 5.5 10 "O" 0 104 55 "Diverging"]
 ["RdYlGn" 11 "div" 6 1 "A" 165 0 38 "Diverging"]
 ["RdYlGn" 11 "div" 6 2 "B" 215 48 39 "Diverging"]
 ["RdYlGn" 11 "div" 6 3 "D" 244 109 67 "Diverging"]
 ["RdYlGn" 11 "div" 6 4 "F" 253 174 97 "Diverging"]
 ["RdYlGn" 11 "div" 6 5 "G" 254 224 139 "Diverging"]
 ["RdYlGn" 11 "div" 6 6 "H" 255 255 191 "Diverging"]
 ["RdYlGn" 11 "div" 6 7 "I" 217 239 139 "Diverging"]
 ["RdYlGn" 11 "div" 6 8 "J" 166 217 106 "Diverging"]
 ["RdYlGn" 11 "div" 6 9 "L" 102 189 99 "Diverging"]
 ["RdYlGn" 11 "div" 6 10 "N" 26 152 80 "Diverging"]
 ["RdYlGn" 11 "div" 6 11 "O" 0 104 55 "Diverging"]
 ["Set1" 3 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 3 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 3 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 4 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 4 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 4 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 4 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 5 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 5 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 5 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 5 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 5 "qual" nil 5 "E" 255 127 0 "Qualitative"]
 ["Set1" 6 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 6 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 6 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 6 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 6 "qual" nil 5 "E" 255 127 0 "Qualitative"]
 ["Set1" 6 "qual" nil 6 "F" 255 255 51 "Qualitative"]
 ["Set1" 7 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 7 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 7 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 7 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 7 "qual" nil 5 "E" 255 127 0 "Qualitative"]
 ["Set1" 7 "qual" nil 6 "F" 255 255 51 "Qualitative"]
 ["Set1" 7 "qual" nil 7 "G" 166 86 40 "Qualitative"]
 ["Set1" 8 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 8 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 8 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 8 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 8 "qual" nil 5 "E" 255 127 0 "Qualitative"]
 ["Set1" 8 "qual" nil 6 "F" 255 255 51 "Qualitative"]
 ["Set1" 8 "qual" nil 7 "G" 166 86 40 "Qualitative"]
 ["Set1" 8 "qual" nil 8 "H" 247 129 191 "Qualitative"]
 ["Set1" 9 "qual" nil 1 "A" 228 26 28 "Qualitative"]
 ["Set1" 9 "qual" nil 2 "B" 55 126 184 "Qualitative"]
 ["Set1" 9 "qual" nil 3 "C" 77 175 74 "Qualitative"]
 ["Set1" 9 "qual" nil 4 "D" 152 78 163 "Qualitative"]
 ["Set1" 9 "qual" nil 5 "E" 255 127 0 "Qualitative"]
 ["Set1" 9 "qual" nil 6 "F" 255 255 51 "Qualitative"]
 ["Set1" 9 "qual" nil 7 "G" 166 86 40 "Qualitative"]
 ["Set1" 9 "qual" nil 8 "H" 247 129 191 "Qualitative"]
 ["Set1" 9 "qual" nil 9 "I" 153 153 153 "Qualitative"]
 ["Set2" 3 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 3 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 3 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 4 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 4 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 4 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 4 "qual" nil 4 "D" 231 138 195 "Qualitative"]
 ["Set2" 5 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 5 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 5 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 5 "qual" nil 4 "D" 231 138 195 "Qualitative"]
 ["Set2" 5 "qual" nil 5 "E" 166 216 84 "Qualitative"]
 ["Set2" 6 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 6 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 6 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 6 "qual" nil 4 "D" 231 138 195 "Qualitative"]
 ["Set2" 6 "qual" nil 5 "E" 166 216 84 "Qualitative"]
 ["Set2" 6 "qual" nil 6 "F" 255 217 47 "Qualitative"]
 ["Set2" 7 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 7 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 7 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 7 "qual" nil 4 "D" 231 138 195 "Qualitative"]
 ["Set2" 7 "qual" nil 5 "E" 166 216 84 "Qualitative"]
 ["Set2" 7 "qual" nil 6 "F" 255 217 47 "Qualitative"]
 ["Set2" 7 "qual" nil 7 "G" 229 196 148 "Qualitative"]
 ["Set2" 8 "qual" nil 1 "A" 102 194 165 "Qualitative"]
 ["Set2" 8 "qual" nil 2 "B" 252 141 98 "Qualitative"]
 ["Set2" 8 "qual" nil 3 "C" 141 160 203 "Qualitative"]
 ["Set2" 8 "qual" nil 4 "D" 231 138 195 "Qualitative"]
 ["Set2" 8 "qual" nil 5 "E" 166 216 84 "Qualitative"]
 ["Set2" 8 "qual" nil 6 "F" 255 217 47 "Qualitative"]
 ["Set2" 8 "qual" nil 7 "G" 229 196 148 "Qualitative"]
 ["Set2" 8 "qual" nil 8 "H" 179 179 179 "Qualitative"]
 ["Set3" 3 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 3 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 3 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 4 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 4 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 4 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 4 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 5 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 5 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 5 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 5 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 5 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 6 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 6 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 6 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 6 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 6 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 6 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 7 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 7 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 7 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 7 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 7 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 7 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 7 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 8 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 8 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 8 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 8 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 8 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 8 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 8 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 8 "qual" nil 8 "H" 252 205 229 "Qualitative"]
 ["Set3" 9 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 9 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 9 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 9 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 9 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 9 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 9 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 9 "qual" nil 8 "H" 252 205 229 "Qualitative"]
 ["Set3" 9 "qual" nil 9 "I" 217 217 217 "Qualitative"]
 ["Set3" 10 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 10 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 10 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 10 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 10 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 10 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 10 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 10 "qual" nil 8 "H" 252 205 229 "Qualitative"]
 ["Set3" 10 "qual" nil 9 "I" 217 217 217 "Qualitative"]
 ["Set3" 10 "qual" nil 10 "J" 188 128 189 "Qualitative"]
 ["Set3" 11 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 11 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 11 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 11 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 11 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 11 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 11 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 11 "qual" nil 8 "H" 252 205 229 "Qualitative"]
 ["Set3" 11 "qual" nil 9 "I" 217 217 217 "Qualitative"]
 ["Set3" 11 "qual" nil 10 "J" 188 128 189 "Qualitative"]
 ["Set3" 11 "qual" nil 11 "K" 204 235 197 "Qualitative"]
 ["Set3" 12 "qual" nil 1 "A" 141 211 199 "Qualitative"]
 ["Set3" 12 "qual" nil 2 "B" 255 255 179 "Qualitative"]
 ["Set3" 12 "qual" nil 3 "C" 190 186 218 "Qualitative"]
 ["Set3" 12 "qual" nil 4 "D" 251 128 114 "Qualitative"]
 ["Set3" 12 "qual" nil 5 "E" 128 177 211 "Qualitative"]
 ["Set3" 12 "qual" nil 6 "F" 253 180 98 "Qualitative"]
 ["Set3" 12 "qual" nil 7 "G" 179 222 105 "Qualitative"]
 ["Set3" 12 "qual" nil 8 "H" 252 205 229 "Qualitative"]
 ["Set3" 12 "qual" nil 9 "I" 217 217 217 "Qualitative"]
 ["Set3" 12 "qual" nil 10 "J" 188 128 189 "Qualitative"]
 ["Set3" 12 "qual" nil 11 "K" 204 235 197 "Qualitative"]
 ["Set3" 12 "qual" nil 12 "L" 255 237 111 "Qualitative"]
 ["Spectral" 3 "div" 2 1 "E" 252 141 89 "Diverging"]
 ["Spectral" 3 "div" 2 2 "H" 255 255 191 "Diverging"]
 ["Spectral" 3 "div" 2 3 "K" 153 213 148 "Diverging"]
 ["Spectral" 4 "div" 2.5 1 "C" 215 25 28 "Diverging"]
 ["Spectral" 4 "div" 2.5 2 "F" 253 174 97 "Diverging"]
 ["Spectral" 4 "div" 2.5 3 "J" 171 221 164 "Diverging"]
 ["Spectral" 4 "div" 2.5 4 "M" 43 131 186 "Diverging"]
 ["Spectral" 5 "div" 3 1 "C" 215 25 28 "Diverging"]
 ["Spectral" 5 "div" 3 2 "F" 253 174 97 "Diverging"]
 ["Spectral" 5 "div" 3 3 "H" 255 255 191 "Diverging"]
 ["Spectral" 5 "div" 3 4 "J" 171 221 164 "Diverging"]
 ["Spectral" 5 "div" 3 5 "M" 43 131 186 "Diverging"]
 ["Spectral" 6 "div" 3.5 1 "B" 213 62 79 "Diverging"]
 ["Spectral" 6 "div" 3.5 2 "E" 252 141 89 "Diverging"]
 ["Spectral" 6 "div" 3.5 3 "G" 254 224 139 "Diverging"]
 ["Spectral" 6 "div" 3.5 4 "I" 230 245 152 "Diverging"]
 ["Spectral" 6 "div" 3.5 5 "K" 153 213 148 "Diverging"]
 ["Spectral" 6 "div" 3.5 6 "N" 50 136 189 "Diverging"]
 ["Spectral" 7 "div" 4 1 "B" 213 62 79 "Diverging"]
 ["Spectral" 7 "div" 4 2 "E" 252 141 89 "Diverging"]
 ["Spectral" 7 "div" 4 3 "G" 254 224 139 "Diverging"]
 ["Spectral" 7 "div" 4 4 "H" 255 255 191 "Diverging"]
 ["Spectral" 7 "div" 4 5 "I" 230 245 152 "Diverging"]
 ["Spectral" 7 "div" 4 6 "K" 153 213 148 "Diverging"]
 ["Spectral" 7 "div" 4 7 "N" 50 136 189 "Diverging"]
 ["Spectral" 8 "div" 4.5 1 "B" 213 62 79 "Diverging"]
 ["Spectral" 8 "div" 4.5 2 "D" 244 109 67 "Diverging"]
 ["Spectral" 8 "div" 4.5 3 "F" 253 174 97 "Diverging"]
 ["Spectral" 8 "div" 4.5 4 "G" 254 224 139 "Diverging"]
 ["Spectral" 8 "div" 4.5 5 "I" 230 245 152 "Diverging"]
 ["Spectral" 8 "div" 4.5 6 "J" 171 221 164 "Diverging"]
 ["Spectral" 8 "div" 4.5 7 "L" 102 194 165 "Diverging"]
 ["Spectral" 8 "div" 4.5 8 "N" 50 136 189 "Diverging"]
 ["Spectral" 9 "div" 5 1 "B" 213 62 79 "Diverging"]
 ["Spectral" 9 "div" 5 2 "D" 244 109 67 "Diverging"]
 ["Spectral" 9 "div" 5 3 "F" 253 174 97 "Diverging"]
 ["Spectral" 9 "div" 5 4 "G" 254 224 139 "Diverging"]
 ["Spectral" 9 "div" 5 5 "H" 255 255 191 "Diverging"]
 ["Spectral" 9 "div" 5 6 "I" 230 245 152 "Diverging"]
 ["Spectral" 9 "div" 5 7 "J" 171 221 164 "Diverging"]
 ["Spectral" 9 "div" 5 8 "L" 102 194 165 "Diverging"]
 ["Spectral" 9 "div" 5 9 "N" 50 136 189 "Diverging"]
 ["Spectral" 10 "div" 5.5 1 "A" 158 1 66 "Diverging"]
 ["Spectral" 10 "div" 5.5 2 "B" 213 62 79 "Diverging"]
 ["Spectral" 10 "div" 5.5 3 "D" 244 109 67 "Diverging"]
 ["Spectral" 10 "div" 5.5 4 "F" 253 174 97 "Diverging"]
 ["Spectral" 10 "div" 5.5 5 "G" 254 224 139 "Diverging"]
 ["Spectral" 10 "div" 5.5 6 "I" 230 245 152 "Diverging"]
 ["Spectral" 10 "div" 5.5 7 "J" 171 221 164 "Diverging"]
 ["Spectral" 10 "div" 5.5 8 "L" 102 194 165 "Diverging"]
 ["Spectral" 10 "div" 5.5 9 "N" 50 136 189 "Diverging"]
 ["Spectral" 10 "div" 5.5 10 "O" 94 79 162 "Diverging"]
 ["Spectral" 11 "div" 6 1 "A" 158 1 66 "Diverging"]
 ["Spectral" 11 "div" 6 2 "B" 213 62 79 "Diverging"]
 ["Spectral" 11 "div" 6 3 "D" 244 109 67 "Diverging"]
 ["Spectral" 11 "div" 6 4 "F" 253 174 97 "Diverging"]
 ["Spectral" 11 "div" 6 5 "G" 254 224 139 "Diverging"]
 ["Spectral" 11 "div" 6 6 "H" 255 255 191 "Diverging"]
 ["Spectral" 11 "div" 6 7 "I" 230 245 152 "Diverging"]
 ["Spectral" 11 "div" 6 8 "J" 171 221 164 "Diverging"]
 ["Spectral" 11 "div" 6 9 "L" 102 194 165 "Diverging"]
 ["Spectral" 11 "div" 6 10 "N" 50 136 189 "Diverging"]
 ["Spectral" 11 "div" 6 11 "O" 94 79 162 "Diverging"]
 ["YlGn" 3 "seq" nil 1 "C" 247 252 185 "Sequential"]
 ["YlGn" 3 "seq" nil 2 "F" 173 221 142 "Sequential"]
 ["YlGn" 3 "seq" nil 3 "I" 49 163 84 "Sequential"]
 ["YlGn" 4 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGn" 4 "seq" nil 2 "E" 194 230 153 "Sequential"]
 ["YlGn" 4 "seq" nil 3 "G" 120 198 121 "Sequential"]
 ["YlGn" 4 "seq" nil 4 "J" 35 132 67 "Sequential"]
 ["YlGn" 5 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGn" 5 "seq" nil 2 "E" 194 230 153 "Sequential"]
 ["YlGn" 5 "seq" nil 3 "G" 120 198 121 "Sequential"]
 ["YlGn" 5 "seq" nil 4 "I" 49 163 84 "Sequential"]
 ["YlGn" 5 "seq" nil 5 "K" 0 104 55 "Sequential"]
 ["YlGn" 6 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGn" 6 "seq" nil 2 "D" 217 240 163 "Sequential"]
 ["YlGn" 6 "seq" nil 3 "F" 173 221 142 "Sequential"]
 ["YlGn" 6 "seq" nil 4 "G" 120 198 121 "Sequential"]
 ["YlGn" 6 "seq" nil 5 "I" 49 163 84 "Sequential"]
 ["YlGn" 6 "seq" nil 6 "K" 0 104 55 "Sequential"]
 ["YlGn" 7 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGn" 7 "seq" nil 2 "D" 217 240 163 "Sequential"]
 ["YlGn" 7 "seq" nil 3 "F" 173 221 142 "Sequential"]
 ["YlGn" 7 "seq" nil 4 "G" 120 198 121 "Sequential"]
 ["YlGn" 7 "seq" nil 5 "H" 65 171 93 "Sequential"]
 ["YlGn" 7 "seq" nil 6 "J" 35 132 67 "Sequential"]
 ["YlGn" 7 "seq" nil 7 "L" 0 90 50 "Sequential"]
 ["YlGn" 8 "seq" nil 1 "A" 255 255 229 "Sequential"]
 ["YlGn" 8 "seq" nil 2 "C" 247 252 185 "Sequential"]
 ["YlGn" 8 "seq" nil 3 "D" 217 240 163 "Sequential"]
 ["YlGn" 8 "seq" nil 4 "F" 173 221 142 "Sequential"]
 ["YlGn" 8 "seq" nil 5 "G" 120 198 121 "Sequential"]
 ["YlGn" 8 "seq" nil 6 "H" 65 171 93 "Sequential"]
 ["YlGn" 8 "seq" nil 7 "J" 35 132 67 "Sequential"]
 ["YlGn" 8 "seq" nil 8 "L" 0 90 50 "Sequential"]
 ["YlGn" 9 "seq" nil 1 "A" 255 255 229 "Sequential"]
 ["YlGn" 9 "seq" nil 2 "C" 247 252 185 "Sequential"]
 ["YlGn" 9 "seq" nil 3 "D" 217 240 163 "Sequential"]
 ["YlGn" 9 "seq" nil 4 "F" 173 221 142 "Sequential"]
 ["YlGn" 9 "seq" nil 5 "G" 120 198 121 "Sequential"]
 ["YlGn" 9 "seq" nil 6 "H" 65 171 93 "Sequential"]
 ["YlGn" 9 "seq" nil 7 "J" 35 132 67 "Sequential"]
 ["YlGn" 9 "seq" nil 8 "K" 0 104 55 "Sequential"]
 ["YlGn" 9 "seq" nil 9 "M" 0 69 41 "Sequential"]
 ["YlGnBu" 3 "seq" nil 1 "C" 237 248 177 "Sequential"]
 ["YlGnBu" 3 "seq" nil 2 "F" 127 205 187 "Sequential"]
 ["YlGnBu" 3 "seq" nil 3 "I" 44 127 184 "Sequential"]
 ["YlGnBu" 4 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGnBu" 4 "seq" nil 2 "E" 161 218 180 "Sequential"]
 ["YlGnBu" 4 "seq" nil 3 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 4 "seq" nil 4 "J" 34 94 168 "Sequential"]
 ["YlGnBu" 5 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGnBu" 5 "seq" nil 2 "E" 161 218 180 "Sequential"]
 ["YlGnBu" 5 "seq" nil 3 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 5 "seq" nil 4 "I" 44 127 184 "Sequential"]
 ["YlGnBu" 5 "seq" nil 5 "K" 37 52 148 "Sequential"]
 ["YlGnBu" 6 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGnBu" 6 "seq" nil 2 "D" 199 233 180 "Sequential"]
 ["YlGnBu" 6 "seq" nil 3 "F" 127 205 187 "Sequential"]
 ["YlGnBu" 6 "seq" nil 4 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 6 "seq" nil 5 "I" 44 127 184 "Sequential"]
 ["YlGnBu" 6 "seq" nil 6 "K" 37 52 148 "Sequential"]
 ["YlGnBu" 7 "seq" nil 1 "B" 255 255 204 "Sequential"]
 ["YlGnBu" 7 "seq" nil 2 "D" 199 233 180 "Sequential"]
 ["YlGnBu" 7 "seq" nil 3 "F" 127 205 187 "Sequential"]
 ["YlGnBu" 7 "seq" nil 4 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 7 "seq" nil 5 "H" 29 145 192 "Sequential"]
 ["YlGnBu" 7 "seq" nil 6 "J" 34 94 168 "Sequential"]
 ["YlGnBu" 7 "seq" nil 7 "L" 12 44 132 "Sequential"]
 ["YlGnBu" 8 "seq" nil 1 "A" 255 255 217 "Sequential"]
 ["YlGnBu" 8 "seq" nil 2 "C" 237 248 177 "Sequential"]
 ["YlGnBu" 8 "seq" nil 3 "D" 199 233 180 "Sequential"]
 ["YlGnBu" 8 "seq" nil 4 "F" 127 205 187 "Sequential"]
 ["YlGnBu" 8 "seq" nil 5 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 8 "seq" nil 6 "H" 29 145 192 "Sequential"]
 ["YlGnBu" 8 "seq" nil 7 "J" 34 94 168 "Sequential"]
 ["YlGnBu" 8 "seq" nil 8 "L" 12 44 132 "Sequential"]
 ["YlGnBu" 9 "seq" nil 1 "A" 255 255 217 "Sequential"]
 ["YlGnBu" 9 "seq" nil 2 "C" 237 248 177 "Sequential"]
 ["YlGnBu" 9 "seq" nil 3 "D" 199 233 180 "Sequential"]
 ["YlGnBu" 9 "seq" nil 4 "F" 127 205 187 "Sequential"]
 ["YlGnBu" 9 "seq" nil 5 "G" 65 182 196 "Sequential"]
 ["YlGnBu" 9 "seq" nil 6 "H" 29 145 192 "Sequential"]
 ["YlGnBu" 9 "seq" nil 7 "J" 34 94 168 "Sequential"]
 ["YlGnBu" 9 "seq" nil 8 "K" 37 52 148 "Sequential"]
 ["YlGnBu" 9 "seq" nil 9 "M" 8 29 88 "Sequential"]
 ["YlOrBr" 3 "seq" nil 1 "C" 255 247 188 "Sequential"]
 ["YlOrBr" 3 "seq" nil 2 "F" 254 196 79 "Sequential"]
 ["YlOrBr" 3 "seq" nil 3 "I" 217 95 14 "Sequential"]
 ["YlOrBr" 4 "seq" nil 1 "B" 255 255 212 "Sequential"]
 ["YlOrBr" 4 "seq" nil 2 "E" 254 217 142 "Sequential"]
 ["YlOrBr" 4 "seq" nil 3 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 4 "seq" nil 4 "J" 204 76 2 "Sequential"]
 ["YlOrBr" 5 "seq" nil 1 "B" 255 255 212 "Sequential"]
 ["YlOrBr" 5 "seq" nil 2 "E" 254 217 142 "Sequential"]
 ["YlOrBr" 5 "seq" nil 3 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 5 "seq" nil 4 "I" 217 95 14 "Sequential"]
 ["YlOrBr" 5 "seq" nil 5 "K" 153 52 4 "Sequential"]
 ["YlOrBr" 6 "seq" nil 1 "B" 255 255 212 "Sequential"]
 ["YlOrBr" 6 "seq" nil 2 "D" 254 227 145 "Sequential"]
 ["YlOrBr" 6 "seq" nil 3 "F" 254 196 79 "Sequential"]
 ["YlOrBr" 6 "seq" nil 4 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 6 "seq" nil 5 "I" 217 95 14 "Sequential"]
 ["YlOrBr" 6 "seq" nil 6 "K" 153 52 4 "Sequential"]
 ["YlOrBr" 7 "seq" nil 1 "B" 255 255 212 "Sequential"]
 ["YlOrBr" 7 "seq" nil 2 "D" 254 227 145 "Sequential"]
 ["YlOrBr" 7 "seq" nil 3 "F" 254 196 79 "Sequential"]
 ["YlOrBr" 7 "seq" nil 4 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 7 "seq" nil 5 "H" 236 112 20 "Sequential"]
 ["YlOrBr" 7 "seq" nil 6 "J" 204 76 2 "Sequential"]
 ["YlOrBr" 7 "seq" nil 7 "L" 140 45 4 "Sequential"]
 ["YlOrBr" 8 "seq" nil 1 "A" 255 255 229 "Sequential"]
 ["YlOrBr" 8 "seq" nil 2 "C" 255 247 188 "Sequential"]
 ["YlOrBr" 8 "seq" nil 3 "D" 254 227 145 "Sequential"]
 ["YlOrBr" 8 "seq" nil 4 "F" 254 196 79 "Sequential"]
 ["YlOrBr" 8 "seq" nil 5 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 8 "seq" nil 6 "H" 236 112 20 "Sequential"]
 ["YlOrBr" 8 "seq" nil 7 "J" 204 76 2 "Sequential"]
 ["YlOrBr" 8 "seq" nil 8 "L" 140 45 4 "Sequential"]
 ["YlOrBr" 9 "seq" nil 1 "A" 255 255 229 "Sequential"]
 ["YlOrBr" 9 "seq" nil 2 "C" 255 247 188 "Sequential"]
 ["YlOrBr" 9 "seq" nil 3 "D" 254 227 145 "Sequential"]
 ["YlOrBr" 9 "seq" nil 4 "F" 254 196 79 "Sequential"]
 ["YlOrBr" 9 "seq" nil 5 "G" 254 153 41 "Sequential"]
 ["YlOrBr" 9 "seq" nil 6 "H" 236 112 20 "Sequential"]
 ["YlOrBr" 9 "seq" nil 7 "J" 204 76 2 "Sequential"]
 ["YlOrBr" 9 "seq" nil 8 "K" 153 52 4 "Sequential"]
 ["YlOrBr" 9 "seq" nil 9 "M" 102 37 6 "Sequential"]
 ["YlOrRd" 3 "seq" nil 1 "C" 255 237 160 "Sequential"]
 ["YlOrRd" 3 "seq" nil 2 "F" 254 178 76 "Sequential"]
 ["YlOrRd" 3 "seq" nil 3 "I" 240 59 32 "Sequential"]
 ["YlOrRd" 4 "seq" nil 1 "B" 255 255 178 "Sequential"]
 ["YlOrRd" 4 "seq" nil 2 "E" 254 204 92 "Sequential"]
 ["YlOrRd" 4 "seq" nil 3 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 4 "seq" nil 4 "J" 227 26 28 "Sequential"]
 ["YlOrRd" 5 "seq" nil 1 "B" 255 255 178 "Sequential"]
 ["YlOrRd" 5 "seq" nil 2 "E" 254 204 92 "Sequential"]
 ["YlOrRd" 5 "seq" nil 3 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 5 "seq" nil 4 "I" 240 59 32 "Sequential"]
 ["YlOrRd" 5 "seq" nil 5 "K" 189 0 38 "Sequential"]
 ["YlOrRd" 6 "seq" nil 1 "B" 255 255 178 "Sequential"]
 ["YlOrRd" 6 "seq" nil 2 "D" 254 217 118 "Sequential"]
 ["YlOrRd" 6 "seq" nil 3 "F" 254 178 76 "Sequential"]
 ["YlOrRd" 6 "seq" nil 4 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 6 "seq" nil 5 "I" 240 59 32 "Sequential"]
 ["YlOrRd" 6 "seq" nil 6 "K" 189 0 38 "Sequential"]
 ["YlOrRd" 7 "seq" nil 1 "B" 255 255 178 "Sequential"]
 ["YlOrRd" 7 "seq" nil 2 "D" 254 217 118 "Sequential"]
 ["YlOrRd" 7 "seq" nil 3 "F" 254 178 76 "Sequential"]
 ["YlOrRd" 7 "seq" nil 4 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 7 "seq" nil 5 "H" 252 78 42 "Sequential"]
 ["YlOrRd" 7 "seq" nil 6 "J" 227 26 28 "Sequential"]
 ["YlOrRd" 7 "seq" nil 7 "L" 177 0 38 "Sequential"]
 ["YlOrRd" 8 "seq" nil 1 "A" 255 255 204 "Sequential"]
 ["YlOrRd" 8 "seq" nil 2 "C" 255 237 160 "Sequential"]
 ["YlOrRd" 8 "seq" nil 3 "D" 254 217 118 "Sequential"]
 ["YlOrRd" 8 "seq" nil 4 "F" 254 178 76 "Sequential"]
 ["YlOrRd" 8 "seq" nil 5 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 8 "seq" nil 6 "H" 252 78 42 "Sequential"]
 ["YlOrRd" 8 "seq" nil 7 "J" 227 26 28 "Sequential"]
 ["YlOrRd" 8 "seq" nil 8 "L" 177 0 38 "Sequential"]
 ["YlOrRd" 9 "seq" nil 1 "A" 255 255 204 "Sequential"]
 ["YlOrRd" 9 "seq" nil 2 "C" 255 237 160 "Sequential"]
 ["YlOrRd" 9 "seq" nil 3 "D" 254 217 118 "Sequential"]
 ["YlOrRd" 9 "seq" nil 4 "F" 254 178 76 "Sequential"]
 ["YlOrRd" 9 "seq" nil 5 "G" 253 141 60 "Sequential"]
 ["YlOrRd" 9 "seq" nil 6 "H" 252 78 42 "Sequential"]
 ["YlOrRd" 9 "seq" nil 7 "J" 227 26 28 "Sequential"]
 ["YlOrRd" 9 "seq" nil 8 "K" 189 0 38 "Sequential"]
 ["YlOrRd" 9 "seq" nil 9 "M" 128 0 38 "Sequential"]])
