(ns foo
  (:require [domkm.silk :as silk]
            [domkm.silk.serve :as serve]))

; Patterns can be matched and unmatched with `domkm.silk/match` and `domkm.silk/unmatch` respectively.

; Strings only match themselves.

(silk/match "foo" "foo")
;=> {}
(silk/unmatch "foo" {})
;=> "foo"

; Keywords are wildcards.

(silk/match :foo "bar")
;=> {:foo "bar"}
(silk/unmatch :foo {:foo "bar"})
;=> "bar"

; There are also built in patterns for common use cases.

(silk/match (silk/integer :answer) "42")
;=> {:answer 42}
(silk/unmatch (silk/integer :answer) {:answer 42})
;=> "42"

(silk/match (silk/uuid :id) "c11902f0-21b6-4645-a218-9fa40ef69333")
;=> {:id #uuid "c11902f0-21b6-4645-a218-9fa40ef69333"}
(silk/unmatch (silk/uuid :id) {:id #uuid "c11902f0-21b6-4645-a218-9fa40ef69333"})
;=> "c11902f0-21b6-4645-a218-9fa40ef69333"

(silk/match (silk/alternative :char ["a" "b" "c"]) "a")
;=> {:car "a"}
(silk/match (silk/alternative :char ["a" "b" "c"]) "b")
;=> {:char "b"}
(silk/match (silk/alternative :char ["a" "b" "c"]) "x")
;=> nil

(silk/match (silk/composite ["user-" (silk/integer :id)]) "user-42")
;=> {:id 42}
(silk/unmatch (silk/composite ["user-" (silk/integer :id)]) {:id 42})
;=> "user-42"

(silk/match (silk/option :this "that") "foo")
;=> {:this "foo"}
(silk/match (silk/option :this "that") nil)
;=> {:this "else"}
(silk/unmatch (silk/option :this "that") {:this "foo"})
;=> "foo"
(silk/unmatch (silk/option :this "that") {})
;=> "that"

; Patterns can match data structures.

(silk/match ["users" (silk/integer :id)] ["users" "42"])
;=> {:id 42}

; Routes are a seqable of 2-tuples. The first element is a route key and the second element is something that can be turned into a URL pattern.

; If the second element is a vector, the first and second elements are `assoc`iated into the third element under `:path` and `:query` keys respectively.

(silk/url-pattern [["i" "am" "a" "path"] {"i" "am" "a" "query"} {:scheme "https"}])
;=> {:query {"i" "am", "a" "query"}, :path ["i" "am" "a" "path"], :scheme "https"}

; If the second element is a map, it is left unchanged.

(silk/url-pattern {:query {"i" "am" "a" "query"} :path ["i" "am" "a" "path"] :scheme "https"})
;=> {:query {"i" "am", "a" "query"}, :path ["i" "am" "a" "path"], :scheme "https"}

(silk/route [:route-key [["users" :username]]])
;=> #<Route domkm.silk.Route@6ebe4324>

; Routes are patterns.
(silk/match (silk/route [:route-key [["users" :username]]]) {:path ["users" "domkm"]})
;=> {:username "domkm", :domkm.silk/key :route-key, :domkm.silk/pattern {:path ["users" :username]}}
(silk/unmatch (silk/route [:route-key [["users" :username]]]) {:username "domkm"})
;=> #domkm.silk.URL{:scheme nil, :user nil, :host nil, :port nil, :path ["users" "domkm"], :query nil, :fragment nil}

; None of that is particularly useful unless you can match and unmatch route collections. Fortunately, a collection of routes is also a pattern.

(def user-routes
  (silk/routes [[:users-index [["users"]]]
                [:users-show [["users" (silk/integer :id)]]]]))

(silk/match user-routes {:path ["users" "42"]})
;=> {:id 42, :domkm.silk/key :users-show, :domkm.silk/routes #<Routes domkm.silk.Routes@c6f8bbc>, ...}
(silk/unmatch user-routes {:id 42 :domkm.silk/key :users-show})
;=> #domkm.silk.URL{:scheme nil, :user nil, :host nil, :port nil, :path ["users" "42"], :query nil, :fragment nil}

; If you don't care about the match order, you can create routes with a map.

(def page-routes
  (silk/routes {:home-page [[]] ; match "/"
                :other-page [["pages" :title]]}))

; Routes can be constrained by request methods.

(def api-routes
  (silk/routes {:api-data (serve/method :post [["api"] {"limit" (silk/option (silk/integer :limit) "100")
                                                        "offset" (silk/option (silk/integer :offset) "0")}])}))

(silk/match api-routes {:path ["api"]})
;=> nil
(silk/match api-routes {:path ["api"] :request {:request-method :post}})
;=> {:limit 100, :offset 0, :domkm.silk/key :api-data, ...}

; Routes can be combined.

(def all-routes
  (silk/routes [user-routes
                page-routes
                [:comments [["comments"] {"id" (silk/uuid :id)}]]
                api-routes]))

; Note that all matching and unmatching is pure and bidirectional.

; Matching and unmatching patterns is powerful but verbose.
; Silk provides a higher-level interface via `domkm.silk/arrive` and `domkm.silk/depart`

(silk/arrive all-routes "/pages/about")
;=> {:title "about", :domkm.silk/key :other-page, ...}

; You can also provide a handler function.

(silk/arrive all-routes "/pages/about" :title)
;=> "about"

; Forming routes is almost as easy.

(silk/depart all-routes :other-page {:title "about"})
;=> "/pages/about"

; As with `domkm.silk/arrive`, you can provide a handler function.
(silk/depart all-routes :other-page {:title "about"} clojure.string/upper-case)
;=> "/PAGES/ABOUT"


; Go forth and route!
