(ns doccla.oth-client.clinician.api.patients.questionnaire-results
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [doccla.oth-client.schemas :as schemas]))

(def severity-enum
  [:enum "none" "green" "orange" "blue" "yellow" "red"])

(def acknowledgement-details-schema
  [:map
   [:by [:map
         [:first-name string?]
         [:last-name string?]
         [:username string?]
         [:links [:map
                  [:clinician [:re schemas/url-regex]]]]]]
   [:note string?]
   [:date [:re schemas/iso-8601-regex]]
   [:visible-for-patient boolean?]
   [:closure-notes [:vector string?]]])

(def questions-and-replies-summary-schema
  [:map
   [:text string?]
   [:patient-text string?]
   [:reply-type string?]
   [:reply-id [:re schemas/url-regex]]
   [:reply [:or [:map [:answer [:or string? boolean? number?]]
                 [:patient-answer {:optional true} string?]]
            [:map [:omitted [:enum true]]]
            [:map [:measurements [:vector
                                  schemas/measurement-schema]]]
            [:map [:reference [:map
                               [:measurement-capture [:re schemas/url-regex]]]]]]]])

(def questionnaire-results-schema
  [:map
   [:processing-status string?]
   [:name string?]
   [:version integer?]
   [:severity severity-enum]
   [:result-date [:re schemas/iso-8601-regex]]
   [:acknowledgement acknowledgement-details-schema]
   [:questions [:vector questions-and-replies-summary-schema]]
   [:links [:map
            [:self [:re schemas/url-regex]]
            [:questionnaire [:re schemas/url-regex]]
            [:patient [:re schemas/url-regex]]]]])
;
(m/=> get-questionnaire-results [:=>
                                 [:cat schemas/opts-schema [:or string? int?] [:or string? int?]]
                                 [:or schemas/error-schema (schemas/success-schema questionnaire-results-schema)]])

(defn ^:mockable get-questionnaire-results-by-link
  [opts url]
  (let [res (client/get url (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f questionnaire-results-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

(defn ^:mockable get-questionnaire-results
  [opts patient-id questionnaire-id]
  (let [url  (str (:base-url opts) "/clinician/api/patients/" patient-id "/questionnaire-results/" questionnaire-id)]
    (get-questionnaire-results-by-link opts url)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
 ;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-questionnaire-results-by-link {:base-url "https://doccla-dev.oth.io"
                                      :validate-output? true
                                      :auth {:type :id-secret :id "" :secret ""}}
                                     "https://doccla-dev.oth.io/clinician/api/patients/1843/questionnaire-results/4569"))
