(ns doccla.oth-client.schemas)

(def base-64-regex #"^data:image.+;base64,")
(def simple-date-regex #"^(\d{4})[\-](0[1-9]|1[0-2])[\-]([12]\d|0[1-9]|3[01])$")
(def iso-8601-regex #"^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2})")
(def url-regex #"^http.+[^/]$")
(def ip-address-regex #"^(\d{1,3}\.){3}\d{1,3}$")

(def opts-schema
  [:map
   [:base-url [:re #"^http.+[^/]$"]]
   [:validate-output? :boolean]
   [:auth [:multi
           {:dispatch :type}
           [:token     [:map [:type [:= :token]] [:token string?]]]
           [:id-secret [:map
                        [:type [:= :id-secret]]
                        [:id string?]
                        [:secret string?]]]]]])

(def error-schema
  [:map
   [:code int?]
   [:success? [:= false]]
   [:error string?]])

(defn success-schema
  "Given a response schema, returns a success schema that wraps that response"
  [response-schema]
  [:map
   [:code int?]
   [:success? [:= true]]
   [:response response-schema]])

(def measurement-types-enum
  [:enum "news2" "consciousness" "pulse" "weight" "weight_pound" "temperature" "temperature_fahrenheit" "protein_in_urine" "glucose_in_urine"
   "blood_in_urine" "nitrite_in_urine" "leukocytes_in_urine" "hemoglobin" "crp" "saturation" "lung_function"
   "bloodsugar" "bloodsugar_mg_dl" "daily_steps" "blood_pressure" "pain_scale" "vas_score" "borg_rating" "ctg"
   "height" "respiratory_rate" "peak_flow" "body_cell_mass" "fat_mass" "phase_angle" "fev1" "fev6"
   "inflow_bag_color" "inflow_bag_volume" "outflow_bag_volume" "ultrafiltration_volume" "pump_flow" "pump_speed"
   "pulse_index" "pump_power"])

(def threshold-schema
  [:map
   [:measurement-type measurement-types-enum]
   [:limits
    [:vector
     [:map
      [:name string?]
      [:alert-high {:optional true} some?]
      [:alert-low {:optional true} some?]
      [:warning-high {:optional true} some?]
      [:warning-low {:optional true} some?]]]]
   [:threshold-type
    [:map
     [:point {:optional true}
      [:map
       [:type string?]]]
     [:scope string?]]]])

(def blood-pressure-schema
  [:map
   [:timestamp [:re iso-8601-regex]]
   [:type [:enum "blood_pressure", "blood_pressure_weekly_average"]]
   [:measurement [:map [:systolic number?]
                  [:diastolic number?]
                  [:unit [:enum "mmHg"]]]]])

(def blood-sugar-schema
  [:map
   [:timestamp [:re iso-8601-regex]]
   [:type [:enum "bloodsugar"]]
   [:measurement [:map [:unit [:enum "mmol/L"]]
                  [:value number?]
                  [:is-after-meal {:optional true} boolean?]
                  [:is-before-meal {:optional true} boolean?]
                  [:is-fasting {:optional true} boolean?]
                  [:is-control-measurement {:optional true} boolean?]
                  [:ignored {:optional true} map?]]]])

(def blood-sugar-mg-dl
  [:map
   [:timestamp [:re iso-8601-regex]]
   [:type [:enum "bloodsugar_mg_dl"]]
   [:measurement [:map [:unit [:enum "mg/dL"]]
                  [:value number?]
                  [:is-after-meal {:optional true} boolean?]
                  [:is-before-meal {:optional true} boolean?]
                  [:is-fasting {:optional true} boolean?]
                  [:is-control-measurement {:optional true} boolean?]
                  [:ignored {:optional true} map?]]]])

(def measurement-types-simple-enum
  [:enum  "news2" "consciousness" "phase_angle" "fat_mass" "body_cell_mass" "weight" "weight_pound" "pulse" "temperature"
   "temperature_fahrenheit" "pain_scale" "blood_in_urine" "protein_in_urine" "leukocytes_in_urine"
   "glucose_in_urine" "nitrite_in_urine" "hemoglobin" "saturation" "crp" "height" "daily_steps"
   "daily_steps_weekly_average" "respiratory_rate" "peak_flow" "sit_to_stand" "fev1" "fev6"
   "fev1/fev6" "fef25-75%" "fev1%" "fev6%" "copd_prediction" "oxygen_flow"])

(def simple-value-schema
  [:map
   [:timestamp [:re iso-8601-regex]]
   [:type measurement-types-simple-enum]
   [:measurement [:map [:unit [:enum nil? "kg" "BPM" "mmol/L" "°C" "°F" "g/dL" "%" "mg/L" "mL" "mV" "L" "mmHg" "-" "cm" "RR" "L/min" "L/s" "min" "hour" "step count" "repetitions" "°" "lb"]]
                  [:value number?]
                  [:ignored {:optional true} map?]]]])

(def urine-value-schema
  [:map
   [:timestamp [:re iso-8601-regex]]
   [:type measurement-types-simple-enum]
   [:measurement [:map [:unit [:enum "Ery/μL" "g/L" "Leu/µL" "mmol/L" "-"]]
                  [:value string?]
                  [:ignored {:optional true} map?]]]])

(def measurement-origin-schema
  [:map
   [:manual-measurement {:optional true} [:map [:entered-by [:enum "citizen" "clinician"]]]]
   [:calculated-measurement {:optional true} [:map [:calculated-by string?]]]
   [:device-measurement {:optional true} map?]])

(def triggered-threshold-schema
  [:map
   [:type {:optional true} string?]
   [:result {:optional true} string?]
   [:measurement-type {:optional true} string?]
   [:links {:optional true} [:map
                             [:threshold [:re url-regex]]]]])
(def measurements-schema
  [:map
   [:results [:vector [:and
                       [:or
                        blood-pressure-schema
                        blood-sugar-schema
                        blood-sugar-mg-dl
                        simple-value-schema
                        urine-value-schema]

                       [:map
                        [:severity {:optional true} string?]
                        [:triggered-thresholds {:optional true} [:vector triggered-threshold-schema]]
                        [:comment {:optional true} string?]
                        [:origin {:optional true} measurement-origin-schema]
                        [:links [:map
                                 [:measurement [:re url-regex]]
                                 [:patient {:optional true} [:re url-regex]]]]]]]]
   [:offset int?]
   [:max int?]
   [:total int?]
   [:links [:map
            [:self [:re url-regex]]
            [:previous {:optional true} [:re url-regex]]
            [:next {:optional true} [:re url-regex]]]]])
