(ns doccla.oth-client.organizations.organizations
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [doccla.oth-client.schemas :as schemas]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def organization-schema
  [:map
   [:healthcare-system {:optional true} string?]
   [:links [:map
            [:metadata-fields string?]
            [:self string?]]]
   [:lookup-service {:optional true} string?]
   [:metadata-fields [:vector any?]]
   [:name string?]])

(m/=> get-organization [:=>
                        [:cat schemas/opts-schema string?]
                        [:or schemas/error-schema (schemas/success-schema organization-schema)]])

(defn post-processor
  [opts]
  (fn [data]
    (let [f (if (:validate-output? opts) m/coerce m/decode)]
      (f organization-schema data utils/prune-map-transformer))))

(defn ^:mockable get-organization
  "Retrieve an organization by its UUID"
  [opts uuid]
  (let [url (str (:base-url opts) "/organizations/organizations/" uuid)
        req (utils/opts->request opts)
        res (client/get url req)]
    (utils/->output [200] (post-processor opts) res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope   #{:input}
                 :report  utils/input-validate-fail!})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-organization {:base-url         "https://doccla-dev.oth.io"
                     :validate-output? true
                     :auth             {:type :id-secret :id "" :secret ""}}
                    "123e4567-e89b-12d3-a456-426614174000"))
