(ns doccla.oth-client.object-storage.objects
  (:require [clj-http.client :as client]
            [malli.core :as m]
            [doccla.oth-client.utils :as utils]
            [doccla.oth-client.schemas :as schemas]
            [malli.instrument :as mi]
            [malli.clj-kondo :as clj-kondo]))

(def object-uploaded-schema
  [:map
   [:links [:map
            [:self [:re schemas/url-regex]]]]])

(defn ^:mockable get-object
  "Downloads from the Object Store"
  [opts object-uuid]
  (let [default-req-map {:headers          {"Accept"       "*/*"}
                         :as               :byte-array
                         :throw-exceptions false}
        req-opts        (utils/opts->request opts default-req-map)
        url             (str (:base-url opts) "/object-storage/objects/" object-uuid)
        res             (client/get url req-opts)]
    (utils/->output [200] identity identity res)))

(defn ^:mockable create-object
  "Uploads to the Object Store"
  [opts]
  (let [default-req-map {:headers          {"Content-Type" "*/*"
                                            "Accept"       "application/json"}
                         :as               :json-kebab-keys
                         :throw-exceptions false}
        req-opts        (utils/opts->request opts default-req-map)
        url             (str (:base-url opts) "/object-storage/objects")
        res             (client/post url req-opts)
        post-processor  (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                     (f object-uploaded-schema data utils/prune-map-transformer)))]
    (utils/->output [201] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
 ;; Enable mocks
(utils/make-mockable)

(comment

  (require '[clojure.java.io :as io])

  (def oth-client {:base-url "REPLACE-ME"
                   :validate-output? true
                   :auth {:type   :id-secret
                          :id     "REPLACE-ME"
                          :secret "REPLACE-ME"}})

  ;; Upload example
  (def resp (with-open [in (io/input-stream "test_upload.pdf")]
              (create-object (assoc oth-client :content-type :pdf :body in))))

  ;; Download examples

  ;; By default :byte-array is specified, so the response is read fully into memory as a byte array
  (with-open [out (io/output-stream "test_byte-array_download.pdf")]
    (let [b-array (:response (get-object oth-client "REPLACE-ME"))]
      (.write out #^bytes b-array)))

  ;; By specifying :stream, the response is coerced into an input stream and not read fully into memory
  (with-open [out (io/output-stream "test_input-stream_download.pdf")
              in  (:response (get-object (assoc oth-client :as :stream) "REPLACE-ME"))]
    (io/copy in out)))
