(ns doccla.oth-client.kairos
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.schemas :as schemas]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(defn ^:mockable query
  "Make a kairos/graphql query to OTH.
   opts - Standard OTH client options map with :base-url and :auth
   q - GraphQL query string
   And optionally, a map of graphql options to pass to the query"
  [{:keys [base-url] :as opts} q & {:keys [variables operation-name extensions]
                                    :or {variables {}
                                         operation-name nil
                                         extensions {}}}]
  (let [body {:query q
              :variables variables
              :operationName operation-name
              :extensions extensions}
        req (-> (utils/opts->request-with-body opts body)
                (assoc :accept "application/graphql-response+json"))
        response (client/post (str base-url "/kairos/api") req)
        result (utils/->output [200 201 204] identity response)
        errors (-> result :response :errors)] ;; graphql may return 200 with errors
    (if (seq errors)
      (-> result
          (assoc :success? false :error errors)
          (dissoc :response))
      result)))

(m/=> query [:=>
             [:cat schemas/opts-schema string?  [:? [:map
                                                     [:variables {:optional true} [:map]]
                                                     [:operation-name {:optional true} string?]
                                                     [:extensions {:optional true} [:map]]]]]
             [:or schemas/error-schema [:map
                                        [:data {:optional true} [:map]]
                                        [:errors {:optional true} [:vector [:map]]]
                                        [:extensions {:optional true} [:map]]]]])

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})

(clj-kondo/emit!)
(utils/make-mockable)

(comment
  (require '[clojure.edn :as edn])
  (def creds (edn/read-string (slurp "creds/oth.edn")))
  (def opts {:base-url    "https://doccla-dev.oth.io"
             :validate-output? true
             :auth {:type :id-secret :id (:id creds) :secret (:secret creds)}})

  (query opts "query {
                  __schema {
                    types {
                      name
                    }
                  }
                }"))